/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest.geoip;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.logging.HeaderWarning;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.ingest.IngestService;
import org.elasticsearch.ingest.geoip.ConfigDatabases;
import org.elasticsearch.ingest.geoip.DatabaseReaderLazyLoader;
import org.elasticsearch.ingest.geoip.GeoIpCache;
import org.elasticsearch.ingest.geoip.GeoIpDatabase;
import org.elasticsearch.ingest.geoip.GeoIpDatabaseProvider;
import org.elasticsearch.ingest.geoip.GeoIpProcessor;
import org.elasticsearch.ingest.geoip.GeoIpTaskState;
import org.elasticsearch.ingest.geoip.TarInputStream;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.watcher.ResourceWatcherService;

public final class DatabaseNodeService
implements GeoIpDatabaseProvider,
Closeable {
    private static final Logger LOGGER = LogManager.getLogger(DatabaseNodeService.class);
    private final Client client;
    private final GeoIpCache cache;
    private final Path geoipTmpBaseDirectory;
    private Path geoipTmpDirectory;
    private final ConfigDatabases configDatabases;
    private final Consumer<Runnable> genericExecutor;
    private final ClusterService clusterService;
    private IngestService ingestService;
    private final ConcurrentMap<String, DatabaseReaderLazyLoader> databases = new ConcurrentHashMap<String, DatabaseReaderLazyLoader>();

    DatabaseNodeService(Environment environment, Client client, GeoIpCache cache, Consumer<Runnable> genericExecutor, ClusterService clusterService) {
        this(environment.tmpFile(), (Client)new OriginSettingClient(client, "ingest"), cache, new ConfigDatabases(environment, cache), genericExecutor, clusterService);
    }

    DatabaseNodeService(Path tmpDir, Client client, GeoIpCache cache, ConfigDatabases configDatabases, Consumer<Runnable> genericExecutor, ClusterService clusterService) {
        this.client = client;
        this.cache = cache;
        this.geoipTmpBaseDirectory = tmpDir.resolve("geoip-databases");
        this.configDatabases = configDatabases;
        this.genericExecutor = genericExecutor;
        this.clusterService = clusterService;
    }

    public void initialize(String nodeId, ResourceWatcherService resourceWatcher, IngestService ingestServiceArg) throws IOException {
        this.configDatabases.initialize(resourceWatcher);
        this.geoipTmpDirectory = this.geoipTmpBaseDirectory.resolve(nodeId);
        Files.walkFileTree(this.geoipTmpDirectory, (FileVisitor<? super Path>)new FileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                try {
                    LOGGER.info("deleting stale file [{}]", (Object)file);
                    Files.deleteIfExists(file);
                }
                catch (IOException e) {
                    LOGGER.warn("can't delete stale file [" + file + "]", (Throwable)e);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException e) {
                if (!(e instanceof NoSuchFileException)) {
                    LOGGER.warn("can't delete stale file [" + file + "]", (Throwable)e);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                return FileVisitResult.CONTINUE;
            }
        });
        if (!Files.exists(this.geoipTmpDirectory, new LinkOption[0])) {
            Files.createDirectories(this.geoipTmpDirectory, new FileAttribute[0]);
        }
        LOGGER.debug("initialized database node service, using geoip-databases directory [{}]", (Object)this.geoipTmpDirectory);
        this.ingestService = ingestServiceArg;
        this.clusterService.addListener(event -> this.checkDatabases(event.state()));
    }

    @Override
    public Boolean isValid(String databaseFile) {
        ClusterState currentState = this.clusterService.state();
        assert (currentState != null);
        PersistentTasksCustomMetadata.PersistentTask task = PersistentTasksCustomMetadata.getTaskWithId((ClusterState)currentState, (String)"geoip-downloader");
        if (task == null || task.getState() == null) {
            return true;
        }
        GeoIpTaskState state = (GeoIpTaskState)task.getState();
        GeoIpTaskState.Metadata metadata = state.getDatabases().get(databaseFile);
        if (metadata == null) {
            return true;
        }
        boolean valid = metadata.isValid(currentState.metadata().settings());
        if (valid && metadata.isCloseToExpiration()) {
            HeaderWarning.addWarning((String)"database [{}] was not updated for over 25 days, geoip processor will stop working if there is no update for 30 days", (Object[])new Object[]{databaseFile});
        }
        return valid;
    }

    DatabaseReaderLazyLoader getDatabaseReaderLazyLoader(String name) {
        DatabaseReaderLazyLoader instance;
        do {
            if ((instance = (DatabaseReaderLazyLoader)this.databases.get(name)) != null) continue;
            instance = this.configDatabases.getDatabase(name);
        } while (instance != null && !instance.preLookup());
        return instance;
    }

    @Override
    public GeoIpDatabase getDatabase(String name) {
        return this.getDatabaseReaderLazyLoader(name);
    }

    List<DatabaseReaderLazyLoader> getAllDatabases() {
        ArrayList<DatabaseReaderLazyLoader> all = new ArrayList<DatabaseReaderLazyLoader>(this.configDatabases.getConfigDatabases().values());
        this.databases.forEach((key, value) -> all.add((DatabaseReaderLazyLoader)value));
        return all;
    }

    DatabaseReaderLazyLoader get(String key) {
        return (DatabaseReaderLazyLoader)this.databases.get(key);
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.databases.values());
    }

    void checkDatabases(ClusterState state) {
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        DiscoveryNode localNode = state.nodes().getLocalNode();
        if (!localNode.isIngestNode()) {
            LOGGER.trace("Not checking databases because local node is not ingest node");
            return;
        }
        PersistentTasksCustomMetadata persistentTasks = (PersistentTasksCustomMetadata)state.metadata().custom("persistent_tasks");
        if (persistentTasks == null) {
            LOGGER.trace("Not checking databases because persistent tasks are null");
            return;
        }
        IndexAbstraction databasesAbstraction = (IndexAbstraction)state.getMetadata().getIndicesLookup().get(".geoip_databases");
        if (databasesAbstraction == null) {
            LOGGER.trace("Not checking databases because geoip databases index does not exist");
            return;
        }
        Index databasesIndex = databasesAbstraction.getWriteIndex();
        IndexRoutingTable databasesIndexRT = state.getRoutingTable().index(databasesIndex);
        if (databasesIndexRT == null || !databasesIndexRT.allPrimaryShardsActive()) {
            LOGGER.trace("Not checking databases because geoip databases index does not have all active primary shards");
            return;
        }
        PersistentTasksCustomMetadata.PersistentTask task = PersistentTasksCustomMetadata.getTaskWithId((ClusterState)state, (String)"geoip-downloader");
        GeoIpTaskState taskState = task == null || task.getState() == null ? GeoIpTaskState.EMPTY : (GeoIpTaskState)task.getState();
        taskState.getDatabases().entrySet().stream().filter(e -> ((GeoIpTaskState.Metadata)e.getValue()).isValid(state.getMetadata().settings())).forEach(e -> {
            String localMd5;
            String name = (String)e.getKey();
            GeoIpTaskState.Metadata metadata = (GeoIpTaskState.Metadata)e.getValue();
            DatabaseReaderLazyLoader reference = (DatabaseReaderLazyLoader)this.databases.get(name);
            String remoteMd5 = metadata.md5();
            String string = localMd5 = reference != null ? reference.getMd5() : null;
            if (Objects.equals(localMd5, remoteMd5)) {
                LOGGER.debug("Current reference of [{}] is up to date [{}] with was recorded in CS [{}]", (Object)name, (Object)localMd5, (Object)remoteMd5);
                return;
            }
            try {
                this.retrieveAndUpdateDatabase(name, metadata);
            }
            catch (Exception ex) {
                LOGGER.error(() -> "attempt to download database [" + name + "] failed", (Throwable)ex);
            }
        });
        ArrayList<String> staleEntries = new ArrayList<String>(this.databases.keySet());
        staleEntries.removeAll(taskState.getDatabases().entrySet().stream().filter(e -> ((GeoIpTaskState.Metadata)e.getValue()).isValid(state.getMetadata().settings())).map(Map.Entry::getKey).collect(Collectors.toSet()));
        this.removeStaleEntries(staleEntries);
    }

    void retrieveAndUpdateDatabase(String databaseName, GeoIpTaskState.Metadata metadata) throws IOException {
        Path databaseTmpGzFile;
        LOGGER.trace("Retrieving database {}", (Object)databaseName);
        String recordedMd5 = metadata.md5();
        try {
            databaseTmpGzFile = Files.createFile(this.geoipTmpDirectory.resolve(databaseName + ".tmp.gz"), new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            LOGGER.debug("database update [{}] already in progress, skipping...", (Object)databaseName);
            return;
        }
        DatabaseReaderLazyLoader lazyLoader = (DatabaseReaderLazyLoader)this.databases.get(databaseName);
        if (lazyLoader != null && recordedMd5.equals(lazyLoader.getMd5())) {
            LOGGER.debug("deleting tmp file because database [{}] has already been updated.", (Object)databaseName);
            Files.delete(databaseTmpGzFile);
            return;
        }
        Path databaseTmpFile = Files.createFile(this.geoipTmpDirectory.resolve(databaseName + ".tmp"), new FileAttribute[0]);
        LOGGER.debug("retrieve geoip database [{}] from [{}] to [{}]", (Object)databaseName, (Object)".geoip_databases", (Object)databaseTmpGzFile);
        this.retrieveDatabase(databaseName, recordedMd5, metadata, (CheckedConsumer<byte[], IOException>)((CheckedConsumer)bytes -> Files.write(databaseTmpGzFile, bytes, StandardOpenOption.APPEND)), (CheckedRunnable<Exception>)((CheckedRunnable)() -> {
            LOGGER.debug("decompressing [{}]", (Object)databaseTmpGzFile.getFileName());
            Path databaseFile = this.geoipTmpDirectory.resolve(databaseName);
            try (TarInputStream is = new TarInputStream(new GZIPInputStream((InputStream)new BufferedInputStream(Files.newInputStream(databaseTmpGzFile, new OpenOption[0])), 8192));){
                TarInputStream.TarEntry entry;
                while ((entry = is.getNextEntry()) != null) {
                    if (entry.notFile()) continue;
                    String name = entry.name().substring(entry.name().lastIndexOf(47) + 1);
                    if (name.startsWith(databaseName)) {
                        Files.copy(is, databaseTmpFile, StandardCopyOption.REPLACE_EXISTING);
                        continue;
                    }
                    Files.copy(is, this.geoipTmpDirectory.resolve(databaseName + "_" + name), StandardCopyOption.REPLACE_EXISTING);
                }
            }
            LOGGER.debug("moving database from [{}] to [{}]", (Object)databaseTmpFile, (Object)databaseFile);
            Files.move(databaseTmpFile, databaseFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            this.updateDatabase(databaseName, recordedMd5, databaseFile);
            Files.delete(databaseTmpGzFile);
        }), failure -> {
            LOGGER.error(() -> "failed to retrieve database [" + databaseName + "]", (Throwable)failure);
            try {
                Files.deleteIfExists(databaseTmpFile);
                Files.deleteIfExists(databaseTmpGzFile);
            }
            catch (IOException ioe) {
                ioe.addSuppressed((Throwable)failure);
                LOGGER.error("Unable to delete tmp database file after failure", (Throwable)ioe);
            }
        });
    }

    void updateDatabase(String databaseFileName, String recordedMd5, Path file) {
        try {
            LOGGER.debug("starting reload of changed geoip database file [{}]", (Object)file);
            DatabaseReaderLazyLoader loader = new DatabaseReaderLazyLoader(this.cache, file, recordedMd5);
            DatabaseReaderLazyLoader existing = this.databases.put(databaseFileName, loader);
            if (existing != null) {
                existing.close();
            } else {
                Predicate<GeoIpProcessor.DatabaseUnavailableProcessor> predicate = p -> databaseFileName.equals(p.getDatabaseName());
                Collection ids = this.ingestService.getPipelineWithProcessorType(GeoIpProcessor.DatabaseUnavailableProcessor.class, predicate);
                if (!ids.isEmpty()) {
                    LOGGER.debug("pipelines [{}] found to reload", (Object)ids);
                    for (String id : ids) {
                        try {
                            this.ingestService.reloadPipeline(id);
                            LOGGER.trace("successfully reloaded pipeline [{}] after downloading of database [{}] for the first time", (Object)id, (Object)databaseFileName);
                        }
                        catch (Exception e) {
                            LOGGER.debug(() -> Strings.format((String)"failed to reload pipeline [%s] after downloading of database [%s]", (Object[])new Object[]{id, databaseFileName}), (Throwable)e);
                        }
                    }
                } else {
                    LOGGER.debug("no pipelines found to reload");
                }
            }
            LOGGER.info("successfully loaded geoip database file [{}]", (Object)file.getFileName());
        }
        catch (Exception e) {
            LOGGER.error(() -> "failed to update database [" + databaseFileName + "]", (Throwable)e);
        }
    }

    void removeStaleEntries(Collection<String> staleEntries) {
        for (String staleEntry : staleEntries) {
            try {
                LOGGER.debug("database [{}] no longer exists, cleaning up...", (Object)staleEntry);
                DatabaseReaderLazyLoader existing = (DatabaseReaderLazyLoader)this.databases.remove(staleEntry);
                assert (existing != null);
                existing.close(true);
            }
            catch (Exception e) {
                LOGGER.error(() -> "failed to clean database [" + staleEntry + "]", (Throwable)e);
            }
        }
    }

    void retrieveDatabase(String databaseName, String expectedMd5, GeoIpTaskState.Metadata metadata, CheckedConsumer<byte[], IOException> chunkConsumer, CheckedRunnable<Exception> completedHandler, Consumer<Exception> failureHandler) {
        this.genericExecutor.accept(() -> {
            MessageDigest md = MessageDigests.md5();
            int firstChunk = metadata.firstChunk();
            int lastChunk = metadata.lastChunk();
            try {
                for (int chunk = firstChunk; chunk <= lastChunk; ++chunk) {
                    SearchRequest searchRequest = new SearchRequest(new String[]{".geoip_databases"});
                    String id = String.format(Locale.ROOT, "%s_%d_%d", databaseName, chunk, metadata.lastUpdate());
                    searchRequest.source().query((QueryBuilder)new TermQueryBuilder("_id", id));
                    SearchResponse searchResponse = (SearchResponse)this.client.search(searchRequest).actionGet();
                    SearchHit[] hits = searchResponse.getHits().getHits();
                    if (searchResponse.getHits().getHits().length == 0) {
                        failureHandler.accept((Exception)new ResourceNotFoundException("chunk document with id [" + id + "] not found", new Object[0]));
                        return;
                    }
                    byte[] data = (byte[])hits[0].getSourceAsMap().get("data");
                    md.update(data);
                    chunkConsumer.accept((Object)data);
                }
                String actualMd5 = MessageDigests.toHexString((byte[])md.digest());
                if (Objects.equals(expectedMd5, actualMd5)) {
                    completedHandler.run();
                } else {
                    failureHandler.accept(new RuntimeException("expected md5 hash [" + expectedMd5 + "], but got md5 hash [" + actualMd5 + "]"));
                }
            }
            catch (Exception e) {
                failureHandler.accept(e);
            }
        });
    }

    public Set<String> getAvailableDatabases() {
        return Set.copyOf(this.databases.keySet());
    }

    public Set<String> getConfigDatabases() {
        return this.configDatabases.getConfigDatabases().keySet();
    }

    public Set<String> getFilesInTemp() {
        Set<String> set;
        block8: {
            Stream<Path> files = Files.list(this.geoipTmpDirectory);
            try {
                set = files.map(Path::getFileName).map(Path::toString).collect(Collectors.toSet());
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            files.close();
        }
        return set;
    }
}

