/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.health;

import java.io.Closeable;
import java.time.Clock;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.logging.ESLogMessage;
import org.elasticsearch.common.scheduler.SchedulerEngine;
import org.elasticsearch.common.scheduler.TimeValueSchedule;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.health.HealthIndicatorResult;
import org.elasticsearch.health.HealthService;
import org.elasticsearch.health.HealthStatus;
import org.elasticsearch.health.node.selection.HealthNode;

public class HealthPeriodicLogger
implements ClusterStateListener,
Closeable,
SchedulerEngine.Listener {
    public static final String HEALTH_FIELD_PREFIX = "elasticsearch.health";
    public static final String MESSAGE_FIELD = "message";
    public static final Setting<TimeValue> POLL_INTERVAL_SETTING = Setting.timeSetting("health.periodic_logger.poll_interval", TimeValue.timeValueSeconds((long)60L), TimeValue.timeValueSeconds((long)15L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Boolean> ENABLED_SETTING = Setting.boolSetting("health.periodic_logger.enabled", false, Setting.Property.Dynamic, Setting.Property.NodeScope);
    protected static final String HEALTH_PERIODIC_LOGGER_JOB_NAME = "health_periodic_logger";
    private final Settings settings;
    private final ClusterService clusterService;
    private final Client client;
    private final HealthService healthService;
    private final Clock clock;
    volatile boolean isHealthNode = false;
    private final AtomicBoolean currentlyRunning = new AtomicBoolean(false);
    private final SetOnce<SchedulerEngine> scheduler = new SetOnce();
    private volatile TimeValue pollInterval;
    private volatile boolean enabled;
    private static final Logger logger = LogManager.getLogger(HealthPeriodicLogger.class);
    final ActionListener<List<HealthIndicatorResult>> resultsListener = new ActionListener<List<HealthIndicatorResult>>(){

        @Override
        public void onResponse(List<HealthIndicatorResult> healthIndicatorResults) {
            try {
                Map<String, Object> resultsMap = HealthPeriodicLogger.convertToLoggedFields(healthIndicatorResults);
                if (!resultsMap.isEmpty()) {
                    ESLogMessage msg = new ESLogMessage().withFields(resultsMap);
                    logger.info((Message)msg);
                }
            }
            catch (Exception e) {
                logger.warn("Health Periodic Logger error:{}", (Object)e.toString());
            }
        }

        @Override
        public void onFailure(Exception e) {
            logger.warn("Health Periodic Logger error:{}", (Object)e.toString());
        }
    };

    public static HealthPeriodicLogger create(Settings settings, ClusterService clusterService, Client client, HealthService healthService) {
        HealthPeriodicLogger logger = new HealthPeriodicLogger(settings, clusterService, client, healthService);
        logger.registerListeners();
        return logger;
    }

    private HealthPeriodicLogger(Settings settings, ClusterService clusterService, Client client, HealthService healthService) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.client = client;
        this.healthService = healthService;
        this.clock = Clock.systemUTC();
        this.pollInterval = POLL_INTERVAL_SETTING.get(settings);
        this.enabled = ENABLED_SETTING.get(settings);
    }

    private void registerListeners() {
        if (this.enabled) {
            this.clusterService.addListener(this);
        }
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(ENABLED_SETTING, this::enable);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(POLL_INTERVAL_SETTING, this::updatePollInterval);
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (event.state().blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        DiscoveryNode healthNode = HealthNode.findHealthNode(event.state());
        if (healthNode == null) {
            this.isHealthNode = false;
            this.maybeCancelJob();
            return;
        }
        boolean isCurrentlyHealthNode = healthNode.getId().equals(this.clusterService.localNode().getId());
        if (this.isHealthNode != isCurrentlyHealthNode) {
            this.isHealthNode = isCurrentlyHealthNode;
            if (this.isHealthNode) {
                this.maybeScheduleJob();
            } else {
                this.maybeCancelJob();
            }
        }
    }

    @Override
    public void close() {
        SchedulerEngine engine = (SchedulerEngine)this.scheduler.get();
        if (engine != null) {
            engine.stop();
        }
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        if (event.getJobName().equals(HEALTH_PERIODIC_LOGGER_JOB_NAME) && this.enabled) {
            this.tryToLogHealth();
        }
    }

    void tryToLogHealth() {
        if (!this.currentlyRunning.compareAndExchange(false, true)) {
            RunOnce release = new RunOnce(() -> this.currentlyRunning.set(false));
            try {
                ActionListener<List<HealthIndicatorResult>> listenerWithRelease = ActionListener.runAfter(this.resultsListener, release);
                this.healthService.getHealth(this.client, null, false, 0, listenerWithRelease);
            }
            catch (Exception e) {
                logger.warn(() -> "The health periodic logger encountered an error.", (Throwable)e);
                release.run();
            }
        }
    }

    SchedulerEngine getScheduler() {
        return (SchedulerEngine)this.scheduler.get();
    }

    static Map<String, Object> convertToLoggedFields(List<HealthIndicatorResult> indicatorResults) {
        if (indicatorResults == null || indicatorResults.isEmpty()) {
            return Map.of();
        }
        HashMap<String, Object> result = new HashMap<String, Object>();
        HealthStatus status = HealthStatus.merge(indicatorResults.stream().map(HealthIndicatorResult::status));
        result.put(String.format(Locale.ROOT, "%s.overall.status", HEALTH_FIELD_PREFIX), status.xContentValue());
        indicatorResults.forEach(indicatorResult -> result.put(String.format(Locale.ROOT, "%s.%s.status", HEALTH_FIELD_PREFIX, indicatorResult.name()), indicatorResult.status().xContentValue()));
        List<String> nonGreen = indicatorResults.stream().filter(p -> p.status() != HealthStatus.GREEN).map(HealthIndicatorResult::name).sorted().toList();
        if (nonGreen.isEmpty()) {
            result.put(MESSAGE_FIELD, String.format(Locale.ROOT, "health=%s", status.xContentValue()));
        } else {
            result.put(MESSAGE_FIELD, String.format(Locale.ROOT, "health=%s [%s]", status.xContentValue(), String.join((CharSequence)",", nonGreen)));
        }
        return result;
    }

    private void maybeScheduleJob() {
        if (!this.isHealthNode) {
            return;
        }
        if (!this.enabled) {
            return;
        }
        if (this.isClusterServiceStoppedOrClosed()) {
            logger.trace("Skipping scheduling a health periodic logger job due to the cluster lifecycle state being: [{}] ", (Object)this.clusterService.lifecycleState());
            return;
        }
        if (this.scheduler.get() == null) {
            this.scheduler.set((Object)new SchedulerEngine(this.settings, this.clock));
            ((SchedulerEngine)this.scheduler.get()).register(this);
        }
        assert (this.scheduler.get() != null) : "scheduler should be available";
        SchedulerEngine.Job scheduledJob = new SchedulerEngine.Job(HEALTH_PERIODIC_LOGGER_JOB_NAME, new TimeValueSchedule(this.pollInterval));
        ((SchedulerEngine)this.scheduler.get()).add(scheduledJob);
    }

    private void maybeCancelJob() {
        if (this.scheduler.get() != null) {
            ((SchedulerEngine)this.scheduler.get()).remove(HEALTH_PERIODIC_LOGGER_JOB_NAME);
        }
    }

    private void enable(boolean enabled) {
        this.enabled = enabled;
        if (enabled) {
            this.clusterService.addListener(this);
            this.maybeScheduleJob();
        } else {
            this.clusterService.removeListener(this);
            this.maybeCancelJob();
        }
    }

    private void updatePollInterval(TimeValue newInterval) {
        this.pollInterval = newInterval;
        this.maybeScheduleJob();
    }

    private boolean isClusterServiceStoppedOrClosed() {
        Lifecycle.State state = this.clusterService.lifecycleState();
        return state == Lifecycle.State.STOPPED || state == Lifecycle.State.CLOSED;
    }
}

