/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.inference.registry;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.core.Strings;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.index.query.ConstantScoreQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.reindex.DeleteByQueryAction;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.inference.Model;
import org.elasticsearch.inference.TaskType;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;
import org.elasticsearch.xpack.inference.services.ServiceUtils;

public class ModelRegistry {
    private static final String TASK_TYPE_FIELD = "task_type";
    private static final String MODEL_ID_FIELD = "model_id";
    private static final Logger logger = LogManager.getLogger(ModelRegistry.class);
    private final OriginSettingClient client;

    public ModelRegistry(Client client) {
        this.client = new OriginSettingClient(client, "inference");
    }

    public void getModelWithSecrets(String modelId, ActionListener<UnparsedModel> listener) {
        ActionListener searchListener = ActionListener.wrap(searchResponse -> {
            if (searchResponse.getHits().getHits().length == 0) {
                listener.onFailure((Exception)new ResourceNotFoundException("Model not found [{}]", new Object[]{modelId}));
                return;
            }
            SearchHit[] hits = searchResponse.getHits().getHits();
            listener.onResponse((Object)UnparsedModel.unparsedModelFromMap(this.createModelConfigMap(hits, modelId)));
        }, arg_0 -> listener.onFailure(arg_0));
        QueryBuilder queryBuilder = this.documentIdQuery(modelId);
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*", ".secrets-inference*"}).setQuery(queryBuilder).setSize(2).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getModel(String modelId, ActionListener<UnparsedModel> listener) {
        ActionListener searchListener = ActionListener.wrap(searchResponse -> {
            if (searchResponse.getHits().getHits().length == 0) {
                listener.onFailure((Exception)new ResourceNotFoundException("Model not found [{}]", new Object[]{modelId}));
                return;
            }
            SearchHit[] hits = searchResponse.getHits().getHits();
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(hits).stream().map(UnparsedModel::unparsedModelFromMap).toList();
            assert (modelConfigs.size() == 1);
            listener.onResponse((Object)modelConfigs.get(0));
        }, arg_0 -> listener.onFailure(arg_0));
        QueryBuilder queryBuilder = this.documentIdQuery(modelId);
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery(queryBuilder).setSize(1).setTrackTotalHits(false).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getModelsByTaskType(TaskType taskType, ActionListener<List<UnparsedModel>> listener) {
        ActionListener searchListener = ActionListener.wrap(searchResponse -> {
            if (searchResponse.getHits().getHits().length == 0) {
                listener.onResponse(List.of());
                return;
            }
            SearchHit[] hits = searchResponse.getHits().getHits();
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(hits).stream().map(UnparsedModel::unparsedModelFromMap).toList();
            listener.onResponse(modelConfigs);
        }, arg_0 -> listener.onFailure(arg_0));
        ConstantScoreQueryBuilder queryBuilder = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.termsQuery((String)TASK_TYPE_FIELD, (String[])new String[]{taskType.toString()}));
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery((QueryBuilder)queryBuilder).setSize(10000).setTrackTotalHits(false).addSort(MODEL_ID_FIELD, SortOrder.ASC).request();
        this.client.search(modelSearch, searchListener);
    }

    public void getAllModels(ActionListener<List<UnparsedModel>> listener) {
        ActionListener searchListener = ActionListener.wrap(searchResponse -> {
            if (searchResponse.getHits().getHits().length == 0) {
                listener.onResponse(List.of());
                return;
            }
            SearchHit[] hits = searchResponse.getHits().getHits();
            List<UnparsedModel> modelConfigs = this.parseHitsAsModels(hits).stream().map(UnparsedModel::unparsedModelFromMap).toList();
            listener.onResponse(modelConfigs);
        }, arg_0 -> listener.onFailure(arg_0));
        ConstantScoreQueryBuilder queryBuilder = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.existsQuery((String)TASK_TYPE_FIELD));
        SearchRequest modelSearch = (SearchRequest)this.client.prepareSearch(new String[]{".inference*"}).setQuery((QueryBuilder)queryBuilder).setSize(10000).setTrackTotalHits(false).addSort(MODEL_ID_FIELD, SortOrder.ASC).request();
        this.client.search(modelSearch, searchListener);
    }

    private List<ModelConfigMap> parseHitsAsModels(SearchHit[] hits) {
        ArrayList<ModelConfigMap> modelConfigs = new ArrayList<ModelConfigMap>();
        for (SearchHit hit : hits) {
            modelConfigs.add(new ModelConfigMap(hit.getSourceAsMap(), Map.of()));
        }
        return modelConfigs;
    }

    private ModelConfigMap createModelConfigMap(SearchHit[] hits, String modelId) {
        Map mappedHits = Arrays.stream(hits).collect(Collectors.toMap(hit -> {
            if (hit.getIndex().startsWith(".inference")) {
                return ".inference";
            }
            if (hit.getIndex().startsWith(".secrets-inference")) {
                return ".secrets-inference";
            }
            logger.warn(Strings.format((String)"Found invalid index for model [%s] at index [%s]", (Object[])new Object[]{modelId, hit.getIndex()}));
            throw new IllegalArgumentException(Strings.format((String)"Invalid result while loading model [%s] index: [%s]. Try deleting and reinitializing the service", (Object[])new Object[]{modelId, hit.getIndex()}));
        }, Function.identity()));
        if (!mappedHits.containsKey(".inference") || !mappedHits.containsKey(".secrets-inference") || mappedHits.size() > 2) {
            logger.warn(Strings.format((String)"Failed to load model [%s], found model parts from index prefixes: [%s]", (Object[])new Object[]{modelId, mappedHits.keySet()}));
            throw new IllegalStateException(Strings.format((String)"Failed to load model, model [%s] is in an invalid state. Try deleting and reinitializing the service", (Object[])new Object[]{modelId}));
        }
        return new ModelConfigMap(((SearchHit)mappedHits.get(".inference")).getSourceAsMap(), ((SearchHit)mappedHits.get(".secrets-inference")).getSourceAsMap());
    }

    public void storeModel(Model model, ActionListener<Boolean> listener) {
        ActionListener<BulkResponse> bulkResponseActionListener = ModelRegistry.getStoreModelListener(model, listener);
        IndexRequest configRequest = ModelRegistry.createIndexRequest(Model.documentId((String)model.getConfigurations().getModelId()), ".inference", (ToXContentObject)model.getConfigurations(), false);
        IndexRequest secretsRequest = ModelRegistry.createIndexRequest(Model.documentId((String)model.getConfigurations().getModelId()), ".secrets-inference", (ToXContentObject)model.getSecrets(), false);
        ((BulkRequestBuilder)this.client.prepareBulk().add(configRequest).add(secretsRequest).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE)).execute(bulkResponseActionListener);
    }

    private static ActionListener<BulkResponse> getStoreModelListener(Model model, ActionListener<Boolean> listener) {
        return ActionListener.wrap(bulkItemResponses -> {
            String modelId = model.getConfigurations().getModelId();
            if (bulkItemResponses.getItems().length == 0) {
                logger.warn(Strings.format((String)"Storing model [%s] failed, no items were received from the bulk response", (Object[])new Object[]{modelId}));
                listener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to store inference model [%s], invalid bulk response received. Try reinitializing the service", (Object[])new Object[]{modelId}), RestStatus.INTERNAL_SERVER_ERROR, new Object[0]));
                return;
            }
            BulkItemResponse.Failure failure = ModelRegistry.getFirstBulkFailure(bulkItemResponses);
            if (failure == null) {
                listener.onResponse((Object)true);
                return;
            }
            ModelRegistry.logBulkFailures(model.getConfigurations().getModelId(), bulkItemResponses);
            if (ExceptionsHelper.unwrapCause((Throwable)failure.getCause()) instanceof VersionConflictEngineException) {
                listener.onFailure((Exception)new ResourceAlreadyExistsException("Inference model [{}] already exists", new Object[]{modelId}));
                return;
            }
            listener.onFailure((Exception)new ElasticsearchStatusException(Strings.format((String)"Failed to store inference model [%s]", (Object[])new Object[]{modelId}), RestStatus.INTERNAL_SERVER_ERROR, (Throwable)failure.getCause(), new Object[0]));
        }, e -> {
            String errorMessage = Strings.format((String)"Failed to store inference model [%s]", (Object[])new Object[]{model.getConfigurations().getModelId()});
            logger.warn(errorMessage, (Throwable)e);
            listener.onFailure((Exception)new ElasticsearchStatusException(errorMessage, RestStatus.INTERNAL_SERVER_ERROR, (Throwable)e, new Object[0]));
        });
    }

    private static void logBulkFailures(String modelId, BulkResponse bulkResponse) {
        for (BulkItemResponse item : bulkResponse.getItems()) {
            if (!item.isFailed()) continue;
            logger.warn(Strings.format((String)"Failed to store inference model [%s] index: [%s] bulk failure message [%s]", (Object[])new Object[]{modelId, item.getIndex(), item.getFailureMessage()}));
        }
    }

    private static BulkItemResponse.Failure getFirstBulkFailure(BulkResponse bulkResponse) {
        for (BulkItemResponse item : bulkResponse.getItems()) {
            if (!item.isFailed()) continue;
            return item.getFailure();
        }
        return null;
    }

    public void deleteModel(String modelId, ActionListener<Boolean> listener) {
        DeleteByQueryRequest request = (DeleteByQueryRequest)new DeleteByQueryRequest().setAbortOnVersionConflict(false);
        request.indices(new String[]{".inference*", ".secrets-inference*"});
        request.setQuery(this.documentIdQuery(modelId));
        request.setRefresh(true);
        this.client.execute((ActionType)DeleteByQueryAction.INSTANCE, (ActionRequest)request, ActionListener.wrap(r -> listener.onResponse((Object)Boolean.TRUE), arg_0 -> listener.onFailure(arg_0)));
    }

    private static IndexRequest createIndexRequest(String docId, String indexName, ToXContentObject body, boolean allowOverwriting) {
        IndexRequest indexRequest;
        block8: {
            XContentBuilder builder = XContentFactory.jsonBuilder();
            try {
                IndexRequest request = new IndexRequest(indexName);
                XContentBuilder source = body.toXContent(builder, ToXContent.EMPTY_PARAMS);
                DocWriteRequest.OpType operation = allowOverwriting ? DocWriteRequest.OpType.INDEX : DocWriteRequest.OpType.CREATE;
                indexRequest = request.opType(operation).id(docId).source(source);
                if (builder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new ElasticsearchException(Strings.format((String)"Unexpected serialization exception for index [%s] doc [%s]", (Object[])new Object[]{indexName, docId}), (Throwable)ex, new Object[0]);
                }
            }
            builder.close();
        }
        return indexRequest;
    }

    private QueryBuilder documentIdQuery(String modelId) {
        return QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.idsQuery().addIds(new String[]{Model.documentId((String)modelId)}));
    }

    public record ModelConfigMap(Map<String, Object> config, Map<String, Object> secrets) {
    }

    public record UnparsedModel(String modelId, TaskType taskType, String service, Map<String, Object> settings, Map<String, Object> secrets) {
        public static UnparsedModel unparsedModelFromMap(ModelConfigMap modelConfigMap) {
            if (modelConfigMap.config() == null) {
                throw new ElasticsearchStatusException("Missing config map", RestStatus.BAD_REQUEST, new Object[0]);
            }
            String modelId = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), ModelRegistry.MODEL_ID_FIELD);
            String service = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), "service");
            String taskTypeStr = ServiceUtils.removeStringOrThrowIfNull(modelConfigMap.config(), TaskType.NAME);
            TaskType taskType = TaskType.fromString((String)taskTypeStr);
            return new UnparsedModel(modelId, taskType, service, modelConfigMap.config(), modelConfigMap.secrets());
        }
    }
}

