/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.Request;
import com.amazonaws.Response;
import com.amazonaws.metrics.RequestMetricCollector;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.StorageClass;
import com.amazonaws.util.AWSRequestMetrics;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobContainer;
import org.elasticsearch.repositories.s3.S3Service;
import org.elasticsearch.repositories.s3.SocketAccess;
import org.elasticsearch.threadpool.ThreadPool;

class S3BlobStore
implements BlobStore {
    private static final int MAX_BULK_DELETES = 1000;
    private static final Logger logger = LogManager.getLogger(S3BlobStore.class);
    private final S3Service service;
    private final BigArrays bigArrays;
    private final String bucket;
    private final ByteSizeValue bufferSize;
    private final boolean serverSideEncryption;
    private final CannedAccessControlList cannedACL;
    private final StorageClass storageClass;
    private final RepositoryMetadata repositoryMetadata;
    private final ThreadPool threadPool;
    private final Executor snapshotExecutor;
    private final Stats stats = new Stats();
    final RequestMetricCollector getMetricCollector;
    final RequestMetricCollector listMetricCollector;
    final RequestMetricCollector putMetricCollector;
    final RequestMetricCollector multiPartUploadMetricCollector;
    final RequestMetricCollector deleteMetricCollector;
    final RequestMetricCollector abortPartUploadMetricCollector;

    S3BlobStore(S3Service service, String bucket, boolean serverSideEncryption, ByteSizeValue bufferSize, String cannedACL, String storageClass, RepositoryMetadata repositoryMetadata, BigArrays bigArrays, ThreadPool threadPool) {
        this.service = service;
        this.bigArrays = bigArrays;
        this.bucket = bucket;
        this.serverSideEncryption = serverSideEncryption;
        this.bufferSize = bufferSize;
        this.cannedACL = S3BlobStore.initCannedACL(cannedACL);
        this.storageClass = S3BlobStore.initStorageClass(storageClass);
        this.repositoryMetadata = repositoryMetadata;
        this.threadPool = threadPool;
        this.snapshotExecutor = threadPool.executor("snapshot");
        this.getMetricCollector = new IgnoreNoResponseMetricsCollector(){

            @Override
            public void collectMetrics(Request<?> request) {
                assert (request.getHttpMethod().name().equals("GET"));
                S3BlobStore.this.stats.getCount.addAndGet(S3BlobStore.this.getRequestCount(request));
            }
        };
        this.listMetricCollector = new IgnoreNoResponseMetricsCollector(){

            @Override
            public void collectMetrics(Request<?> request) {
                assert (request.getHttpMethod().name().equals("GET"));
                S3BlobStore.this.stats.listCount.addAndGet(S3BlobStore.this.getRequestCount(request));
            }
        };
        this.putMetricCollector = new IgnoreNoResponseMetricsCollector(){

            @Override
            public void collectMetrics(Request<?> request) {
                assert (request.getHttpMethod().name().equals("PUT"));
                S3BlobStore.this.stats.putCount.addAndGet(S3BlobStore.this.getRequestCount(request));
            }
        };
        this.multiPartUploadMetricCollector = new IgnoreNoResponseMetricsCollector(){

            @Override
            public void collectMetrics(Request<?> request) {
                assert (request.getHttpMethod().name().equals("PUT") || request.getHttpMethod().name().equals("POST"));
                S3BlobStore.this.stats.postCount.addAndGet(S3BlobStore.this.getRequestCount(request));
            }
        };
        this.deleteMetricCollector = new IgnoreNoResponseMetricsCollector(){

            @Override
            public void collectMetrics(Request<?> request) {
                assert (request.getHttpMethod().name().equals("POST"));
                S3BlobStore.this.stats.deleteCount.addAndGet(S3BlobStore.this.getRequestCount(request));
            }
        };
        this.abortPartUploadMetricCollector = new IgnoreNoResponseMetricsCollector(){

            @Override
            public void collectMetrics(Request<?> request) {
                assert (request.getHttpMethod().name().equals("DELETE"));
                S3BlobStore.this.stats.abortCount.addAndGet(S3BlobStore.this.getRequestCount(request));
            }
        };
    }

    public Executor getSnapshotExecutor() {
        return this.snapshotExecutor;
    }

    public TimeValue getCompareAndExchangeTimeToLive() {
        return this.service.compareAndExchangeTimeToLive;
    }

    public TimeValue getCompareAndExchangeAntiContentionDelay() {
        return this.service.compareAndExchangeAntiContentionDelay;
    }

    private long getRequestCount(Request<?> request) {
        Number requestCount = request.getAWSRequestMetrics().getTimingInfo().getCounter(AWSRequestMetrics.Field.RequestCount.name());
        if (requestCount == null) {
            logger.warn("Expected request count to be tracked for request [{}] but found not count.", request);
            return 0L;
        }
        return requestCount.longValue();
    }

    public String toString() {
        return this.bucket;
    }

    public AmazonS3Reference clientReference() {
        return this.service.client(this.repositoryMetadata);
    }

    int getMaxRetries() {
        return this.service.settings((RepositoryMetadata)this.repositoryMetadata).maxRetries;
    }

    public String bucket() {
        return this.bucket;
    }

    public BigArrays bigArrays() {
        return this.bigArrays;
    }

    public boolean serverSideEncryption() {
        return this.serverSideEncryption;
    }

    public long bufferSizeInBytes() {
        return this.bufferSize.getBytes();
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new S3BlobContainer(path, this);
    }

    public void deleteBlobsIgnoringIfNotExists(OperationPurpose purpose, Iterator<String> blobNames) throws IOException {
        if (!blobNames.hasNext()) {
            return;
        }
        ArrayList partition = new ArrayList();
        try (AmazonS3Reference clientReference = this.clientReference();){
            AtomicReference aex = new AtomicReference();
            SocketAccess.doPrivilegedVoid(() -> {
                blobNames.forEachRemaining(key -> {
                    partition.add(key);
                    if (partition.size() == 1000) {
                        this.deletePartition(purpose, clientReference, partition, aex);
                        partition.clear();
                    }
                });
                if (!partition.isEmpty()) {
                    this.deletePartition(purpose, clientReference, partition, aex);
                }
            });
            if (aex.get() != null) {
                throw (Exception)aex.get();
            }
        }
        catch (Exception e) {
            throw new IOException("Failed to delete blobs " + partition.stream().limit(10L).toList(), e);
        }
    }

    private void deletePartition(OperationPurpose purpose, AmazonS3Reference clientReference, List<String> partition, AtomicReference<Exception> aex) {
        try {
            clientReference.client().deleteObjects(S3BlobStore.bulkDelete(purpose, this, partition));
        }
        catch (MultiObjectDeleteException e) {
            logger.warn(() -> Strings.format((String)"Failed to delete some blobs %s", (Object[])new Object[]{e.getErrors().stream().map(err -> "[" + err.getKey() + "][" + err.getCode() + "][" + err.getMessage() + "]").toList()}), (Throwable)e);
            aex.set((Exception)ExceptionsHelper.useOrSuppress((Throwable)aex.get(), (Throwable)e));
        }
        catch (AmazonClientException e) {
            aex.set((Exception)ExceptionsHelper.useOrSuppress((Throwable)aex.get(), (Throwable)e));
        }
    }

    private static DeleteObjectsRequest bulkDelete(OperationPurpose purpose, S3BlobStore blobStore, List<String> blobs) {
        return (DeleteObjectsRequest)new DeleteObjectsRequest(blobStore.bucket()).withKeys(blobs.toArray(org.elasticsearch.common.Strings.EMPTY_ARRAY)).withQuiet(true).withRequestMetricCollector(blobStore.deleteMetricCollector);
    }

    public void close() throws IOException {
        this.service.close();
    }

    public Map<String, Long> stats() {
        return this.stats.toMap();
    }

    public CannedAccessControlList getCannedACL() {
        return this.cannedACL;
    }

    public StorageClass getStorageClass() {
        return this.storageClass;
    }

    public static StorageClass initStorageClass(String storageClass) {
        if (storageClass == null || storageClass.equals("")) {
            return StorageClass.Standard;
        }
        try {
            StorageClass _storageClass = StorageClass.fromValue((String)storageClass.toUpperCase(Locale.ENGLISH));
            if (_storageClass.equals((Object)StorageClass.Glacier)) {
                throw new BlobStoreException("Glacier storage class is not supported");
            }
            return _storageClass;
        }
        catch (IllegalArgumentException illegalArgumentException) {
            throw new BlobStoreException("`" + storageClass + "` is not a valid S3 Storage Class.");
        }
    }

    public static CannedAccessControlList initCannedACL(String cannedACL) {
        if (cannedACL == null || cannedACL.equals("")) {
            return CannedAccessControlList.Private;
        }
        for (CannedAccessControlList cur : CannedAccessControlList.values()) {
            if (!cur.toString().equalsIgnoreCase(cannedACL)) continue;
            return cur;
        }
        throw new BlobStoreException("cannedACL is not valid: [" + cannedACL + "]");
    }

    ThreadPool getThreadPool() {
        return this.threadPool;
    }

    static class Stats {
        final AtomicLong listCount = new AtomicLong();
        final AtomicLong getCount = new AtomicLong();
        final AtomicLong putCount = new AtomicLong();
        final AtomicLong postCount = new AtomicLong();
        final AtomicLong deleteCount = new AtomicLong();
        final AtomicLong abortCount = new AtomicLong();

        Stats() {
        }

        Map<String, Long> toMap() {
            HashMap<String, Long> results = new HashMap<String, Long>();
            results.put("GetObject", this.getCount.get());
            results.put("ListObjects", this.listCount.get());
            results.put("PutObject", this.putCount.get());
            results.put("PutMultipartObject", this.postCount.get());
            results.put("DeleteObjects", this.deleteCount.get());
            results.put("AbortMultipartObject", this.abortCount.get());
            return results;
        }
    }

    private static abstract class IgnoreNoResponseMetricsCollector
    extends RequestMetricCollector {
        private IgnoreNoResponseMetricsCollector() {
        }

        public final void collectMetrics(Request<?> request, Response<?> response) {
            if (response != null) {
                this.collectMetrics(request);
            }
        }

        protected abstract void collectMetrics(Request<?> var1);
    }
}

