/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSCredentialsProviderChain;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.AnonymousAWSCredentials;
import com.amazonaws.auth.EC2ContainerCredentialsProviderWrapper;
import com.amazonaws.auth.STSAssumeRoleWithWebIdentitySessionCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.http.IdleConnectionReaper;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClient;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Clock;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.env.Environment;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BasicCredentials;
import org.elasticsearch.repositories.s3.S3ClientSettings;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.SocketAccess;

class S3Service
implements Closeable {
    private static final Logger LOGGER = LogManager.getLogger(S3Service.class);
    private volatile Map<S3ClientSettings, AmazonS3Reference> clientsCache = Collections.emptyMap();
    private volatile Map<String, S3ClientSettings> staticClientSettings = Map.of("default", S3ClientSettings.getClientSettings(Settings.EMPTY, "default"));
    private volatile Map<Settings, S3ClientSettings> derivedClientSettings = Collections.emptyMap();
    final CustomWebIdentityTokenCredentialsProvider webIdentityTokenCredentialsProvider;

    S3Service(Environment environment) {
        this.webIdentityTokenCredentialsProvider = new CustomWebIdentityTokenCredentialsProvider(environment, System::getenv, System::getProperty, Clock.systemUTC());
    }

    public synchronized void refreshAndClearCache(Map<String, S3ClientSettings> clientsSettings) {
        this.releaseCachedClients();
        this.staticClientSettings = Maps.ofEntries(clientsSettings.entrySet());
        this.derivedClientSettings = Collections.emptyMap();
        assert (this.staticClientSettings.containsKey("default")) : "always at least have 'default'";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AmazonS3Reference client(RepositoryMetadata repositoryMetadata) {
        S3ClientSettings clientSettings = this.settings(repositoryMetadata);
        AmazonS3Reference clientReference = this.clientsCache.get(clientSettings);
        if (clientReference != null && clientReference.tryIncRef()) {
            return clientReference;
        }
        S3Service s3Service = this;
        synchronized (s3Service) {
            AmazonS3Reference existing = this.clientsCache.get(clientSettings);
            if (existing != null && existing.tryIncRef()) {
                return existing;
            }
            AmazonS3Reference clientReference2 = new AmazonS3Reference(this.buildClient(clientSettings));
            clientReference2.incRef();
            this.clientsCache = Maps.copyMapWithAddedEntry(this.clientsCache, (Object)clientSettings, (Object)((Object)clientReference2));
            return clientReference2;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    S3ClientSettings settings(RepositoryMetadata repositoryMetadata) {
        Settings settings = repositoryMetadata.settings();
        S3ClientSettings existing = this.derivedClientSettings.get(settings);
        if (existing != null) {
            return existing;
        }
        String clientName = (String)S3Repository.CLIENT_NAME.get(settings);
        S3ClientSettings staticSettings = this.staticClientSettings.get(clientName);
        if (staticSettings != null) {
            S3Service s3Service = this;
            synchronized (s3Service) {
                S3ClientSettings existing2 = this.derivedClientSettings.get(settings);
                if (existing2 != null) {
                    return existing2;
                }
                S3ClientSettings newSettings = staticSettings.refine(settings);
                this.derivedClientSettings = Maps.copyMapWithAddedOrReplacedEntry(this.derivedClientSettings, (Object)settings, (Object)newSettings);
                return newSettings;
            }
        }
        throw new IllegalArgumentException("Unknown s3 client name [" + clientName + "]. Existing client configs: " + Strings.collectionToDelimitedString(this.staticClientSettings.keySet(), (String)","));
    }

    AmazonS3 buildClient(S3ClientSettings clientSettings) {
        AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard();
        builder.withCredentials(S3Service.buildCredentials(LOGGER, clientSettings, this.webIdentityTokenCredentialsProvider));
        builder.withClientConfiguration(S3Service.buildConfiguration(clientSettings));
        Object endpoint = Strings.hasLength((String)clientSettings.endpoint) ? clientSettings.endpoint : "s3.amazonaws.com";
        if (!(((String)endpoint).startsWith("http://") || ((String)endpoint).startsWith("https://"))) {
            endpoint = clientSettings.protocol.toString() + "://" + (String)endpoint;
        }
        String region = Strings.hasLength((String)clientSettings.region) ? clientSettings.region : null;
        LOGGER.debug("using endpoint [{}] and region [{}]", endpoint, (Object)region);
        builder.withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration((String)endpoint, region));
        if (clientSettings.pathStyleAccess) {
            builder.enablePathStyleAccess();
        }
        if (clientSettings.disableChunkedEncoding) {
            builder.disableChunkedEncoding();
        }
        return SocketAccess.doPrivileged(() -> ((AmazonS3ClientBuilder)builder).build());
    }

    static ClientConfiguration buildConfiguration(S3ClientSettings clientSettings) {
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        clientConfiguration.setResponseMetadataCacheSize(0);
        clientConfiguration.setProtocol(clientSettings.protocol);
        if (Strings.hasText((String)clientSettings.proxyHost)) {
            clientConfiguration.setProxyHost(clientSettings.proxyHost);
            clientConfiguration.setProxyPort(clientSettings.proxyPort);
            clientConfiguration.setProxyUsername(clientSettings.proxyUsername);
            clientConfiguration.setProxyPassword(clientSettings.proxyPassword);
        }
        if (Strings.hasLength((String)clientSettings.signerOverride)) {
            clientConfiguration.setSignerOverride(clientSettings.signerOverride);
        }
        clientConfiguration.setMaxErrorRetry(clientSettings.maxRetries);
        clientConfiguration.setUseThrottleRetries(clientSettings.throttleRetries);
        clientConfiguration.setSocketTimeout(clientSettings.readTimeoutMillis);
        return clientConfiguration;
    }

    static AWSCredentialsProvider buildCredentials(Logger logger, S3ClientSettings clientSettings, CustomWebIdentityTokenCredentialsProvider webIdentityTokenCredentialsProvider) {
        S3BasicCredentials credentials = clientSettings.credentials;
        if (credentials == null) {
            if (webIdentityTokenCredentialsProvider.isActive()) {
                logger.debug("Using a custom provider chain of Web Identity Token and instance profile credentials");
                return new PrivilegedAWSCredentialsProvider((AWSCredentialsProvider)new AWSCredentialsProviderChain(List.of(new ErrorLoggingCredentialsProvider(webIdentityTokenCredentialsProvider, LOGGER), new ErrorLoggingCredentialsProvider((AWSCredentialsProvider)new EC2ContainerCredentialsProviderWrapper(), LOGGER))));
            }
            logger.debug("Using instance profile credentials");
            return new PrivilegedAWSCredentialsProvider((AWSCredentialsProvider)new EC2ContainerCredentialsProviderWrapper());
        }
        logger.debug("Using basic key/secret credentials");
        return new AWSStaticCredentialsProvider((AWSCredentials)credentials);
    }

    private synchronized void releaseCachedClients() {
        for (AmazonS3Reference clientReference : this.clientsCache.values()) {
            clientReference.decRef();
        }
        this.clientsCache = Collections.emptyMap();
        this.derivedClientSettings = Collections.emptyMap();
        IdleConnectionReaper.shutdown();
    }

    @Override
    public void close() throws IOException {
        this.releaseCachedClients();
        this.webIdentityTokenCredentialsProvider.shutdown();
    }

    static class CustomWebIdentityTokenCredentialsProvider
    implements AWSCredentialsProvider {
        private static final String STS_HOSTNAME = "https://sts.amazonaws.com";
        private STSAssumeRoleWithWebIdentitySessionCredentialsProvider credentialsProvider;
        private AWSSecurityTokenService stsClient;

        CustomWebIdentityTokenCredentialsProvider(Environment environment, SystemEnvironment systemEnvironment, JvmEnvironment jvmEnvironment, Clock clock) {
            if (systemEnvironment.getEnv("AWS_WEB_IDENTITY_TOKEN_FILE") == null) {
                return;
            }
            Path webIdentityTokenFileSymlink = environment.configFile().resolve("repository-s3/aws-web-identity-token-file");
            if (!Files.exists(webIdentityTokenFileSymlink, new LinkOption[0])) {
                LOGGER.warn("Cannot use AWS Web Identity Tokens: AWS_WEB_IDENTITY_TOKEN_FILE is defined but no corresponding symlink exists in the config directory");
                return;
            }
            if (!Files.isReadable(webIdentityTokenFileSymlink)) {
                throw new IllegalStateException("Unable to read a Web Identity Token symlink in the config directory");
            }
            String roleArn = systemEnvironment.getEnv("AWS_ROLE_ARN");
            if (roleArn == null) {
                LOGGER.warn("Unable to use a web identity token for authentication. The AWS_WEB_IDENTITY_TOKEN_FILE environment variable is set, but either AWS_ROLE_ARN is missing");
                return;
            }
            String roleSessionName = Objects.requireNonNullElseGet(systemEnvironment.getEnv("AWS_ROLE_SESSION_NAME"), () -> "aws-sdk-java-" + clock.millis());
            AWSSecurityTokenServiceClientBuilder stsClientBuilder = AWSSecurityTokenServiceClient.builder();
            String customStsEndpoint = jvmEnvironment.getProperty("com.amazonaws.sdk.stsMetadataServiceEndpointOverride", STS_HOSTNAME);
            stsClientBuilder.withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(customStsEndpoint, null));
            stsClientBuilder.withCredentials((AWSCredentialsProvider)new AWSStaticCredentialsProvider((AWSCredentials)new AnonymousAWSCredentials()));
            this.stsClient = SocketAccess.doPrivileged(() -> ((AWSSecurityTokenServiceClientBuilder)stsClientBuilder).build());
            try {
                this.credentialsProvider = new STSAssumeRoleWithWebIdentitySessionCredentialsProvider.Builder(roleArn, roleSessionName, webIdentityTokenFileSymlink.toString()).withStsClient(this.stsClient).build();
            }
            catch (Exception e) {
                this.stsClient.shutdown();
                throw e;
            }
        }

        boolean isActive() {
            return this.credentialsProvider != null;
        }

        public AWSCredentials getCredentials() {
            Objects.requireNonNull(this.credentialsProvider, "credentialsProvider is not set");
            return this.credentialsProvider.getCredentials();
        }

        public void refresh() {
            if (this.credentialsProvider != null) {
                this.credentialsProvider.refresh();
            }
        }

        public void shutdown() throws IOException {
            if (this.credentialsProvider != null) {
                IOUtils.close((Closeable[])new Closeable[]{this.credentialsProvider, () -> this.stsClient.shutdown()});
            }
        }
    }

    @FunctionalInterface
    static interface SystemEnvironment {
        public String getEnv(String var1);
    }

    @FunctionalInterface
    static interface JvmEnvironment {
        public String getProperty(String var1, String var2);
    }

    static class PrivilegedAWSCredentialsProvider
    implements AWSCredentialsProvider {
        private final AWSCredentialsProvider credentialsProvider;

        private PrivilegedAWSCredentialsProvider(AWSCredentialsProvider credentialsProvider) {
            this.credentialsProvider = credentialsProvider;
        }

        AWSCredentialsProvider getCredentialsProvider() {
            return this.credentialsProvider;
        }

        public AWSCredentials getCredentials() {
            return SocketAccess.doPrivileged(() -> ((AWSCredentialsProvider)this.credentialsProvider).getCredentials());
        }

        public void refresh() {
            SocketAccess.doPrivilegedVoid(() -> ((AWSCredentialsProvider)this.credentialsProvider).refresh());
        }
    }

    static class ErrorLoggingCredentialsProvider
    implements AWSCredentialsProvider {
        private final AWSCredentialsProvider delegate;
        private final Logger logger;

        ErrorLoggingCredentialsProvider(AWSCredentialsProvider delegate, Logger logger) {
            this.delegate = Objects.requireNonNull(delegate);
            this.logger = Objects.requireNonNull(logger);
        }

        public AWSCredentials getCredentials() {
            try {
                return this.delegate.getCredentials();
            }
            catch (Exception e) {
                this.logger.error(() -> "Unable to load credentials from " + this.delegate, (Throwable)e);
                throw e;
            }
        }

        public void refresh() {
            try {
                this.delegate.refresh();
            }
            catch (Exception e) {
                this.logger.error(() -> "Unable to refresh " + this.delegate, (Throwable)e);
                throw e;
            }
        }
    }
}

