/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.watcher.crypto;

import java.io.IOException;
import java.io.InputStream;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.CharArrays;
import org.elasticsearch.core.Streams;
import org.elasticsearch.xpack.core.security.SecurityField;
import org.elasticsearch.xpack.core.watcher.WatcherField;

public class CryptoService {
    public static final String KEY_ALGO = "HmacSHA512";
    public static final int KEY_SIZE = 1024;
    public static final String ENCRYPTED_TEXT_PREFIX = "::es_encrypted::";
    private static final String DEFAULT_ENCRYPTION_ALGORITHM = "AES/CTR/NoPadding";
    private static final String DEFAULT_KEY_ALGORITH = "AES";
    private static final int DEFAULT_KEY_LENGTH = 128;
    private static final Setting<String> ENCRYPTION_ALGO_SETTING = new Setting(SecurityField.setting("encryption.algorithm"), s -> "AES/CTR/NoPadding", s -> s, new Setting.Property[]{Setting.Property.NodeScope});
    private static final Setting<Integer> ENCRYPTION_KEY_LENGTH_SETTING = Setting.intSetting((String)SecurityField.setting("encryption_key.length"), (int)128, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private static final Setting<String> ENCRYPTION_KEY_ALGO_SETTING = new Setting(SecurityField.setting("encryption_key.algorithm"), "AES", s -> s, new Setting.Property[]{Setting.Property.NodeScope});
    private static final Logger logger = LogManager.getLogger(CryptoService.class);
    private final SecureRandom secureRandom = new SecureRandom();
    private final String encryptionAlgorithm;
    private final int ivLength;
    private final SecretKey encryptionKey;

    public CryptoService(Settings settings) throws IOException {
        this.encryptionAlgorithm = (String)ENCRYPTION_ALGO_SETTING.get(settings);
        int keyLength = (Integer)ENCRYPTION_KEY_LENGTH_SETTING.get(settings);
        this.ivLength = keyLength / 8;
        String keyAlgorithm = (String)ENCRYPTION_KEY_ALGO_SETTING.get(settings);
        if (keyLength % 8 != 0) {
            throw new IllegalArgumentException("invalid key length [" + keyLength + "]. value must be a multiple of 8");
        }
        try (InputStream in = (InputStream)WatcherField.ENCRYPTION_KEY_SETTING.get(settings);){
            if (in == null) {
                throw new ElasticsearchException("setting [" + WatcherField.ENCRYPTION_KEY_SETTING.getKey() + "] must be set in keystore", new Object[0]);
            }
            SecretKey systemKey = CryptoService.readSystemKey(in);
            try {
                this.encryptionKey = CryptoService.encryptionKey(systemKey, keyLength, keyAlgorithm);
            }
            catch (NoSuchAlgorithmException nsae) {
                throw new ElasticsearchException("failed to start crypto service. could not load encryption key", (Throwable)nsae, new Object[0]);
            }
        }
        assert (this.encryptionKey != null) : "the encryption key should never be null";
    }

    private static SecretKey readSystemKey(InputStream in) throws IOException {
        int keySizeBytes = 128;
        byte[] keyBytes = new byte[128];
        int read = Streams.readFully((InputStream)in, (byte[])keyBytes);
        if (read != 128) {
            throw new IllegalArgumentException("key size did not match expected value; was the key generated with elasticsearch-syskeygen?");
        }
        return new SecretKeySpec(keyBytes, KEY_ALGO);
    }

    public char[] encrypt(char[] chars) {
        byte[] charBytes = CharArrays.toUtf8Bytes((char[])chars);
        String base64 = Base64.getEncoder().encodeToString(this.encryptInternal(charBytes, this.encryptionKey));
        return ENCRYPTED_TEXT_PREFIX.concat(base64).toCharArray();
    }

    public char[] decrypt(char[] chars) {
        byte[] bytes;
        if (!this.isEncrypted(chars)) {
            return chars;
        }
        String encrypted = new String(chars, ENCRYPTED_TEXT_PREFIX.length(), chars.length - ENCRYPTED_TEXT_PREFIX.length());
        try {
            bytes = Base64.getDecoder().decode(encrypted);
        }
        catch (IllegalArgumentException e) {
            throw new ElasticsearchException("unable to decode encrypted data", (Throwable)e, new Object[0]);
        }
        byte[] decrypted = this.decryptInternal(bytes, this.encryptionKey);
        return CharArrays.utf8BytesToChars((byte[])decrypted);
    }

    protected boolean isEncrypted(char[] chars) {
        return CharArrays.charsBeginsWith((String)ENCRYPTED_TEXT_PREFIX, (char[])chars);
    }

    private byte[] encryptInternal(byte[] bytes, SecretKey key) {
        byte[] iv = new byte[this.ivLength];
        this.secureRandom.nextBytes(iv);
        Cipher cipher = CryptoService.cipher(1, this.encryptionAlgorithm, key, iv);
        try {
            byte[] encrypted = cipher.doFinal(bytes);
            byte[] output = new byte[iv.length + encrypted.length];
            System.arraycopy(iv, 0, output, 0, iv.length);
            System.arraycopy(encrypted, 0, output, iv.length, encrypted.length);
            return output;
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new ElasticsearchException("error encrypting data", (Throwable)e, new Object[0]);
        }
    }

    private byte[] decryptInternal(byte[] bytes, SecretKey key) {
        if (bytes.length < this.ivLength) {
            logger.error("received data for decryption with size [{}] that is less than IV length [{}]", (Object)bytes.length, (Object)this.ivLength);
            throw new IllegalArgumentException("invalid data to decrypt");
        }
        byte[] iv = new byte[this.ivLength];
        System.arraycopy(bytes, 0, iv, 0, this.ivLength);
        byte[] data = new byte[bytes.length - this.ivLength];
        System.arraycopy(bytes, this.ivLength, data, 0, bytes.length - this.ivLength);
        Cipher cipher = CryptoService.cipher(2, this.encryptionAlgorithm, key, iv);
        try {
            return cipher.doFinal(data);
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new IllegalStateException("error decrypting data", e);
        }
    }

    private static Cipher cipher(int mode, String encryptionAlgorithm, SecretKey key, byte[] initializationVector) {
        try {
            Cipher cipher = Cipher.getInstance(encryptionAlgorithm);
            cipher.init(mode, (Key)key, new IvParameterSpec(initializationVector));
            return cipher;
        }
        catch (Exception e) {
            throw new ElasticsearchException("error creating cipher", (Throwable)e, new Object[0]);
        }
    }

    private static SecretKey encryptionKey(SecretKey systemKey, int keyLength, String algorithm) throws NoSuchAlgorithmException {
        byte[] bytes = systemKey.getEncoded();
        if (bytes.length * 8 < keyLength) {
            throw new IllegalArgumentException("at least " + keyLength + " bits should be provided as key data");
        }
        MessageDigest messageDigest = MessageDigest.getInstance("SHA-256");
        byte[] digest = messageDigest.digest(bytes);
        assert (digest.length == 32);
        if (digest.length * 8 < keyLength) {
            throw new IllegalArgumentException("requested key length is too large");
        }
        byte[] truncatedDigest = Arrays.copyOfRange(digest, 0, keyLength / 8);
        return new SecretKeySpec(truncatedDigest, algorithm);
    }

    public static void addSettings(List<Setting<?>> settings) {
        settings.add(ENCRYPTION_KEY_LENGTH_SETTING);
        settings.add(ENCRYPTION_KEY_ALGO_SETTING);
        settings.add(ENCRYPTION_ALGO_SETTING);
    }
}

