/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.security.authc;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.common.VersionId;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.util.ArrayUtils;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.Assertions;
import org.elasticsearch.core.CheckedFunction;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.transport.RemoteClusterPortSettings;
import org.elasticsearch.xcontent.ConstructingObjectParser;
import org.elasticsearch.xcontent.ParseField;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContent;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.core.security.authc.AuthenticationResult;
import org.elasticsearch.xpack.core.security.authc.CrossClusterAccessSubjectInfo;
import org.elasticsearch.xpack.core.security.authc.RealmDomain;
import org.elasticsearch.xpack.core.security.authc.Subject;
import org.elasticsearch.xpack.core.security.authc.support.AuthenticationContextSerializer;
import org.elasticsearch.xpack.core.security.authz.RoleDescriptor;
import org.elasticsearch.xpack.core.security.user.AnonymousUser;
import org.elasticsearch.xpack.core.security.user.InternalUser;
import org.elasticsearch.xpack.core.security.user.InternalUsers;
import org.elasticsearch.xpack.core.security.user.User;

public final class Authentication
implements ToXContentObject {
    private static final Logger logger = LogManager.getLogger(Authentication.class);
    private static final TransportVersion VERSION_AUTHENTICATION_TYPE = TransportVersion.fromId((int)6070099);
    public static final TransportVersion VERSION_API_KEY_ROLES_AS_BYTES = TransportVersion.V_7_9_0;
    public static final TransportVersion VERSION_REALM_DOMAINS = TransportVersion.V_8_2_0;
    public static final TransportVersion VERSION_METADATA_BEYOND_GENERIC_MAP = TransportVersion.V_8_8_0;
    private final AuthenticationType type;
    private final Subject authenticatingSubject;
    private final Subject effectiveSubject;
    private static final Map<String, CheckedFunction<StreamInput, Object, IOException>> METADATA_VALUE_READER = Map.of("_security_cross_cluster_access_authentication", Authentication::new, "_security_cross_cluster_access_role_descriptors", in -> in.readList(CrossClusterAccessSubjectInfo.RoleDescriptorsBytes::new));
    private static final Map<String, Writeable.Writer<?>> METADATA_VALUE_WRITER = Map.of("_security_cross_cluster_access_authentication", (out, v) -> ((Authentication)v).writeTo(out), "_security_cross_cluster_access_role_descriptors", (out, v) -> {
        List roleDescriptorsBytesList = (List)v;
        out.writeCollection((Collection)roleDescriptorsBytesList, (o, rdb) -> rdb.writeTo(o));
    });
    public static ConstructingObjectParser<RealmRef, Void> REALM_REF_PARSER = new ConstructingObjectParser("realm_ref", false, (args, v) -> new RealmRef((String)args[0], (String)args[1], (String)args[2], (RealmDomain)args[3]));

    private Authentication(Subject subject, AuthenticationType type) {
        this(subject, subject, type);
    }

    private Authentication(Subject effectiveSubject, Subject authenticatingSubject, AuthenticationType type) {
        this.effectiveSubject = Objects.requireNonNull(effectiveSubject, "effective subject cannot be null");
        this.authenticatingSubject = Objects.requireNonNull(authenticatingSubject, "authenticating subject cannot be null");
        this.type = Objects.requireNonNull(type, "authentication type cannot be null");
        if (Assertions.ENABLED) {
            this.checkConsistency();
        }
    }

    public Authentication(StreamInput in) throws IOException {
        Map<String, Object> metadata;
        User innerUser;
        User outerUser = AuthenticationSerializationHelper.readUserWithoutTrailingBoolean(in);
        boolean hasInnerUser = outerUser instanceof InternalUser ? false : in.readBoolean();
        if (hasInnerUser) {
            innerUser = AuthenticationSerializationHelper.readUserFrom(in);
            assert (!(innerUser instanceof InternalUser)) : "internal users cannot participate in run-as [" + innerUser + "]";
        } else {
            innerUser = null;
        }
        RealmRef authenticatedBy = new RealmRef(in);
        RealmRef lookedUpBy = in.readBoolean() ? new RealmRef(in) : null;
        assert (innerUser != null || lookedUpBy == null) : "Authentication has no inner-user, but looked-up-by is [" + lookedUpBy + "]";
        TransportVersion version = in.getTransportVersion();
        if (version.onOrAfter((VersionId)VERSION_AUTHENTICATION_TYPE)) {
            this.type = AuthenticationType.values()[in.readVInt()];
            metadata = Authentication.readMetadata(in);
        } else {
            this.type = AuthenticationType.REALM;
            metadata = Map.of();
        }
        if (innerUser != null) {
            this.authenticatingSubject = new Subject(innerUser, authenticatedBy, version, metadata);
            this.effectiveSubject = new Subject(outerUser, lookedUpBy, version, Map.of());
        } else {
            this.authenticatingSubject = this.effectiveSubject = new Subject(outerUser, authenticatedBy, version, metadata);
        }
        if (Assertions.ENABLED) {
            this.checkConsistency();
        }
    }

    public Subject getAuthenticatingSubject() {
        return this.authenticatingSubject;
    }

    public Subject getEffectiveSubject() {
        return this.effectiveSubject;
    }

    public AuthenticationType getAuthenticationType() {
        return this.type;
    }

    public boolean isRunAs() {
        return this.authenticatingSubject != this.effectiveSubject;
    }

    public boolean isFailedRunAs() {
        return this.isRunAs() && this.effectiveSubject.getRealm() == null;
    }

    public Authentication maybeRewriteForOlderVersion(TransportVersion olderVersion) {
        if (this.isCrossClusterAccess() && olderVersion.before((VersionId)RemoteClusterPortSettings.TRANSPORT_VERSION_ADVANCED_REMOTE_CLUSTER_SECURITY)) {
            throw new IllegalArgumentException("versions of Elasticsearch before [" + RemoteClusterPortSettings.TRANSPORT_VERSION_ADVANCED_REMOTE_CLUSTER_SECURITY + "] can't handle cross cluster access authentication and attempted to rewrite for [" + olderVersion + "]");
        }
        Map<String, Object> newMetadata = Authentication.maybeRewriteMetadata(olderVersion, this);
        Authentication newAuthentication = this.isRunAs() ? new Authentication(new Subject(this.effectiveSubject.getUser(), Authentication.maybeRewriteRealmRef(olderVersion, this.effectiveSubject.getRealm()), olderVersion, this.effectiveSubject.getMetadata()), new Subject(this.authenticatingSubject.getUser(), Authentication.maybeRewriteRealmRef(olderVersion, this.authenticatingSubject.getRealm()), olderVersion, newMetadata), this.type) : new Authentication(new Subject(this.authenticatingSubject.getUser(), Authentication.maybeRewriteRealmRef(olderVersion, this.authenticatingSubject.getRealm()), olderVersion, newMetadata), this.type);
        return newAuthentication;
    }

    private static Map<String, Object> maybeRewriteMetadata(TransportVersion olderVersion, Authentication authentication) {
        if (authentication.isAuthenticatedAsApiKey()) {
            return Authentication.maybeRewriteMetadataForApiKeyRoleDescriptors(olderVersion, authentication);
        }
        if (authentication.isCrossClusterAccess()) {
            return Authentication.maybeRewriteMetadataForCrossClusterAccessAuthentication(olderVersion, authentication);
        }
        return authentication.getAuthenticatingSubject().getMetadata();
    }

    public Authentication copyWithFilteredMetadataFields(Set<String> fieldsToKeep) {
        Objects.requireNonNull(fieldsToKeep);
        if (fieldsToKeep.isEmpty()) {
            return this.copyWithEmptyMetadata();
        }
        HashMap<String, Object> metadataCopy = new HashMap<String, Object>(this.authenticatingSubject.getMetadata());
        boolean metadataChanged = metadataCopy.keySet().retainAll(fieldsToKeep);
        if (logger.isTraceEnabled() && metadataChanged) {
            logger.trace("Authentication metadata [{}] for subject [{}] contains fields other than [{}]. These will be removed in the copy.", this.authenticatingSubject.getMetadata().keySet(), (Object)this.authenticatingSubject.getUser().principal(), fieldsToKeep);
        }
        return this.copyWithMetadata(Collections.unmodifiableMap(metadataCopy));
    }

    public Authentication copyWithEmptyMetadata() {
        if (logger.isTraceEnabled() && !this.authenticatingSubject.getMetadata().isEmpty()) {
            logger.trace("Authentication metadata [{}] for subject [{}] is not empty. All fields will be removed in the copy.", this.authenticatingSubject.getMetadata().keySet(), (Object)this.authenticatingSubject.getUser().principal());
        }
        return this.copyWithMetadata(Collections.emptyMap());
    }

    private Authentication copyWithMetadata(Map<String, Object> newMetadata) {
        Objects.requireNonNull(newMetadata);
        return this.isRunAs() ? new Authentication(this.effectiveSubject, new Subject(this.authenticatingSubject.getUser(), this.authenticatingSubject.getRealm(), this.authenticatingSubject.getTransportVersion(), newMetadata), this.type) : new Authentication(new Subject(this.authenticatingSubject.getUser(), this.authenticatingSubject.getRealm(), this.authenticatingSubject.getTransportVersion(), newMetadata), this.type);
    }

    public Authentication runAs(User runAs, @Nullable RealmRef lookupRealmRef) {
        assert (this.supportsRunAs(null));
        assert (!(runAs instanceof AnonymousUser));
        assert (!Authentication.hasSyntheticRealmNameOrType(lookupRealmRef)) : "should not use synthetic realm name/type for lookup realms";
        Objects.requireNonNull(runAs);
        return new Authentication(new Subject(runAs, lookupRealmRef, this.getEffectiveSubject().getTransportVersion(), Map.of()), this.authenticatingSubject, this.type);
    }

    public Authentication token() {
        assert (!this.isAuthenticatedInternally());
        assert (!this.isServiceAccount());
        assert (!this.isCrossClusterAccess());
        Authentication newTokenAuthentication = new Authentication(this.effectiveSubject, this.authenticatingSubject, AuthenticationType.TOKEN);
        return newTokenAuthentication;
    }

    public Authentication maybeAddAnonymousRoles(@Nullable AnonymousUser anonymousUser) {
        boolean shouldAddAnonymousRoleNames;
        boolean bl = shouldAddAnonymousRoleNames = anonymousUser != null && anonymousUser.enabled() && false == anonymousUser.equals(this.getEffectiveSubject().getUser()) && false == this.getEffectiveSubject().getUser() instanceof InternalUser && false == this.isApiKey() && false == this.isCrossClusterAccess() && false == this.isServiceAccount();
        if (!shouldAddAnonymousRoleNames) {
            return this;
        }
        if (anonymousUser.roles().length == 0) {
            throw new IllegalStateException("anonymous is only enabled when the anonymous user has roles");
        }
        String[] allRoleNames = ArrayUtils.concat((String[])this.getEffectiveSubject().getUser().roles(), (String[])anonymousUser.roles());
        if (this.isRunAs()) {
            User user = this.effectiveSubject.getUser();
            return new Authentication(new Subject(new User(user.principal(), allRoleNames, user.fullName(), user.email(), user.metadata(), user.enabled()), this.effectiveSubject.getRealm(), this.effectiveSubject.getTransportVersion(), this.effectiveSubject.getMetadata()), this.authenticatingSubject, this.type);
        }
        User user = this.authenticatingSubject.getUser();
        return new Authentication(new Subject(new User(user.principal(), allRoleNames, user.fullName(), user.email(), user.metadata(), user.enabled()), this.authenticatingSubject.getRealm(), this.authenticatingSubject.getTransportVersion(), this.authenticatingSubject.getMetadata()), this.type);
    }

    boolean isAssignedToDomain() {
        return this.getDomain() != null;
    }

    @Nullable
    RealmDomain getDomain() {
        if (this.isFailedRunAs()) {
            return null;
        }
        return this.getEffectiveSubject().getRealm().getDomain();
    }

    public boolean isAuthenticatedAsApiKey() {
        return this.authenticatingSubject.getType() == Subject.Type.API_KEY;
    }

    private boolean isAuthenticatedAnonymously() {
        return AuthenticationType.ANONYMOUS.equals((Object)this.getAuthenticationType());
    }

    private boolean isAuthenticatedInternally() {
        return AuthenticationType.INTERNAL.equals((Object)this.getAuthenticationType());
    }

    public boolean isServiceAccount() {
        return this.effectiveSubject.getType() == Subject.Type.SERVICE_ACCOUNT;
    }

    public boolean isApiKey() {
        return this.effectiveSubject.getType() == Subject.Type.API_KEY;
    }

    public boolean isCrossClusterAccess() {
        return this.effectiveSubject.getType() == Subject.Type.CROSS_CLUSTER_ACCESS;
    }

    public boolean supportsRunAs(@Nullable AnonymousUser anonymousUser) {
        if (this.isRunAs()) {
            return false;
        }
        if (this.isServiceAccount()) {
            return false;
        }
        if (this.isCrossClusterAccess()) {
            return false;
        }
        if (this.getEffectiveSubject().getUser() instanceof InternalUser) {
            return false;
        }
        if (this.getEffectiveSubject().getUser().equals(anonymousUser)) {
            assert ("__anonymous".equals(this.getAuthenticatingSubject().getRealm().getType()) && "__anonymous".equals(this.getAuthenticatingSubject().getRealm().getName()));
            return false;
        }
        return AuthenticationType.REALM == this.getAuthenticationType() || AuthenticationType.API_KEY == this.getAuthenticationType() || AuthenticationType.TOKEN == this.getAuthenticationType();
    }

    public void writeToContext(ThreadContext ctx) throws IOException, IllegalArgumentException {
        new AuthenticationContextSerializer().writeToContext(this, ctx);
    }

    public String encode() throws IOException {
        return Authentication.doEncode(this.effectiveSubject, this.authenticatingSubject, this.type);
    }

    static String doEncode(Subject effectiveSubject, Subject authenticatingSubject, AuthenticationType type) throws IOException {
        BytesStreamOutput output = new BytesStreamOutput();
        output.setTransportVersion(effectiveSubject.getTransportVersion());
        TransportVersion.writeVersion((TransportVersion)effectiveSubject.getTransportVersion(), (StreamOutput)output);
        Authentication.doWriteTo(effectiveSubject, authenticatingSubject, type, (StreamOutput)output);
        return Base64.getEncoder().encodeToString(BytesReference.toBytes((BytesReference)output.bytes()));
    }

    public void writeTo(StreamOutput out) throws IOException {
        Authentication.doWriteTo(this.effectiveSubject, this.authenticatingSubject, this.type, out);
    }

    private static void doWriteTo(Subject effectiveSubject, Subject authenticatingSubject, AuthenticationType type, StreamOutput out) throws IOException {
        RealmRef lookedUpBy;
        boolean isRunAs;
        boolean isCrossClusterAccess;
        boolean bl = isCrossClusterAccess = effectiveSubject.getType() == Subject.Type.CROSS_CLUSTER_ACCESS;
        if (isCrossClusterAccess && out.getTransportVersion().before((VersionId)RemoteClusterPortSettings.TRANSPORT_VERSION_ADVANCED_REMOTE_CLUSTER_SECURITY)) {
            throw new IllegalArgumentException("versions of Elasticsearch before [" + RemoteClusterPortSettings.TRANSPORT_VERSION_ADVANCED_REMOTE_CLUSTER_SECURITY + "] can't handle cross cluster access authentication and attempted to send to [" + out.getTransportVersion() + "]");
        }
        boolean bl2 = isRunAs = authenticatingSubject != effectiveSubject;
        if (isRunAs) {
            User outerUser = effectiveSubject.getUser();
            User innerUser = authenticatingSubject.getUser();
            assert (!(outerUser instanceof InternalUser) && !(innerUser instanceof InternalUser)) : "internal users cannot participate in run-as (outer=[" + outerUser + "] inner=[" + innerUser + "])";
            User.writeUser(outerUser, out);
            out.writeBoolean(true);
            User.writeUser(innerUser, out);
            out.writeBoolean(false);
        } else {
            User user = effectiveSubject.getUser();
            AuthenticationSerializationHelper.writeUserTo(user, out);
        }
        authenticatingSubject.getRealm().writeTo(out);
        RealmRef realmRef = lookedUpBy = isRunAs ? effectiveSubject.getRealm() : null;
        if (lookedUpBy != null) {
            out.writeBoolean(true);
            lookedUpBy.writeTo(out);
        } else {
            out.writeBoolean(false);
        }
        Map<String, Object> metadata = authenticatingSubject.getMetadata();
        if (out.getTransportVersion().onOrAfter((VersionId)VERSION_AUTHENTICATION_TYPE)) {
            out.writeVInt(type.ordinal());
            Authentication.writeMetadata(out, metadata);
        } else assert (type == AuthenticationType.REALM && metadata.isEmpty()) : Strings.format((String)"authentication with version [%s] must have authentication type %s and empty metadata, but got [%s] and [%s]", (Object[])new Object[]{out.getTransportVersion(), AuthenticationType.REALM, type, metadata});
    }

    public boolean canAccessResourcesOf(Authentication resourceCreatorAuthentication) {
        assert (EnumSet.of(AuthenticationType.REALM, AuthenticationType.API_KEY, AuthenticationType.TOKEN, AuthenticationType.ANONYMOUS, AuthenticationType.INTERNAL).containsAll(EnumSet.of(this.getAuthenticationType(), resourceCreatorAuthentication.getAuthenticationType()))) : "cross AuthenticationType comparison for canAccessResourcesOf is not applicable for: " + EnumSet.of(this.getAuthenticationType(), resourceCreatorAuthentication.getAuthenticationType());
        Subject mySubject = this.getEffectiveSubject();
        Subject creatorSubject = resourceCreatorAuthentication.getEffectiveSubject();
        return mySubject.canAccessResourcesOf(creatorSubject);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Authentication that = (Authentication)o;
        return this.type == that.type && this.authenticatingSubject.equals(that.authenticatingSubject) && this.effectiveSubject.equals(that.effectiveSubject);
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.type, this.authenticatingSubject, this.effectiveSubject});
    }

    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.startObject();
        this.toXContentFragment(builder);
        return builder.endObject();
    }

    public void toXContentFragment(XContentBuilder builder) throws IOException {
        RealmRef lookedUpBy;
        User user = this.effectiveSubject.getUser();
        builder.field(User.Fields.USERNAME.getPreferredName(), user.principal());
        builder.array(User.Fields.ROLES.getPreferredName(), user.roles());
        builder.field(User.Fields.FULL_NAME.getPreferredName(), user.fullName());
        builder.field(User.Fields.EMAIL.getPreferredName(), user.email());
        if (this.isServiceAccount()) {
            String tokenName = (String)this.getAuthenticatingSubject().getMetadata().get("_token_name");
            assert (tokenName != null) : "token name cannot be null";
            String tokenSource = (String)this.getAuthenticatingSubject().getMetadata().get("_token_source");
            assert (tokenSource != null) : "token source cannot be null";
            builder.field(User.Fields.TOKEN.getPreferredName(), Map.of("name", tokenName, "type", "_service_account_" + tokenSource));
        }
        builder.field(User.Fields.METADATA.getPreferredName(), user.metadata());
        builder.field(User.Fields.ENABLED.getPreferredName(), user.enabled());
        builder.startObject(User.Fields.AUTHENTICATION_REALM.getPreferredName());
        builder.field(User.Fields.REALM_NAME.getPreferredName(), this.getAuthenticatingSubject().getRealm().getName());
        builder.field(User.Fields.REALM_TYPE.getPreferredName(), this.getAuthenticatingSubject().getRealm().getType());
        if (this.getAuthenticatingSubject().getRealm().getDomain() != null) {
            builder.field(User.Fields.REALM_DOMAIN.getPreferredName(), this.getAuthenticatingSubject().getRealm().getDomain().name());
        }
        builder.endObject();
        builder.startObject(User.Fields.LOOKUP_REALM.getPreferredName());
        RealmRef realmRef = lookedUpBy = this.isRunAs() ? this.getEffectiveSubject().getRealm() : null;
        if (lookedUpBy != null) {
            builder.field(User.Fields.REALM_NAME.getPreferredName(), lookedUpBy.getName());
            builder.field(User.Fields.REALM_TYPE.getPreferredName(), lookedUpBy.getType());
            if (lookedUpBy.getDomain() != null) {
                builder.field(User.Fields.REALM_DOMAIN.getPreferredName(), lookedUpBy.getDomain().name());
            }
        } else {
            builder.field(User.Fields.REALM_NAME.getPreferredName(), this.getAuthenticatingSubject().getRealm().getName());
            builder.field(User.Fields.REALM_TYPE.getPreferredName(), this.getAuthenticatingSubject().getRealm().getType());
            if (this.getAuthenticatingSubject().getRealm().getDomain() != null) {
                builder.field(User.Fields.REALM_DOMAIN.getPreferredName(), this.getAuthenticatingSubject().getRealm().getDomain().name());
            }
        }
        builder.endObject();
        builder.field(User.Fields.AUTHENTICATION_TYPE.getPreferredName(), this.getAuthenticationType().name().toLowerCase(Locale.ROOT));
        if (this.isApiKey() || this.isCrossClusterAccess()) {
            String apiKeyId = (String)this.getAuthenticatingSubject().getMetadata().get("_security_api_key_id");
            String apiKeyName = (String)this.getAuthenticatingSubject().getMetadata().get("_security_api_key_name");
            if (apiKeyName == null) {
                builder.field("api_key", Map.of("id", apiKeyId));
            } else {
                builder.field("api_key", Map.of("id", apiKeyId, "name", apiKeyName));
            }
        }
    }

    public static Authentication getAuthenticationFromCrossClusterAccessMetadata(Authentication authentication) {
        if (authentication.isCrossClusterAccess()) {
            return (Authentication)authentication.getAuthenticatingSubject().getMetadata().get("_security_cross_cluster_access_authentication");
        }
        String message = "authentication is not cross_cluster_access";
        assert (false) : message;
        throw new IllegalArgumentException(message);
    }

    private static Map<String, Object> readMetadata(StreamInput in) throws IOException {
        if (in.getTransportVersion().onOrAfter((VersionId)VERSION_METADATA_BEYOND_GENERIC_MAP)) {
            int size = in.readVInt();
            Map metadata = Maps.newHashMapWithExpectedSize((int)size);
            for (int i = 0; i < size; ++i) {
                String key = in.readString();
                Object value = METADATA_VALUE_READER.getOrDefault(key, (CheckedFunction<StreamInput, Object, IOException>)((CheckedFunction)StreamInput::readGenericValue)).apply((Object)in);
                metadata.put(key, value);
            }
            return metadata;
        }
        return in.readMap();
    }

    private static void writeMetadata(StreamOutput out, Map<String, Object> metadata) throws IOException {
        if (out.getTransportVersion().onOrAfter((VersionId)VERSION_METADATA_BEYOND_GENERIC_MAP)) {
            out.writeVInt(metadata.size());
            for (Map.Entry<String, Object> entry : metadata.entrySet()) {
                out.writeString(entry.getKey());
                Writeable.Writer valueWriter = METADATA_VALUE_WRITER.getOrDefault(entry.getKey(), StreamOutput::writeGenericValue);
                valueWriter.write(out, entry.getValue());
            }
        } else {
            out.writeGenericMap(metadata);
        }
    }

    public void checkConsistency() {
        if (this.isRunAs() ? !$assertionsDisabled && this.authenticatingSubject == this.effectiveSubject : !$assertionsDisabled && this.authenticatingSubject != this.effectiveSubject) {
            throw new AssertionError((Object)"isRunAs logic does not hold");
        }
        switch (this.getAuthenticationType()) {
            case ANONYMOUS: {
                this.checkConsistencyForAnonymousAuthenticationType();
                break;
            }
            case INTERNAL: {
                this.checkConsistencyForInternalAuthenticationType();
                break;
            }
            case API_KEY: {
                this.checkConsistencyForApiKeyAuthenticationType();
                break;
            }
            case REALM: {
                this.checkConsistencyForRealmAuthenticationType();
                break;
            }
            case TOKEN: {
                this.checkConsistencyForTokenAuthenticationType();
                break;
            }
            default: {
                assert (false) : "unknown authentication type " + this.type;
                break;
            }
        }
    }

    private void checkConsistencyForAnonymousAuthenticationType() {
        RealmRef authenticatingRealm = this.authenticatingSubject.getRealm();
        if (!authenticatingRealm.isAnonymousRealm()) {
            throw new IllegalArgumentException(Strings.format((String)"Anonymous authentication cannot have realm type [%s]", (Object[])new Object[]{authenticatingRealm.type}));
        }
        Authentication.checkNoDomain(authenticatingRealm, "Anonymous");
        Authentication.checkNoInternalUser(this.authenticatingSubject, "Anonymous");
        Authentication.checkNoRunAs(this, "Anonymous");
    }

    private void checkConsistencyForInternalAuthenticationType() {
        RealmRef authenticatingRealm = this.authenticatingSubject.getRealm();
        if (!authenticatingRealm.isFallbackRealm() && !authenticatingRealm.isAttachRealm()) {
            throw new IllegalArgumentException(Strings.format((String)"Internal authentication cannot have realm type [%s]", (Object[])new Object[]{authenticatingRealm.type}));
        }
        Authentication.checkNoDomain(authenticatingRealm, "Internal");
        if (!(this.authenticatingSubject.getUser() instanceof InternalUser)) {
            throw new IllegalArgumentException("Internal authentication must have internal user");
        }
        Authentication.checkNoRunAs(this, "Internal");
    }

    private void checkConsistencyForApiKeyAuthenticationType() {
        RealmRef authenticatingRealm = this.authenticatingSubject.getRealm();
        if (!authenticatingRealm.isApiKeyRealm() && !authenticatingRealm.isCrossClusterAccessRealm()) {
            throw new IllegalArgumentException(Strings.format((String)"API key authentication cannot have realm type [%s]", (Object[])new Object[]{authenticatingRealm.type}));
        }
        this.checkConsistencyForApiKeyAuthenticatingSubject("API key");
        if (Subject.Type.CROSS_CLUSTER_ACCESS == this.authenticatingSubject.getType()) {
            if (this.authenticatingSubject.getMetadata().get("_security_cross_cluster_access_authentication") == null) {
                throw new IllegalArgumentException("Cross cluster access authentication requires metadata to contain a non-null serialized cross cluster access authentication field");
            }
            Authentication innerAuthentication = (Authentication)this.authenticatingSubject.getMetadata().get("_security_cross_cluster_access_authentication");
            if (innerAuthentication.isCrossClusterAccess()) {
                throw new IllegalArgumentException("Cross cluster access authentication cannot contain another cross cluster access authentication in its metadata");
            }
            if (this.authenticatingSubject.getMetadata().get("_security_cross_cluster_access_role_descriptors") == null) {
                throw new IllegalArgumentException("Cross cluster access authentication requires metadata to contain a non-null serialized cross cluster access role descriptors field");
            }
            Authentication.checkNoRunAs(this, "Cross cluster access");
        } else if (this.isRunAs()) {
            Authentication.checkRunAsConsistency(this.effectiveSubject, this.authenticatingSubject);
        }
    }

    private void checkConsistencyForRealmAuthenticationType() {
        if (Subject.Type.USER != this.authenticatingSubject.getType()) {
            throw new IllegalArgumentException("Realm authentication must have subject type of user");
        }
        if (this.isRunAs()) {
            Authentication.checkRunAsConsistency(this.effectiveSubject, this.authenticatingSubject);
        }
    }

    private void checkConsistencyForTokenAuthenticationType() {
        RealmRef authenticatingRealm = this.authenticatingSubject.getRealm();
        assert (!(authenticatingRealm.isAttachRealm() || authenticatingRealm.isFallbackRealm() || authenticatingRealm.isCrossClusterAccessRealm())) : "Token authentication cannot have authenticating realm " + authenticatingRealm;
        Authentication.checkNoInternalUser(this.authenticatingSubject, "Token");
        if (Subject.Type.SERVICE_ACCOUNT == this.authenticatingSubject.getType()) {
            Authentication.checkNoDomain(authenticatingRealm, "Service account");
            Authentication.checkNoRole(this.authenticatingSubject, "Service account");
            Authentication.checkNoRunAs(this, "Service account");
        } else {
            if (Subject.Type.API_KEY == this.authenticatingSubject.getType()) {
                this.checkConsistencyForApiKeyAuthenticatingSubject("API key token");
            }
            if (this.isRunAs()) {
                Authentication.checkRunAsConsistency(this.effectiveSubject, this.authenticatingSubject);
            }
        }
    }

    private static void checkRunAsConsistency(Subject effectiveSubject, Subject authenticatingSubject) {
        if (!effectiveSubject.getTransportVersion().equals((Object)authenticatingSubject.getTransportVersion())) {
            throw new IllegalArgumentException(Strings.format((String)"inconsistent versions between effective subject [%s] and authenticating subject [%s]", (Object[])new Object[]{effectiveSubject.getTransportVersion(), authenticatingSubject.getTransportVersion()}));
        }
        if (Subject.Type.USER != effectiveSubject.getType()) {
            throw new IllegalArgumentException(Strings.format((String)"Run-as subject type cannot be [%s]", (Object[])new Object[]{effectiveSubject.getType()}));
        }
        if (!effectiveSubject.getMetadata().isEmpty()) {
            throw new IllegalArgumentException("Run-as subject must have empty metadata");
        }
        assert (!Authentication.hasSyntheticRealmNameOrType(effectiveSubject.getRealm())) : "run-as subject cannot be from a synthetic realm";
    }

    private void checkConsistencyForApiKeyAuthenticatingSubject(String prefixMessage) {
        RealmRef authenticatingRealm = this.authenticatingSubject.getRealm();
        Authentication.checkNoDomain(authenticatingRealm, prefixMessage);
        Authentication.checkNoInternalUser(this.authenticatingSubject, prefixMessage);
        Authentication.checkNoRole(this.authenticatingSubject, prefixMessage);
        if (this.authenticatingSubject.getMetadata().get("_security_api_key_id") == null) {
            throw new IllegalArgumentException(prefixMessage + " authentication requires metadata to contain a non-null API key ID");
        }
    }

    private static void checkNoInternalUser(Subject subject, String prefixMessage) {
        if (subject.getUser() instanceof InternalUser) {
            throw new IllegalArgumentException(Strings.format((String)(prefixMessage + " authentication cannot have internal user [%s]"), (Object[])new Object[]{subject.getUser().principal()}));
        }
    }

    private static void checkNoDomain(RealmRef realm, String prefixMessage) {
        if (realm.getDomain() != null) {
            throw new IllegalArgumentException(prefixMessage + " authentication cannot have domain");
        }
    }

    private static void checkNoRole(Subject subject, String prefixMessage) {
        if (subject.getUser().roles().length != 0) {
            throw new IllegalArgumentException(prefixMessage + " authentication user must have no role");
        }
    }

    private static void checkNoRunAs(Authentication authentication, String prefixMessage) {
        if (authentication.isRunAs()) {
            throw new IllegalArgumentException(prefixMessage + " authentication cannot run-as other user");
        }
    }

    private static boolean hasSyntheticRealmNameOrType(@Nullable RealmRef realmRef) {
        if (realmRef == null) {
            return false;
        }
        if (List.of("_es_api_key", "_service_account", "__anonymous", "__fallback", "__attach", "_es_cross_cluster_access").contains(realmRef.getName())) {
            return true;
        }
        return List.of("_es_api_key", "_service_account", "__anonymous", "__fallback", "__attach", "_es_cross_cluster_access").contains(realmRef.getType());
    }

    public String toString() {
        StringBuilder builder = new StringBuilder("Authentication[effectiveSubject=").append(this.effectiveSubject);
        if (this.isRunAs()) {
            builder.append(",authenticatingSubject=").append(this.authenticatingSubject);
        }
        builder.append(",type=").append((Object)this.type);
        builder.append("]");
        return builder.toString();
    }

    public static boolean isFileOrNativeRealm(String realmType) {
        return "file".equals(realmType) || "native".equals(realmType);
    }

    public static Authentication newInternalAuthentication(InternalUser internalUser, TransportVersion version, String nodeName) {
        RealmRef authenticatedBy = RealmRef.newInternalAttachRealmRef(nodeName);
        Authentication authentication = new Authentication(new Subject(internalUser, authenticatedBy, version, Map.of()), AuthenticationType.INTERNAL);
        return authentication;
    }

    public static Authentication newInternalFallbackAuthentication(User fallbackUser, String nodeName) {
        RealmRef authenticatedBy = RealmRef.newInternalFallbackRealmRef(nodeName);
        Authentication authentication = new Authentication(new Subject(fallbackUser, authenticatedBy, TransportVersion.current(), Map.of()), AuthenticationType.INTERNAL);
        return authentication;
    }

    public static Authentication newAnonymousAuthentication(AnonymousUser anonymousUser, String nodeName) {
        RealmRef authenticatedBy = RealmRef.newAnonymousRealmRef(nodeName);
        Authentication authentication = new Authentication(new Subject(anonymousUser, authenticatedBy, TransportVersion.current(), Map.of()), AuthenticationType.ANONYMOUS);
        return authentication;
    }

    public static Authentication newServiceAccountAuthentication(User serviceAccountUser, String nodeName, Map<String, Object> metadata) {
        RealmRef authenticatedBy = RealmRef.newServiceAccountRealmRef(nodeName);
        Authentication authentication = new Authentication(new Subject(serviceAccountUser, authenticatedBy, TransportVersion.current(), metadata), AuthenticationType.TOKEN);
        return authentication;
    }

    public static Authentication newRealmAuthentication(User user, RealmRef realmRef) {
        Authentication authentication = new Authentication(new Subject(user, realmRef, TransportVersion.current(), Map.of()), AuthenticationType.REALM);
        assert (!authentication.isServiceAccount());
        assert (!authentication.isApiKey());
        assert (!authentication.isCrossClusterAccess());
        assert (!authentication.isAuthenticatedInternally());
        assert (!authentication.isAuthenticatedAnonymously());
        return authentication;
    }

    public static Authentication newApiKeyAuthentication(AuthenticationResult<User> authResult, String nodeName) {
        assert (authResult.isAuthenticated()) : "API Key authn result must be successful";
        User apiKeyUser = authResult.getValue();
        assert (apiKeyUser.roles().length == 0) : "The user associated to an API key authentication must have no role";
        RealmRef authenticatedBy = RealmRef.newApiKeyRealmRef(nodeName);
        Authentication authentication = new Authentication(new Subject(apiKeyUser, authenticatedBy, TransportVersion.current(), authResult.getMetadata()), AuthenticationType.API_KEY);
        return authentication;
    }

    public Authentication toCrossClusterAccess(CrossClusterAccessSubjectInfo crossClusterAccessSubjectInfo) {
        assert (this.isApiKey()) : "can only convert API key authentication to cross cluster access";
        assert (!this.isRunAs()) : "cross cluster access does not support authentication with run-as";
        assert (this.getEffectiveSubject().getUser().roles().length == 0) : "the user associated with a cross cluster access authentication must have no role";
        HashMap<String, Object> metadata = new HashMap<String, Object>(this.getAuthenticatingSubject().getMetadata());
        RealmRef authenticatedBy = RealmRef.newCrossClusterAccessRealmRef(this.getAuthenticatingSubject().getRealm().getNodeName());
        return new Authentication(new Subject(this.getEffectiveSubject().getUser(), authenticatedBy, TransportVersion.current(), crossClusterAccessSubjectInfo.copyWithCrossClusterAccessEntries(metadata)), this.getAuthenticationType());
    }

    static RealmRef maybeRewriteRealmRef(TransportVersion streamVersion, RealmRef realmRef) {
        if (realmRef != null && realmRef.getDomain() != null && streamVersion.before((VersionId)VERSION_REALM_DOMAINS)) {
            logger.info("Rewriting realm [" + realmRef + "] without domain");
            return new RealmRef(realmRef.getName(), realmRef.getType(), realmRef.getNodeName(), null);
        }
        return realmRef;
    }

    private static Map<String, Object> maybeRewriteMetadataForApiKeyRoleDescriptors(TransportVersion streamVersion, Authentication authentication) {
        Map<String, Object> metadata = authentication.getAuthenticatingSubject().getMetadata();
        if (authentication.isAuthenticatedAsApiKey()) {
            assert (metadata.containsKey("_security_api_key_role_descriptors")) : "metadata must contain role descriptor for API key authentication";
            assert (metadata.containsKey("_security_api_key_limited_by_role_descriptors")) : "metadata must contain limited role descriptor for API key authentication";
            if (authentication.getEffectiveSubject().getTransportVersion().onOrAfter((VersionId)RemoteClusterPortSettings.TRANSPORT_VERSION_ADVANCED_REMOTE_CLUSTER_SECURITY) && streamVersion.before((VersionId)RemoteClusterPortSettings.TRANSPORT_VERSION_ADVANCED_REMOTE_CLUSTER_SECURITY)) {
                metadata = new HashMap<String, Object>(metadata);
                metadata.put("_security_api_key_role_descriptors", Authentication.maybeRemoveRemoteIndicesFromRoleDescriptors((BytesReference)metadata.get("_security_api_key_role_descriptors")));
                metadata.put("_security_api_key_limited_by_role_descriptors", Authentication.maybeRemoveRemoteIndicesFromRoleDescriptors((BytesReference)metadata.get("_security_api_key_limited_by_role_descriptors")));
            }
            if (authentication.getEffectiveSubject().getTransportVersion().onOrAfter((VersionId)VERSION_API_KEY_ROLES_AS_BYTES) && streamVersion.before((VersionId)VERSION_API_KEY_ROLES_AS_BYTES)) {
                metadata = new HashMap<String, Object>(metadata);
                metadata.put("_security_api_key_role_descriptors", Authentication.convertRoleDescriptorsBytesToMap((BytesReference)metadata.get("_security_api_key_role_descriptors")));
                metadata.put("_security_api_key_limited_by_role_descriptors", Authentication.convertRoleDescriptorsBytesToMap((BytesReference)metadata.get("_security_api_key_limited_by_role_descriptors")));
            } else if (authentication.getEffectiveSubject().getTransportVersion().before((VersionId)VERSION_API_KEY_ROLES_AS_BYTES) && streamVersion.onOrAfter((VersionId)VERSION_API_KEY_ROLES_AS_BYTES)) {
                metadata = new HashMap<String, Object>(metadata);
                metadata.put("_security_api_key_role_descriptors", Authentication.convertRoleDescriptorsMapToBytes((Map)metadata.get("_security_api_key_role_descriptors")));
                metadata.put("_security_api_key_limited_by_role_descriptors", Authentication.convertRoleDescriptorsMapToBytes((Map)metadata.get("_security_api_key_limited_by_role_descriptors")));
            }
        }
        return metadata;
    }

    static Map<String, Object> maybeRewriteMetadataForCrossClusterAccessAuthentication(TransportVersion olderVersion, Authentication authentication) {
        assert (authentication.isCrossClusterAccess()) : "authentication must be cross cluster access";
        Map<String, Object> metadata = authentication.getAuthenticatingSubject().getMetadata();
        assert (metadata.containsKey("_security_cross_cluster_access_authentication")) : "metadata must contain authentication object for cross cluster access authentication";
        Authentication authenticationFromMetadata = (Authentication)metadata.get("_security_cross_cluster_access_authentication");
        TransportVersion effectiveSubjectVersion = authenticationFromMetadata.getEffectiveSubject().getTransportVersion();
        if (effectiveSubjectVersion.after((VersionId)olderVersion)) {
            logger.trace(() -> "Cross cluster access authentication has authentication field in metadata [" + authenticationFromMetadata + "] that may require a rewrite from version [" + effectiveSubjectVersion + "] to [" + olderVersion + "]");
            HashMap<String, Object> rewrittenMetadata = new HashMap<String, Object>(metadata);
            rewrittenMetadata.put("_security_cross_cluster_access_authentication", authenticationFromMetadata.maybeRewriteForOlderVersion(olderVersion));
            return rewrittenMetadata;
        }
        return metadata;
    }

    private static Map<String, Object> convertRoleDescriptorsBytesToMap(BytesReference roleDescriptorsBytes) {
        return (Map)XContentHelper.convertToMap((BytesReference)roleDescriptorsBytes, (boolean)false, (XContentType)XContentType.JSON).v2();
    }

    private static BytesReference convertRoleDescriptorsMapToBytes(Map<String, Object> roleDescriptorsMap) {
        BytesReference bytesReference;
        block8: {
            XContentBuilder builder = XContentBuilder.builder((XContent)XContentType.JSON.xContent());
            try {
                builder.map(roleDescriptorsMap);
                bytesReference = BytesReference.bytes((XContentBuilder)builder);
                if (builder == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (builder != null) {
                        try {
                            builder.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            builder.close();
        }
        return bytesReference;
    }

    static BytesReference maybeRemoveRemoteIndicesFromRoleDescriptors(BytesReference roleDescriptorsBytes) {
        if (roleDescriptorsBytes == null || roleDescriptorsBytes.length() == 0) {
            return roleDescriptorsBytes;
        }
        Map<String, Object> roleDescriptorsMap = Authentication.convertRoleDescriptorsBytesToMap(roleDescriptorsBytes);
        AtomicBoolean removedAtLeastOne = new AtomicBoolean(false);
        roleDescriptorsMap.forEach((key, value) -> {
            if (value instanceof Map) {
                boolean removed;
                Map roleDescriptor = (Map)value;
                boolean bl = removed = roleDescriptor.remove(RoleDescriptor.Fields.REMOTE_INDICES.getPreferredName()) != null;
                if (removed) {
                    removedAtLeastOne.set(true);
                }
            }
        });
        if (removedAtLeastOne.get()) {
            return Authentication.convertRoleDescriptorsMapToBytes(roleDescriptorsMap);
        }
        return roleDescriptorsBytes;
    }

    static boolean equivalentRealms(String name1, String type1, String name2, String type2) {
        if (!type1.equals(type2)) {
            return false;
        }
        if (Authentication.isFileOrNativeRealm(type1)) {
            return true;
        }
        return name1.equals(name2);
    }

    static {
        REALM_REF_PARSER.declareString(ConstructingObjectParser.constructorArg(), new ParseField("name", new String[0]));
        REALM_REF_PARSER.declareString(ConstructingObjectParser.constructorArg(), new ParseField("type", new String[0]));
        REALM_REF_PARSER.declareString(ConstructingObjectParser.constructorArg(), new ParseField("node_name", new String[0]));
        REALM_REF_PARSER.declareObject(ConstructingObjectParser.optionalConstructorArg(), (p, c) -> (RealmDomain)RealmDomain.REALM_DOMAIN_PARSER.parse(p, c), new ParseField("domain", new String[0]));
    }

    public static enum AuthenticationType {
        REALM,
        API_KEY,
        TOKEN,
        ANONYMOUS,
        INTERNAL;

    }

    public static class AuthenticationSerializationHelper {
        private AuthenticationSerializationHelper() {
        }

        public static User readUserFrom(StreamInput input) throws IOException {
            User user = AuthenticationSerializationHelper.readUserWithoutTrailingBoolean(input);
            if (!(user instanceof InternalUser)) {
                boolean hasInnerUser = input.readBoolean();
                assert (!hasInnerUser) : "inner user is not allowed";
                if (hasInnerUser) {
                    throw new IllegalStateException("inner user is not allowed");
                }
            }
            return user;
        }

        public static void writeUserTo(User user, StreamOutput output) throws IOException {
            if (user instanceof InternalUser) {
                InternalUser internal = (InternalUser)user;
                AuthenticationSerializationHelper.writeInternalUser(internal, output);
            } else {
                User.writeUser(user, output);
                output.writeBoolean(false);
            }
        }

        private static User readUserWithoutTrailingBoolean(StreamInput input) throws IOException {
            boolean isInternalUser = input.readBoolean();
            String username = input.readString();
            if (isInternalUser) {
                return InternalUsers.getUser(username);
            }
            String[] roles = input.readStringArray();
            Map metadata = input.readMap();
            String fullName = input.readOptionalString();
            String email = input.readOptionalString();
            boolean enabled = input.readBoolean();
            return new User(username, roles, fullName, email, metadata, enabled);
        }

        private static void writeInternalUser(InternalUser user, StreamOutput output) throws IOException {
            output.writeBoolean(true);
            output.writeString(user.principal());
        }
    }

    public static class RealmRef
    implements Writeable,
    ToXContentObject {
        private final String nodeName;
        private final String name;
        private final String type;
        @Nullable
        private final RealmDomain domain;

        public RealmRef(String name, String type, String nodeName) {
            this(name, type, nodeName, null);
        }

        public RealmRef(String name, String type, String nodeName, @Nullable RealmDomain domain) {
            this.nodeName = Objects.requireNonNull(nodeName, "node name cannot be null");
            this.name = Objects.requireNonNull(name, "realm name cannot be null");
            this.type = Objects.requireNonNull(type, "realm type cannot be null");
            this.domain = domain;
        }

        public RealmRef(StreamInput in) throws IOException {
            this.nodeName = in.readString();
            this.name = in.readString();
            this.type = in.readString();
            this.domain = in.getTransportVersion().onOrAfter((VersionId)VERSION_REALM_DOMAINS) ? (RealmDomain)in.readOptionalWriteable(RealmDomain::readFrom) : null;
        }

        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.nodeName);
            out.writeString(this.name);
            out.writeString(this.type);
            if (out.getTransportVersion().onOrAfter((VersionId)VERSION_REALM_DOMAINS)) {
                out.writeOptionalWriteable((Writeable)this.domain);
            }
        }

        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject();
            builder.field("name", this.name);
            builder.field("type", this.type);
            builder.field("node_name", this.nodeName);
            if (this.domain != null) {
                builder.field("domain", (ToXContent)this.domain);
            }
            builder.endObject();
            return builder;
        }

        public String getNodeName() {
            return this.nodeName;
        }

        public String getName() {
            return this.name;
        }

        public String getType() {
            return this.type;
        }

        @Nullable
        public RealmDomain getDomain() {
            return this.domain;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RealmRef realmRef = (RealmRef)o;
            if (!this.nodeName.equals(realmRef.nodeName)) {
                return false;
            }
            if (!this.type.equals(realmRef.type)) {
                return false;
            }
            return Objects.equals(this.domain, realmRef.domain);
        }

        public int hashCode() {
            int result = this.nodeName.hashCode();
            result = 31 * result + this.name.hashCode();
            result = 31 * result + this.type.hashCode();
            if (this.domain != null) {
                result = 31 * result + this.domain.hashCode();
            }
            return result;
        }

        public String toString() {
            if (this.domain != null) {
                return "{Realm[" + this.type + "." + this.name + "] under Domain[" + this.domain.name() + "] on Node[" + this.nodeName + "]}";
            }
            return "{Realm[" + this.type + "." + this.name + "] on Node[" + this.nodeName + "]}";
        }

        private boolean isFallbackRealm() {
            return "__fallback".equals(this.name) && "__fallback".equals(this.type);
        }

        private boolean isAttachRealm() {
            return "__attach".equals(this.name) && "__attach".equals(this.type);
        }

        private boolean isAnonymousRealm() {
            return "__anonymous".equals(this.name) && "__anonymous".equals(this.type);
        }

        private boolean isApiKeyRealm() {
            return "_es_api_key".equals(this.name) && "_es_api_key".equals(this.type);
        }

        private boolean isServiceAccountRealm() {
            return "_service_account".equals(this.name) && "_service_account".equals(this.type);
        }

        private boolean isCrossClusterAccessRealm() {
            return "_es_cross_cluster_access".equals(this.name) && "_es_cross_cluster_access".equals(this.type);
        }

        static RealmRef newInternalAttachRealmRef(String nodeName) {
            return new RealmRef("__attach", "__attach", nodeName, null);
        }

        static RealmRef newInternalFallbackRealmRef(String nodeName) {
            RealmRef realmRef = new RealmRef("__fallback", "__fallback", nodeName, null);
            return realmRef;
        }

        public static RealmRef newAnonymousRealmRef(String nodeName) {
            return new RealmRef("__anonymous", "__anonymous", nodeName, null);
        }

        static RealmRef newServiceAccountRealmRef(String nodeName) {
            return new RealmRef("_service_account", "_service_account", nodeName, null);
        }

        static RealmRef newApiKeyRealmRef(String nodeName) {
            return new RealmRef("_es_api_key", "_es_api_key", nodeName, null);
        }

        static RealmRef newCrossClusterAccessRealmRef(String nodeName) {
            return new RealmRef("_es_cross_cluster_access", "_es_cross_cluster_access", nodeName, null);
        }
    }
}

