/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.license;

import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.VersionId;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.license.License;
import org.elasticsearch.license.LicenseOverrides;
import org.elasticsearch.license.LicenseSettings;
import org.elasticsearch.license.LicensesMetadata;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.license.internal.XPackLicenseStatus;
import org.elasticsearch.protocol.xpack.license.LicenseStatus;
import org.elasticsearch.rest.RestStatus;

public class LicenseUtils {
    public static final String EXPIRED_FEATURE_METADATA = "es.license.expired.feature";
    public static final DateFormatter DATE_FORMATTER = DateFormatter.forPattern((String)"EEEE, MMMM dd, yyyy");

    public static ElasticsearchSecurityException newComplianceException(String feature) {
        ElasticsearchSecurityException e = new ElasticsearchSecurityException("current license is non-compliant for [{}]", RestStatus.FORBIDDEN, new Object[]{feature});
        e.addMetadata(EXPIRED_FEATURE_METADATA, new String[]{feature});
        return e;
    }

    public static boolean isLicenseExpiredException(ElasticsearchSecurityException exception) {
        return exception != null && exception.getMetadata(EXPIRED_FEATURE_METADATA) != null;
    }

    public static boolean licenseNeedsExtended(License license) {
        return License.LicenseType.isBasic(license.type()) && LicenseUtils.getExpiryDate(license) != LicenseSettings.BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS;
    }

    public static boolean signatureNeedsUpdate(License license, DiscoveryNodes currentNodes) {
        String typeName = license.type();
        return (License.LicenseType.isBasic(typeName) || License.LicenseType.isTrial(typeName)) && license.version() < 4 && LicenseUtils.compatibleLicenseVersion(currentNodes) >= 4;
    }

    public static int compatibleLicenseVersion(DiscoveryNodes currentNodes) {
        return LicenseUtils.getMaxLicenseVersion(currentNodes.getMinNodeVersion());
    }

    public static int getMaxLicenseVersion(Version version) {
        if (version != null && version.before((VersionId)Version.V_7_6_0)) {
            return 4;
        }
        return 5;
    }

    public static long getExpiryDate(License license) {
        String licenseUidHash = MessageDigests.toHexString((byte[])MessageDigests.sha256().digest(license.uid().getBytes(StandardCharsets.UTF_8)));
        return LicenseOverrides.overrideDateForLicense(licenseUidHash).map(date -> date.toInstant().toEpochMilli()).orElse(license.expiryDate());
    }

    public static LicenseStatus status(License license) {
        long now = System.currentTimeMillis();
        if (license.issueDate() > now) {
            return LicenseStatus.INVALID;
        }
        if (LicenseUtils.getExpiryDate(license) < now) {
            return LicenseStatus.EXPIRED;
        }
        return LicenseStatus.ACTIVE;
    }

    public static XPackLicenseStatus getXPackLicenseStatus(License license, Clock clock) {
        long now = clock.millis();
        Objects.requireNonNull(license, "license must not be null");
        Objects.requireNonNull(clock, "clock must not be null");
        if (license == LicensesMetadata.LICENSE_TOMBSTONE) {
            return new XPackLicenseStatus(License.OperationMode.MISSING, false, LicenseUtils.getExpiryWarning(LicenseUtils.getExpiryDate(license), now));
        }
        boolean active = LicenseUtils.getExpiryDate(license) == LicenseSettings.BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS ? true : now >= license.issueDate() && now < LicenseUtils.getExpiryDate(license);
        return new XPackLicenseStatus(license.operationMode(), active, LicenseUtils.getExpiryWarning(LicenseUtils.getExpiryDate(license), now));
    }

    public static Map<String, String[]> getAckMessages(License newLicense, License currentLicense) {
        HashMap<String, String[]> acknowledgeMessages = new HashMap<String, String[]>();
        if (!License.isAutoGeneratedLicense(currentLicense.signature()) && currentLicense.issueDate() > newLicense.issueDate()) {
            acknowledgeMessages.put("license", new String[]{"The new license is older than the currently installed license. Are you sure you want to override the current license?"});
        }
        XPackLicenseState.ACKNOWLEDGMENT_MESSAGES.forEach((feature, ackMessages) -> {
            String[] messages = (String[])ackMessages.apply(currentLicense.operationMode(), newLicense.operationMode());
            if (messages.length > 0) {
                acknowledgeMessages.put((String)feature, messages);
            }
        });
        return acknowledgeMessages;
    }

    public static String getExpiryWarning(long licenseExpiryDate, long currentTime) {
        long diff = licenseExpiryDate - currentTime;
        if (LicenseSettings.LICENSE_EXPIRATION_WARNING_PERIOD.getMillis() > diff) {
            long days = TimeUnit.MILLISECONDS.toDays(diff);
            String expiryMessage = days == 0L && diff > 0L ? "expires today" : (diff > 0L ? String.format(Locale.ROOT, "will expire in [%d] days", days) : String.format(Locale.ROOT, "expired on [%s]", DATE_FORMATTER.formatMillis(licenseExpiryDate)));
            return "Your license " + expiryMessage + ". Contact your administrator or update your license for continued use of features";
        }
        return null;
    }
}

