/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.searchablesnapshots.cache.blob;

import java.time.Instant;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.IndexFileNames;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.action.support.TransportActions;
import org.elasticsearch.blobcache.common.ByteRange;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.VersionId;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.LuceneFilesExtensions;
import org.elasticsearch.node.NodeClosedException;
import org.elasticsearch.repositories.IndexId;
import org.elasticsearch.snapshots.SnapshotId;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;
import org.elasticsearch.xpack.searchablesnapshots.cache.blob.CachedBlob;

public class BlobStoreCacheService
extends AbstractLifecycleComponent {
    private static final Logger logger = LogManager.getLogger(BlobStoreCacheService.class);
    private static final Version OLD_CACHED_BLOB_SIZE_VERSION = Version.V_7_12_0;
    public static final int DEFAULT_CACHED_BLOB_SIZE = ByteSizeUnit.KB.toIntBytes(1L);
    private static final Cache<String, String> LOG_EXCEEDING_FILES_CACHE = CacheBuilder.builder().setExpireAfterAccess(TimeValue.timeValueMinutes((long)60L)).build();
    static final int MAX_IN_FLIGHT_CACHE_FILLS = Integer.MAX_VALUE;
    private final ClusterService clusterService;
    private final Semaphore inFlightCacheFills;
    private final AtomicBoolean closed;
    private final Client client;
    private final String index;

    public BlobStoreCacheService(ClusterService clusterService, Client client, String index) {
        this.client = new OriginSettingClient(client, "searchable_snapshots");
        this.inFlightCacheFills = new Semaphore(Integer.MAX_VALUE);
        this.closed = new AtomicBoolean(false);
        this.clusterService = clusterService;
        this.index = index;
    }

    protected void doStart() {
    }

    protected void doStop() {
        if (this.closed.compareAndSet(false, true)) {
            logger.debug("blob cache service is stopped");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean waitForInFlightCacheFillsToComplete(long timeout, TimeUnit unit) {
        boolean acquired = false;
        try {
            logger.debug("waiting for in-flight blob cache fills to complete");
            acquired = this.inFlightCacheFills.tryAcquire(Integer.MAX_VALUE, timeout, unit);
        }
        catch (InterruptedException e) {
            logger.warn("interrupted while waiting for in-flight blob cache fills to complete", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        finally {
            if (acquired) {
                this.inFlightCacheFills.release(Integer.MAX_VALUE);
            }
        }
        return acquired;
    }

    int getInFlightCacheFills() {
        return Integer.MAX_VALUE - this.inFlightCacheFills.availablePermits();
    }

    protected void doClose() {
    }

    public CachedBlob get(String repository, SnapshotId snapshotId, IndexId indexId, ShardId shardId, String name, ByteRange range) {
        assert (!Thread.currentThread().getName().contains("[system_read]")) : "must not block [" + Thread.currentThread().getName() + "] for a cache read";
        PlainActionFuture future = PlainActionFuture.newFuture();
        this.getAsync(repository, snapshotId, indexId, shardId, name, range, (ActionListener<CachedBlob>)future);
        try {
            return (CachedBlob)future.actionGet(5L, TimeUnit.SECONDS);
        }
        catch (ElasticsearchTimeoutException e) {
            if (logger.isDebugEnabled()) {
                logger.debug(() -> Strings.format((String)"get from cache index timed out after [5s], retrieving from blob store instead [id=%s]", (Object[])new Object[]{BlobStoreCacheService.generateId(repository, snapshotId, indexId, shardId, name, range)}), (Throwable)e);
            } else {
                logger.warn("get from cache index timed out after [5s], retrieving from blob store instead");
            }
            return CachedBlob.CACHE_NOT_READY;
        }
    }

    final void getAsync(final String repository, final SnapshotId snapshotId, final IndexId indexId, final ShardId shardId, final String name, final ByteRange range, final ActionListener<CachedBlob> listener) {
        if (this.closed.get()) {
            logger.debug("failed to retrieve cached blob from system index [{}], service is closed", (Object)this.index);
            listener.onResponse((Object)CachedBlob.CACHE_NOT_READY);
            return;
        }
        final GetRequest request = new GetRequest(this.index).id(BlobStoreCacheService.generateId(repository, snapshotId, indexId, shardId, name, range));
        this.innerGet(request, new ActionListener<GetResponse>(){

            public void onResponse(GetResponse response) {
                if (response.isExists()) {
                    logger.debug("cache hit : [{}]", (Object)request.id());
                    assert (!response.isSourceEmpty());
                    CachedBlob cachedBlob = CachedBlob.fromSource(response.getSource());
                    assert (BlobStoreCacheService.assertDocId(response, repository, snapshotId, indexId, shardId, name, range));
                    if (cachedBlob.from() != range.start() || cachedBlob.to() != range.end()) {
                        listener.onResponse((Object)CachedBlob.CACHE_MISS);
                    } else {
                        listener.onResponse((Object)cachedBlob);
                    }
                } else {
                    logger.debug("cache miss: [{}]", (Object)request.id());
                    listener.onResponse((Object)CachedBlob.CACHE_MISS);
                }
            }

            public void onFailure(Exception e) {
                if (BlobStoreCacheService.isExpectedCacheGetException(e)) {
                    logger.debug(() -> "failed to retrieve cached blob from system index [" + BlobStoreCacheService.this.index + "]", (Throwable)e);
                } else {
                    logger.warn(() -> "failed to retrieve cached blob from system index [" + BlobStoreCacheService.this.index + "]", (Throwable)e);
                    assert (false) : e;
                }
                listener.onResponse((Object)CachedBlob.CACHE_NOT_READY);
            }
        });
    }

    protected void innerGet(GetRequest request, ActionListener<GetResponse> listener) {
        this.client.get(request, listener);
    }

    private static boolean assertDocId(GetResponse response, String repository, SnapshotId snapshotId, IndexId indexId, ShardId shardId, String name, ByteRange range) {
        String expectedId = BlobStoreCacheService.generateId(repository, snapshotId, indexId, shardId, name, range);
        assert (response.getId().equals(expectedId)) : "Expected a cached blob document with id [" + expectedId + "] but got [" + response.getId() + "]";
        return true;
    }

    private static boolean isExpectedCacheGetException(Exception e) {
        if (TransportActions.isShardNotAvailableException((Throwable)e) || e instanceof ConnectTransportException || e instanceof ClusterBlockException) {
            return true;
        }
        Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
        return cause instanceof NodeClosedException || cause instanceof ConnectTransportException;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void putAsync(String repository, SnapshotId snapshotId, IndexId indexId, ShardId shardId, String name, ByteRange range, BytesReference bytes, long timeInEpochMillis, ActionListener<Void> listener) {
        if (this.closed.get()) {
            listener.onFailure((Exception)new IllegalStateException("Blob cache service is closed"));
            return;
        }
        String id = BlobStoreCacheService.generateId(repository, snapshotId, indexId, shardId, name, range);
        try {
            CachedBlob cachedBlob = new CachedBlob(Instant.ofEpochMilli(timeInEpochMillis), Version.CURRENT, repository, name, BlobStoreCacheService.generatePath(snapshotId, indexId, shardId), bytes, range.start());
            final IndexRequest request = new IndexRequest(this.index).id(id);
            try (XContentBuilder builder = XContentFactory.jsonBuilder();){
                request.source(cachedBlob.toXContent(builder, ToXContent.EMPTY_PARAMS));
            }
            RunOnce release = new RunOnce(() -> {
                int availablePermits = this.inFlightCacheFills.availablePermits();
                assert (availablePermits > 0) : "in-flight available permits should be greater than 0 but got: " + availablePermits;
                this.inFlightCacheFills.release();
            });
            boolean submitted = false;
            this.inFlightCacheFills.acquire();
            try {
                final ActionListener wrappedListener = ActionListener.runAfter(listener, (Runnable)release);
                this.innerPut(request, new ActionListener<IndexResponse>(){

                    public void onResponse(IndexResponse indexResponse) {
                        logger.trace("cache fill ({}): [{}]", (Object)indexResponse.status(), (Object)request.id());
                        wrappedListener.onResponse(null);
                    }

                    public void onFailure(Exception e) {
                        logger.debug(() -> "failure in cache fill: [" + request.id() + "]", (Throwable)e);
                        wrappedListener.onFailure(e);
                    }
                });
                submitted = true;
            }
            finally {
                if (!submitted) {
                    release.run();
                }
            }
        }
        catch (Exception e) {
            logger.warn(() -> "cache fill failure: [" + id + "]", (Throwable)e);
            listener.onFailure(e);
        }
    }

    protected void innerPut(IndexRequest request, ActionListener<IndexResponse> listener) {
        this.client.index(request, listener);
    }

    protected static String generateId(String repository, SnapshotId snapshotId, IndexId indexId, ShardId shardId, String name, ByteRange range) {
        return String.join((CharSequence)"/", repository, snapshotId.getUUID(), indexId.getId(), String.valueOf(shardId.id()), name, "@" + range.start());
    }

    protected static String generatePath(SnapshotId snapshotId, IndexId indexId, ShardId shardId) {
        return String.join((CharSequence)"/", snapshotId.getUUID(), indexId.getId(), String.valueOf(shardId.id()));
    }

    public ByteRange computeBlobCacheByteRange(ShardId shardId, String fileName, long fileLength, ByteSizeValue maxMetadataLength) {
        LuceneFilesExtensions fileExtension = LuceneFilesExtensions.fromExtension((String)IndexFileNames.getExtension((String)fileName));
        if (this.useLegacyCachedBlobSizes()) {
            if (fileLength <= ByteSizeUnit.KB.toBytes(8L)) {
                return ByteRange.of((long)0L, (long)fileLength);
            }
            return ByteRange.of((long)0L, (long)ByteSizeUnit.KB.toBytes(4L));
        }
        if (fileExtension != null && fileExtension.isMetadata()) {
            long maxAllowedLengthInBytes = maxMetadataLength.getBytes();
            if (fileLength > maxAllowedLengthInBytes) {
                BlobStoreCacheService.logExceedingFile(shardId, fileExtension, fileLength, maxMetadataLength);
            }
            return ByteRange.of((long)0L, (long)Math.min(fileLength, maxAllowedLengthInBytes));
        }
        return ByteRange.of((long)0L, (long)Math.min(fileLength, (long)DEFAULT_CACHED_BLOB_SIZE));
    }

    protected boolean useLegacyCachedBlobSizes() {
        Version minNodeVersion = this.clusterService.state().nodes().getMinNodeVersion();
        return minNodeVersion.before((VersionId)OLD_CACHED_BLOB_SIZE_VERSION);
    }

    private static void logExceedingFile(ShardId shardId, LuceneFilesExtensions extension, long length, ByteSizeValue maxAllowedLength) {
        if (logger.isInfoEnabled()) {
            try {
                LOG_EXCEEDING_FILES_CACHE.computeIfAbsent((Object)extension.getExtension(), key -> {
                    logger.info("{} file with extension [{}] is larger ([{}]) than the max. length allowed [{}] to cache metadata files in blob cache", (Object)shardId, (Object)extension, (Object)length, (Object)maxAllowedLength);
                    return key;
                });
            }
            catch (ExecutionException e) {
                logger.warn(() -> Strings.format((String)"%s failed to log information about exceeding file type [%s] with length [%s]", (Object[])new Object[]{shardId, extension, length}), (Throwable)e);
            }
        }
    }
}

