/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.blobstore.testkit;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongPredicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionListenerResponseHandler;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionRequestValidationException;
import org.elasticsearch.action.ActionResponse;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.StepListener;
import org.elasticsearch.action.support.ActionFilters;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.action.support.HandledTransportAction;
import org.elasticsearch.action.support.ThreadedActionListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.InputStreamStreamInput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.repositories.RepositoriesService;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryVerificationException;
import org.elasticsearch.repositories.blobstore.BlobStoreRepository;
import org.elasticsearch.repositories.blobstore.testkit.BlobWriteAbortedException;
import org.elasticsearch.repositories.blobstore.testkit.GetBlobChecksumAction;
import org.elasticsearch.repositories.blobstore.testkit.RandomBlobContent;
import org.elasticsearch.repositories.blobstore.testkit.RandomBlobContentBytesReference;
import org.elasticsearch.repositories.blobstore.testkit.RandomBlobContentStream;
import org.elasticsearch.repositories.blobstore.testkit.SnapshotRepositoryTestKit;
import org.elasticsearch.tasks.CancellableTask;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.tasks.TaskAwareRequest;
import org.elasticsearch.tasks.TaskId;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentFragment;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContentBuilder;

public class BlobAnalyzeAction
extends ActionType<Response> {
    private static final Logger logger = LogManager.getLogger(BlobAnalyzeAction.class);
    public static final BlobAnalyzeAction INSTANCE = new BlobAnalyzeAction();
    public static final String NAME = "cluster:admin/repository/analyze/blob";
    static final long MAX_ATOMIC_WRITE_SIZE = Integer.MAX_VALUE;

    private BlobAnalyzeAction() {
        super(NAME, Response::new);
    }

    public static class ReadDetail
    implements Writeable,
    ToXContentFragment {
        private final String nodeId;
        private final String nodeName;
        private final boolean beforeWriteComplete;
        private final boolean isNotFound;
        private final long firstByteNanos;
        private final long throttleNanos;
        private final long elapsedNanos;

        public ReadDetail(String nodeId, String nodeName, boolean beforeWriteComplete, boolean isNotFound, long firstByteNanos, long elapsedNanos, long throttleNanos) {
            this.nodeId = nodeId;
            this.nodeName = nodeName;
            this.beforeWriteComplete = beforeWriteComplete;
            this.isNotFound = isNotFound;
            this.firstByteNanos = firstByteNanos;
            this.throttleNanos = throttleNanos;
            this.elapsedNanos = elapsedNanos;
        }

        public ReadDetail(StreamInput in) throws IOException {
            this.nodeId = in.readString();
            this.nodeName = in.readString();
            this.beforeWriteComplete = in.readBoolean();
            this.isNotFound = in.readBoolean();
            this.firstByteNanos = in.readVLong();
            this.throttleNanos = in.readVLong();
            this.elapsedNanos = in.readVLong();
        }

        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.nodeId);
            out.writeString(this.nodeName);
            out.writeBoolean(this.beforeWriteComplete);
            out.writeBoolean(this.isNotFound);
            out.writeVLong(this.firstByteNanos);
            out.writeVLong(this.throttleNanos);
            out.writeVLong(this.elapsedNanos);
        }

        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject();
            builder.startObject("node");
            builder.field("id", this.nodeId);
            builder.field("name", this.nodeName);
            builder.endObject();
            if (this.beforeWriteComplete) {
                builder.field("before_write_complete", true);
            }
            if (this.isNotFound) {
                builder.field("found", false);
            } else {
                builder.field("found", true);
                SnapshotRepositoryTestKit.humanReadableNanos(builder, "first_byte_time_nanos", "first_byte_time", this.firstByteNanos);
                SnapshotRepositoryTestKit.humanReadableNanos(builder, "elapsed_nanos", "elapsed", this.elapsedNanos);
                SnapshotRepositoryTestKit.humanReadableNanos(builder, "throttled_nanos", "throttled", this.throttleNanos);
            }
            builder.endObject();
            return builder;
        }

        long getFirstByteNanos() {
            return this.firstByteNanos;
        }

        long getThrottledNanos() {
            return this.throttleNanos;
        }

        long getElapsedNanos() {
            return this.elapsedNanos;
        }
    }

    public static class Response
    extends ActionResponse
    implements ToXContentObject {
        private final String nodeId;
        private final String nodeName;
        private final String blobName;
        private final long blobLength;
        private final boolean readEarly;
        private final boolean overwrite;
        private final long checksumStart;
        private final long checksumEnd;
        private final long writeElapsedNanos;
        private final long overwriteElapsedNanos;
        private final long writeThrottledNanos;
        private final List<ReadDetail> readDetails;

        public Response(String nodeId, String nodeName, String blobName, long blobLength, boolean readEarly, boolean overwrite, long checksumStart, long checksumEnd, long writeElapsedNanos, long overwriteElapsedNanos, long writeThrottledNanos, List<ReadDetail> readDetails) {
            this.nodeId = nodeId;
            this.nodeName = nodeName;
            this.blobName = blobName;
            this.blobLength = blobLength;
            this.readEarly = readEarly;
            this.overwrite = overwrite;
            this.checksumStart = checksumStart;
            this.checksumEnd = checksumEnd;
            this.writeElapsedNanos = writeElapsedNanos;
            this.overwriteElapsedNanos = overwriteElapsedNanos;
            this.writeThrottledNanos = writeThrottledNanos;
            this.readDetails = readDetails;
        }

        public Response(StreamInput in) throws IOException {
            super(in);
            this.nodeId = in.readString();
            this.nodeName = in.readString();
            this.blobName = in.readString();
            this.blobLength = in.readVLong();
            this.readEarly = in.readBoolean();
            this.overwrite = in.readBoolean();
            this.checksumStart = in.readVLong();
            this.checksumEnd = in.readVLong();
            this.writeElapsedNanos = in.readVLong();
            this.overwriteElapsedNanos = in.readVLong();
            this.writeThrottledNanos = in.readVLong();
            this.readDetails = in.readList(ReadDetail::new);
        }

        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.nodeId);
            out.writeString(this.nodeName);
            out.writeString(this.blobName);
            out.writeVLong(this.blobLength);
            out.writeBoolean(this.readEarly);
            out.writeBoolean(this.overwrite);
            out.writeVLong(this.checksumStart);
            out.writeVLong(this.checksumEnd);
            out.writeVLong(this.writeElapsedNanos);
            out.writeVLong(this.overwriteElapsedNanos);
            out.writeVLong(this.writeThrottledNanos);
            out.writeList(this.readDetails);
        }

        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject();
            builder.startObject("blob");
            builder.field("name", this.blobName);
            builder.humanReadableField("size_bytes", "size", (Object)new ByteSizeValue(this.blobLength));
            builder.field("read_start", this.checksumStart);
            builder.field("read_end", this.checksumEnd);
            builder.field("read_early", this.readEarly);
            builder.field("overwritten", this.overwrite);
            builder.endObject();
            builder.startObject("writer_node");
            builder.field("id", this.nodeId);
            builder.field("name", this.nodeName);
            builder.endObject();
            SnapshotRepositoryTestKit.humanReadableNanos(builder, "write_elapsed_nanos", "write_elapsed", this.writeElapsedNanos);
            if (this.overwrite) {
                SnapshotRepositoryTestKit.humanReadableNanos(builder, "overwrite_elapsed_nanos", "overwrite_elapsed", this.overwriteElapsedNanos);
            }
            SnapshotRepositoryTestKit.humanReadableNanos(builder, "write_throttled_nanos", "write_throttled", this.writeThrottledNanos);
            builder.startArray("reads");
            for (ReadDetail readDetail : this.readDetails) {
                readDetail.toXContent(builder, params);
            }
            builder.endArray();
            builder.endObject();
            return builder;
        }

        long getWriteBytes() {
            return this.blobLength + (this.overwrite ? this.blobLength : 0L);
        }

        long getWriteThrottledNanos() {
            return this.writeThrottledNanos;
        }

        long getWriteElapsedNanos() {
            return this.writeElapsedNanos + this.overwriteElapsedNanos;
        }

        List<ReadDetail> getReadDetails() {
            return this.readDetails;
        }

        long getChecksumBytes() {
            return this.checksumEnd - this.checksumStart;
        }
    }

    public static class Request
    extends ActionRequest
    implements TaskAwareRequest {
        private final String repositoryName;
        private final String blobPath;
        private final String blobName;
        private final long targetLength;
        private final long seed;
        private final List<DiscoveryNode> nodes;
        private final int readNodeCount;
        private final int earlyReadNodeCount;
        private final boolean readEarly;
        private final boolean writeAndOverwrite;
        private final boolean abortWrite;

        Request(String repositoryName, String blobPath, String blobName, long targetLength, long seed, List<DiscoveryNode> nodes, int readNodeCount, int earlyReadNodeCount, boolean readEarly, boolean writeAndOverwrite, boolean abortWrite) {
            assert (0L < targetLength);
            assert (targetLength <= Integer.MAX_VALUE || !readEarly && !writeAndOverwrite) : "oversized atomic write";
            assert (!writeAndOverwrite || !abortWrite) : "cannot set writeAndOverwrite and abortWrite";
            this.repositoryName = repositoryName;
            this.blobPath = blobPath;
            this.blobName = blobName;
            this.targetLength = targetLength;
            this.seed = seed;
            this.nodes = nodes;
            this.readNodeCount = readNodeCount;
            this.earlyReadNodeCount = earlyReadNodeCount;
            this.readEarly = readEarly;
            this.writeAndOverwrite = writeAndOverwrite;
            this.abortWrite = abortWrite;
        }

        Request(StreamInput in) throws IOException {
            super(in);
            this.repositoryName = in.readString();
            this.blobPath = in.readString();
            this.blobName = in.readString();
            this.targetLength = in.readVLong();
            this.seed = in.readLong();
            this.nodes = in.readList(DiscoveryNode::new);
            this.readNodeCount = in.readVInt();
            this.earlyReadNodeCount = in.readVInt();
            this.readEarly = in.readBoolean();
            this.writeAndOverwrite = in.readBoolean();
            this.abortWrite = in.getVersion().onOrAfter(Version.V_7_14_0) ? in.readBoolean() : false;
        }

        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeString(this.repositoryName);
            out.writeString(this.blobPath);
            out.writeString(this.blobName);
            out.writeVLong(this.targetLength);
            out.writeLong(this.seed);
            out.writeList(this.nodes);
            out.writeVInt(this.readNodeCount);
            out.writeVInt(this.earlyReadNodeCount);
            out.writeBoolean(this.readEarly);
            out.writeBoolean(this.writeAndOverwrite);
            if (out.getVersion().onOrAfter(Version.V_7_14_0)) {
                out.writeBoolean(this.abortWrite);
            } else if (this.abortWrite) {
                throw new IllegalStateException("cannot send abortWrite request to node of version [" + out.getVersion() + "]");
            }
        }

        public ActionRequestValidationException validate() {
            return null;
        }

        public String getDescription() {
            return "blob analysis [" + this.repositoryName + ":" + this.blobPath + "/" + this.blobName + ", length=" + this.targetLength + ", seed=" + this.seed + ", readEarly=" + this.readEarly + ", writeAndOverwrite=" + this.writeAndOverwrite + ", abortWrite=" + this.abortWrite + "]";
        }

        public String toString() {
            return "BlobAnalyzeAction.Request{" + this.getDescription() + "}";
        }

        public Task createTask(long id, String type, String action, TaskId parentTaskId, Map<String, String> headers) {
            return new CancellableTask(id, type, action, this.getDescription(), parentTaskId, headers){

                public boolean shouldCancelChildrenOnCancellation() {
                    return true;
                }
            };
        }

        public String getRepositoryName() {
            return this.repositoryName;
        }

        public String getBlobPath() {
            return this.blobPath;
        }

        public String getBlobName() {
            return this.blobName;
        }

        public long getTargetLength() {
            return this.targetLength;
        }

        public boolean getAbortWrite() {
            return this.abortWrite;
        }
    }

    private static class WriteDetails {
        final long bytesWritten;
        final long elapsedNanos;
        final long throttledNanos;
        final long checksum;

        private WriteDetails(long bytesWritten, long elapsedNanos, long throttledNanos, long checksum) {
            this.bytesWritten = bytesWritten;
            this.elapsedNanos = elapsedNanos;
            this.throttledNanos = throttledNanos;
            this.checksum = checksum;
        }
    }

    private static class NodeResponse {
        final DiscoveryNode node;
        final boolean beforeWriteComplete;
        final GetBlobChecksumAction.Response response;

        NodeResponse(DiscoveryNode node, boolean beforeWriteComplete, GetBlobChecksumAction.Response response) {
            this.node = node;
            this.beforeWriteComplete = beforeWriteComplete;
            this.response = response;
        }
    }

    static class BlobAnalysis {
        private final TransportService transportService;
        private final CancellableTask task;
        private final Request request;
        private final BlobStoreRepository repository;
        private final BlobContainer blobContainer;
        private final ActionListener<Response> listener;
        private final Random random;
        private final boolean checksumWholeBlob;
        private final long checksumStart;
        private final long checksumEnd;
        private final List<DiscoveryNode> earlyReadNodes;
        private final List<DiscoveryNode> readNodes;
        private final GroupedActionListener<NodeResponse> readNodesListener;
        private final StepListener<WriteDetails> write1Step = new StepListener();
        private final StepListener<WriteDetails> write2Step = new StepListener();

        BlobAnalysis(TransportService transportService, CancellableTask task, Request request, BlobStoreRepository repository, BlobContainer blobContainer, ActionListener<Response> listener) {
            this.transportService = transportService;
            this.task = task;
            this.request = request;
            this.repository = repository;
            this.blobContainer = blobContainer;
            this.listener = listener;
            this.random = new Random(this.request.seed);
            this.checksumWholeBlob = this.random.nextBoolean();
            if (this.checksumWholeBlob) {
                this.checksumStart = 0L;
                this.checksumEnd = request.targetLength;
            } else {
                this.checksumStart = this.randomLongBetween(0L, request.targetLength);
                this.checksumEnd = this.randomLongBetween(this.checksumStart + 1L, request.targetLength + 1L);
            }
            ArrayList<DiscoveryNode> nodes = new ArrayList<DiscoveryNode>(request.nodes);
            if (request.readEarly) {
                Collections.shuffle(nodes, this.random);
                this.earlyReadNodes = nodes.stream().limit(request.earlyReadNodeCount).collect(Collectors.toList());
            } else {
                this.earlyReadNodes = List.of();
            }
            Collections.shuffle(nodes, this.random);
            this.readNodes = nodes.stream().limit(request.readNodeCount).collect(Collectors.toList());
            StepListener readsCompleteStep = new StepListener();
            this.readNodesListener = new GroupedActionListener((ActionListener)new ThreadedActionListener(logger, transportService.getThreadPool(), "snapshot", (ActionListener)readsCompleteStep, false), this.earlyReadNodes.size() + this.readNodes.size());
            this.write1Step.whenComplete(write1Details -> this.write2Step.whenComplete(write2Details -> readsCompleteStep.whenComplete(responses -> this.onReadsComplete((Collection<NodeResponse>)responses, (WriteDetails)write1Details, (WriteDetails)write2Details), this::cleanUpAndReturnFailure), this::cancelReadsCleanUpAndReturnFailure), this::cancelReadsCleanUpAndReturnFailure);
        }

        void run() {
            this.writeRandomBlob(this.request.readEarly || this.request.getAbortWrite() || this.request.targetLength <= Integer.MAX_VALUE && this.random.nextBoolean(), true, this::onLastReadForInitialWrite, this.write1Step);
            if (this.request.writeAndOverwrite) {
                assert (this.request.targetLength <= Integer.MAX_VALUE) : "oversized atomic write";
                this.write1Step.whenComplete(ignored -> this.writeRandomBlob(true, false, this::doReadAfterWrite, this.write2Step), ignored -> {});
            } else {
                this.write2Step.onResponse(null);
                this.doReadAfterWrite();
            }
        }

        private void writeRandomBlob(boolean atomic, boolean failIfExists, Runnable onLastRead, StepListener<WriteDetails> stepListener) {
            assert (!atomic || this.request.targetLength <= Integer.MAX_VALUE) : "oversized atomic write";
            RandomBlobContent content = new RandomBlobContent(this.request.getRepositoryName(), this.random.nextLong(), () -> ((CancellableTask)this.task).isCancelled(), onLastRead);
            final AtomicLong throttledNanos = new AtomicLong();
            if (logger.isTraceEnabled()) {
                logger.trace("writing blob [atomic={}, failIfExists={}] for [{}]", (Object)atomic, (Object)failIfExists, (Object)this.request.getDescription());
            }
            long startNanos = System.nanoTime();
            ActionListener.completeWith(stepListener, () -> {
                if (atomic || this.request.targetLength <= Integer.MAX_VALUE && this.random.nextBoolean()) {
                    RandomBlobContentBytesReference bytesReference = new RandomBlobContentBytesReference(content, Math.toIntExact(this.request.getTargetLength())){

                        public StreamInput streamInput() throws IOException {
                            return new InputStreamStreamInput(repository.maybeRateLimitSnapshots((InputStream)super.streamInput(), throttledNanos::addAndGet));
                        }
                    };
                    if (atomic) {
                        try {
                            this.blobContainer.writeBlobAtomic(this.request.blobName, (BytesReference)bytesReference, failIfExists);
                        }
                        catch (BlobWriteAbortedException e) {
                            assert (this.request.getAbortWrite()) : "write unexpectedly aborted";
                        }
                    } else {
                        this.blobContainer.writeBlob(this.request.blobName, (BytesReference)bytesReference, failIfExists);
                    }
                } else {
                    this.blobContainer.writeBlob(this.request.blobName, this.repository.maybeRateLimitSnapshots((InputStream)new RandomBlobContentStream(content, this.request.getTargetLength()), throttledNanos::addAndGet), this.request.targetLength, failIfExists);
                }
                long elapsedNanos = System.nanoTime() - startNanos;
                long checksum = content.getChecksum(this.checksumStart, this.checksumEnd);
                if (logger.isTraceEnabled()) {
                    logger.trace("finished writing blob for [{}], got checksum [{}]", (Object)this.request.getDescription(), (Object)checksum);
                }
                return new WriteDetails(this.request.targetLength, elapsedNanos, throttledNanos.get(), checksum);
            });
        }

        private void onLastReadForInitialWrite() {
            if (!this.earlyReadNodes.isEmpty()) {
                if (logger.isTraceEnabled()) {
                    logger.trace("sending read request to [{}] for [{}] before write complete", this.earlyReadNodes, (Object)this.request.getDescription());
                }
                this.readOnNodes(this.earlyReadNodes, true);
            }
            if (this.request.getAbortWrite()) {
                throw new BlobWriteAbortedException();
            }
        }

        private void doReadAfterWrite() {
            if (logger.isTraceEnabled()) {
                logger.trace("sending read request to [{}] for [{}] after write complete", this.readNodes, (Object)this.request.getDescription());
            }
            this.readOnNodes(this.readNodes, false);
        }

        private void readOnNodes(List<DiscoveryNode> nodes, final boolean beforeWriteComplete) {
            for (final DiscoveryNode node : nodes) {
                if (this.task.isCancelled()) {
                    this.readNodesListener.onResponse((Object)new NodeResponse(node, beforeWriteComplete, GetBlobChecksumAction.Response.BLOB_NOT_FOUND));
                    continue;
                }
                final GetBlobChecksumAction.Request blobChecksumRequest = this.getBlobChecksumRequest();
                this.transportService.sendChildRequest(node, "cluster:admin/repository/analyze/blob/read", (TransportRequest)blobChecksumRequest, (Task)this.task, TransportRequestOptions.EMPTY, (TransportResponseHandler)new ActionListenerResponseHandler((ActionListener)new ActionListener<GetBlobChecksumAction.Response>(){

                    public void onResponse(GetBlobChecksumAction.Response response) {
                        readNodesListener.onResponse((Object)this.makeNodeResponse(node, beforeWriteComplete, response));
                    }

                    public void onFailure(Exception e) {
                        readNodesListener.onFailure((Exception)new RepositoryVerificationException(request.getRepositoryName(), "[" + blobChecksumRequest + "] (" + (beforeWriteComplete ? "before" : "after") + " write complete) failed on node [" + node + "]", (Throwable)e));
                    }
                }, GetBlobChecksumAction.Response::new));
            }
        }

        private GetBlobChecksumAction.Request getBlobChecksumRequest() {
            return new GetBlobChecksumAction.Request(this.request.getRepositoryName(), this.request.getBlobPath(), this.request.getBlobName(), this.checksumStart, this.checksumWholeBlob ? 0L : this.checksumEnd);
        }

        private NodeResponse makeNodeResponse(DiscoveryNode node, boolean beforeWriteComplete, GetBlobChecksumAction.Response response) {
            logger.trace("received read response [{}] from [{}] for [{}] [beforeWriteComplete={}]", (Object)response, (Object)node, (Object)this.request.getDescription(), (Object)beforeWriteComplete);
            return new NodeResponse(node, beforeWriteComplete, response);
        }

        private void cancelReadsCleanUpAndReturnFailure(Exception exception) {
            this.transportService.getTaskManager().cancelTaskAndDescendants(this.task, "task failed", false, ActionListener.wrap(() -> {}));
            this.cleanUpAndReturnFailure(exception);
        }

        private void cleanUpAndReturnFailure(Exception exception) {
            if (logger.isTraceEnabled()) {
                logger.trace((Message)new ParameterizedMessage("analysis failed [{}] cleaning up", (Object)this.request.getDescription()), (Throwable)exception);
            }
            try {
                this.blobContainer.deleteBlobsIgnoringIfNotExists(Iterators.single((Object)this.request.blobName));
            }
            catch (IOException ioException) {
                exception.addSuppressed(ioException);
                logger.warn((Message)new ParameterizedMessage("failure during post-failure cleanup while analysing repository [{}], you may need to manually remove [{}/{}]", new Object[]{this.request.getRepositoryName(), this.request.getBlobPath(), this.request.getBlobName()}), (Throwable)exception);
            }
            this.listener.onFailure((Exception)new RepositoryVerificationException(this.request.getRepositoryName(), "failure processing [" + this.request.getDescription() + "]", (Throwable)exception));
        }

        private void onReadsComplete(Collection<NodeResponse> responses, WriteDetails write1Details, @Nullable WriteDetails write2Details) {
            Object expectedChecksumDescription;
            LongPredicate checksumPredicate;
            if (this.task.isCancelled()) {
                this.cleanUpAndReturnFailure((Exception)new RepositoryVerificationException(this.request.getRepositoryName(), "cancelled during checksum verification"));
                return;
            }
            long checksumLength = this.checksumEnd - this.checksumStart;
            if (write2Details == null) {
                checksumPredicate = l -> l == write1Details.checksum;
                expectedChecksumDescription = Long.toString(write1Details.checksum);
            } else {
                checksumPredicate = l -> l == write1Details.checksum || l == write2Details.checksum;
                expectedChecksumDescription = write1Details.checksum + " or " + write2Details.checksum;
            }
            boolean anyFound = false;
            RepositoryVerificationException failure = null;
            for (NodeResponse nodeResponse : responses) {
                RepositoryVerificationException nodeFailure;
                GetBlobChecksumAction.Response response = nodeResponse.response;
                if (response.isNotFound()) {
                    nodeFailure = this.request.readEarly || this.request.getAbortWrite() ? null : new RepositoryVerificationException(this.request.getRepositoryName(), "node [" + nodeResponse.node + "] reported blob not found after it was written");
                } else {
                    anyFound = true;
                    long actualChecksum = response.getChecksum();
                    nodeFailure = response.getBytesRead() == checksumLength && checksumPredicate.test(actualChecksum) ? null : new RepositoryVerificationException(this.request.getRepositoryName(), "node [" + nodeResponse.node + "] failed during analysis: expected to read [" + this.checksumStart + "-" + this.checksumEnd + "], [" + checksumLength + "] bytes, with checksum [" + (String)expectedChecksumDescription + "] but read [" + response + "]");
                }
                if (nodeFailure == null) continue;
                if (failure == null) {
                    failure = nodeFailure;
                    continue;
                }
                failure.addSuppressed(nodeFailure);
            }
            if (this.request.getAbortWrite() && anyFound) {
                RepositoryVerificationException atomicityFailure = new RepositoryVerificationException(this.request.getRepositoryName(), "upload of blob was aborted, but blob was erroneously found by at least one node");
                if (failure == null) {
                    failure = atomicityFailure;
                } else {
                    failure.addSuppressed((Throwable)atomicityFailure);
                }
            }
            if (failure != null) {
                this.cleanUpAndReturnFailure((Exception)failure);
                return;
            }
            long overwriteElapsedNanos = write2Details == null ? 0L : write2Details.elapsedNanos;
            long overwriteThrottledNanos = write2Details == null ? 0L : write2Details.throttledNanos;
            this.listener.onResponse((Object)new Response(this.transportService.getLocalNode().getId(), this.transportService.getLocalNode().getName(), this.request.blobName, this.request.targetLength, this.request.readEarly, this.request.writeAndOverwrite, this.checksumStart, this.checksumEnd, write1Details.elapsedNanos, overwriteElapsedNanos, write1Details.throttledNanos + overwriteThrottledNanos, responses.stream().map(nr -> new ReadDetail(nr.node.getId(), nr.node.getName(), nr.beforeWriteComplete, nr.response.isNotFound(), nr.response.getFirstByteNanos(), nr.response.getElapsedNanos(), nr.response.getThrottleNanos())).collect(Collectors.toList())));
        }

        private long randomLongBetween(long min, long max) {
            assert (0L <= min && min <= max);
            long range = max - min;
            return range == 0L ? min : min + (this.random.nextLong() & Long.MAX_VALUE) % range;
        }
    }

    public static class TransportAction
    extends HandledTransportAction<Request, Response> {
        private static final Logger logger = logger;
        private final RepositoriesService repositoriesService;
        private final TransportService transportService;

        @Inject
        public TransportAction(TransportService transportService, ActionFilters actionFilters, RepositoriesService repositoriesService) {
            super(BlobAnalyzeAction.NAME, transportService, actionFilters, Request::new, "snapshot");
            this.repositoriesService = repositoriesService;
            this.transportService = transportService;
        }

        protected void doExecute(Task task, Request request, ActionListener<Response> listener) {
            Repository repository = this.repositoriesService.repository(request.getRepositoryName());
            if (!(repository instanceof BlobStoreRepository)) {
                throw new IllegalArgumentException("repository [" + request.getRepositoryName() + "] is not a blob-store repository");
            }
            if (repository.isReadOnly()) {
                throw new IllegalArgumentException("repository [" + request.getRepositoryName() + "] is read-only");
            }
            BlobStoreRepository blobStoreRepository = (BlobStoreRepository)repository;
            BlobPath path = blobStoreRepository.basePath().add(request.blobPath);
            BlobContainer blobContainer = blobStoreRepository.blobStore().blobContainer(path);
            logger.trace("handling [{}]", (Object)request);
            assert (task instanceof CancellableTask);
            new BlobAnalysis(this.transportService, (CancellableTask)task, request, blobStoreRepository, blobContainer, listener).run();
        }
    }
}

