/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.datafeed.extractor.chunked;

import java.io.IOException;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionRequestBuilder;
import org.elasticsearch.action.search.SearchAction;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.ElasticsearchClient;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.search.aggregations.AggregationBuilder;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.metrics.Max;
import org.elasticsearch.search.aggregations.metrics.Min;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.datafeed.SearchInterval;
import org.elasticsearch.xpack.core.ml.datafeed.extractor.DataExtractor;
import org.elasticsearch.xpack.core.ml.datafeed.extractor.ExtractorUtils;
import org.elasticsearch.xpack.core.rollup.action.RollupSearchAction;
import org.elasticsearch.xpack.ml.datafeed.DatafeedTimingStatsReporter;
import org.elasticsearch.xpack.ml.datafeed.extractor.DataExtractorFactory;
import org.elasticsearch.xpack.ml.datafeed.extractor.aggregation.RollupDataExtractorFactory;
import org.elasticsearch.xpack.ml.datafeed.extractor.chunked.ChunkedDataExtractorContext;

public class ChunkedDataExtractor
implements DataExtractor {
    private static final Logger LOGGER = LogManager.getLogger(ChunkedDataExtractor.class);
    private static final String EARLIEST_TIME = "earliest_time";
    private static final String LATEST_TIME = "latest_time";
    private static final long MIN_CHUNK_SPAN = 60000L;
    private final Client client;
    private final DataExtractorFactory dataExtractorFactory;
    private final ChunkedDataExtractorContext context;
    private final DataSummaryFactory dataSummaryFactory;
    private final DatafeedTimingStatsReporter timingStatsReporter;
    private long currentStart;
    private long currentEnd;
    private long chunkSpan;
    private boolean isCancelled;
    private DataExtractor currentExtractor;

    public ChunkedDataExtractor(Client client, DataExtractorFactory dataExtractorFactory, ChunkedDataExtractorContext context, DatafeedTimingStatsReporter timingStatsReporter) {
        this.client = Objects.requireNonNull(client);
        this.dataExtractorFactory = Objects.requireNonNull(dataExtractorFactory);
        this.context = Objects.requireNonNull(context);
        this.timingStatsReporter = Objects.requireNonNull(timingStatsReporter);
        this.currentStart = context.start;
        this.currentEnd = context.start;
        this.isCancelled = false;
        this.dataSummaryFactory = new DataSummaryFactory();
    }

    public boolean hasNext() {
        boolean currentHasNext;
        boolean bl = currentHasNext = this.currentExtractor != null && this.currentExtractor.hasNext();
        if (this.isCancelled()) {
            return currentHasNext;
        }
        return currentHasNext || this.currentEnd < this.context.end;
    }

    public DataExtractor.Result next() throws IOException {
        if (!this.hasNext()) {
            throw new NoSuchElementException();
        }
        if (this.currentExtractor == null) {
            this.setUpChunkedSearch();
        }
        return this.getNextStream();
    }

    private void setUpChunkedSearch() {
        DataSummary dataSummary = this.dataSummaryFactory.buildDataSummary();
        if (dataSummary.hasData()) {
            this.currentEnd = this.currentStart = this.context.timeAligner.alignToFloor(dataSummary.earliestTime());
            this.chunkSpan = this.context.chunkSpan == null ? dataSummary.estimateChunk() : this.context.chunkSpan.getMillis();
            this.chunkSpan = this.context.timeAligner.alignToCeil(this.chunkSpan);
            LOGGER.debug("[{}] Chunked search configured: kind = {}, dataTimeSpread = {} ms, chunk span = {} ms", (Object)this.context.jobId, (Object)dataSummary.getClass().getSimpleName(), (Object)dataSummary.getDataTimeSpread(), (Object)this.chunkSpan);
        } else {
            this.currentEnd = this.context.end;
            LOGGER.debug("[{}] Chunked search configured: no data found", (Object)this.context.jobId);
        }
    }

    protected SearchResponse executeSearchRequest(ActionRequestBuilder<SearchRequest, SearchResponse> searchRequestBuilder) {
        return (SearchResponse)ClientHelper.executeWithHeaders(this.context.headers, (String)"ml", (Client)this.client, () -> searchRequestBuilder.get());
    }

    private DataExtractor.Result getNextStream() throws IOException {
        SearchInterval lastSearchInterval = new SearchInterval(this.context.start, this.context.end);
        while (this.hasNext()) {
            boolean isNewSearch = false;
            if (this.currentExtractor == null || !this.currentExtractor.hasNext()) {
                this.advanceTime();
                isNewSearch = true;
            }
            DataExtractor.Result result = this.currentExtractor.next();
            lastSearchInterval = result.searchInterval();
            if (result.data().isPresent()) {
                return result;
            }
            if (!isNewSearch || !this.hasNext()) continue;
            this.setUpChunkedSearch();
        }
        return new DataExtractor.Result(lastSearchInterval, Optional.empty());
    }

    private void advanceTime() {
        this.currentStart = this.currentEnd;
        this.currentEnd = Math.min(this.currentStart + this.chunkSpan, this.context.end);
        this.currentExtractor = this.dataExtractorFactory.newExtractor(this.currentStart, this.currentEnd);
        LOGGER.trace("[{}] advances time to [{}, {})", (Object)this.context.jobId, (Object)this.currentStart, (Object)this.currentEnd);
    }

    public boolean isCancelled() {
        return this.isCancelled;
    }

    public void cancel() {
        if (this.currentExtractor != null) {
            this.currentExtractor.cancel();
        }
        this.isCancelled = true;
    }

    public long getEndTime() {
        return this.context.end;
    }

    ChunkedDataExtractorContext getContext() {
        return this.context;
    }

    private class DataSummaryFactory {
        private DataSummaryFactory() {
        }

        private DataSummary buildDataSummary() {
            return ChunkedDataExtractor.this.context.hasAggregations ? this.newAggregatedDataSummary() : this.newScrolledDataSummary();
        }

        private DataSummary newScrolledDataSummary() {
            SearchRequestBuilder searchRequestBuilder = this.rangeSearchRequest();
            SearchResponse searchResponse = ChunkedDataExtractor.this.executeSearchRequest((ActionRequestBuilder<SearchRequest, SearchResponse>)searchRequestBuilder);
            LOGGER.debug("[{}] Scrolling Data summary response was obtained", (Object)ChunkedDataExtractor.this.context.jobId);
            ChunkedDataExtractor.this.timingStatsReporter.reportSearchDuration(searchResponse.getTook());
            long earliestTime = 0L;
            long latestTime = 0L;
            long totalHits = searchResponse.getHits().getTotalHits().value;
            if (totalHits > 0L) {
                Aggregations aggregations = searchResponse.getAggregations();
                Min min = (Min)aggregations.get(ChunkedDataExtractor.EARLIEST_TIME);
                earliestTime = (long)min.getValue();
                Max max = (Max)aggregations.get(ChunkedDataExtractor.LATEST_TIME);
                latestTime = (long)max.getValue();
            }
            return new ScrolledDataSummary(earliestTime, latestTime, totalHits);
        }

        private DataSummary newAggregatedDataSummary() {
            RollupSearchAction.RequestBuilder searchRequestBuilder = ChunkedDataExtractor.this.dataExtractorFactory instanceof RollupDataExtractorFactory ? this.rollupRangeSearchRequest() : this.rangeSearchRequest();
            SearchResponse searchResponse = ChunkedDataExtractor.this.executeSearchRequest((ActionRequestBuilder<SearchRequest, SearchResponse>)searchRequestBuilder);
            LOGGER.debug("[{}] Aggregating Data summary response was obtained", (Object)ChunkedDataExtractor.this.context.jobId);
            ChunkedDataExtractor.this.timingStatsReporter.reportSearchDuration(searchResponse.getTook());
            Aggregations aggregations = searchResponse.getAggregations();
            if (aggregations == null) {
                return AggregatedDataSummary.noDataSummary(ChunkedDataExtractor.this.context.histogramInterval);
            }
            Min min = (Min)aggregations.get(ChunkedDataExtractor.EARLIEST_TIME);
            Max max = (Max)aggregations.get(ChunkedDataExtractor.LATEST_TIME);
            return new AggregatedDataSummary(min.getValue(), max.getValue(), ChunkedDataExtractor.this.context.histogramInterval);
        }

        private SearchSourceBuilder rangeSearchBuilder() {
            return new SearchSourceBuilder().size(0).query(ExtractorUtils.wrapInTimeRangeQuery((QueryBuilder)ChunkedDataExtractor.this.context.query, (String)ChunkedDataExtractor.this.context.timeField, (long)ChunkedDataExtractor.this.currentStart, (long)ChunkedDataExtractor.this.context.end)).runtimeMappings(ChunkedDataExtractor.this.context.runtimeMappings).aggregation((AggregationBuilder)AggregationBuilders.min((String)ChunkedDataExtractor.EARLIEST_TIME).field(ChunkedDataExtractor.this.context.timeField)).aggregation((AggregationBuilder)AggregationBuilders.max((String)ChunkedDataExtractor.LATEST_TIME).field(ChunkedDataExtractor.this.context.timeField));
        }

        private SearchRequestBuilder rangeSearchRequest() {
            return new SearchRequestBuilder((ElasticsearchClient)ChunkedDataExtractor.this.client, SearchAction.INSTANCE).setIndices(ChunkedDataExtractor.this.context.indices).setIndicesOptions(ChunkedDataExtractor.this.context.indicesOptions).setSource(this.rangeSearchBuilder()).setAllowPartialSearchResults(false).setTrackTotalHits(true);
        }

        private RollupSearchAction.RequestBuilder rollupRangeSearchRequest() {
            SearchRequest searchRequest = new SearchRequest().indices(ChunkedDataExtractor.this.context.indices).indicesOptions(ChunkedDataExtractor.this.context.indicesOptions).allowPartialSearchResults(false).source(this.rangeSearchBuilder());
            return new RollupSearchAction.RequestBuilder((ElasticsearchClient)ChunkedDataExtractor.this.client, searchRequest);
        }
    }

    static interface DataSummary {
        public long estimateChunk();

        public boolean hasData();

        public long earliestTime();

        public long getDataTimeSpread();
    }

    static class AggregatedDataSummary
    implements DataSummary {
        private final double earliestTime;
        private final double latestTime;
        private final long histogramIntervalMillis;

        static AggregatedDataSummary noDataSummary(long histogramInterval) {
            return new AggregatedDataSummary(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY, histogramInterval);
        }

        AggregatedDataSummary(double earliestTime, double latestTime, long histogramInterval) {
            this.earliestTime = earliestTime;
            this.latestTime = latestTime;
            this.histogramIntervalMillis = histogramInterval;
        }

        @Override
        public long estimateChunk() {
            return 1000L * this.histogramIntervalMillis;
        }

        @Override
        public boolean hasData() {
            return !(Double.isInfinite(this.earliestTime) || Double.isInfinite(this.latestTime));
        }

        @Override
        public long earliestTime() {
            return (long)this.earliestTime;
        }

        @Override
        public long getDataTimeSpread() {
            return (long)this.latestTime - (long)this.earliestTime;
        }
    }

    private class ScrolledDataSummary
    implements DataSummary {
        private final long earliestTime;
        private final long latestTime;
        private final long totalHits;

        private ScrolledDataSummary(long earliestTime, long latestTime, long totalHits) {
            this.earliestTime = earliestTime;
            this.latestTime = latestTime;
            this.totalHits = totalHits;
        }

        @Override
        public long earliestTime() {
            return this.earliestTime;
        }

        @Override
        public long getDataTimeSpread() {
            return this.latestTime - this.earliestTime;
        }

        @Override
        public long estimateChunk() {
            long dataTimeSpread = this.getDataTimeSpread();
            if (this.totalHits <= 0L || dataTimeSpread <= 0L) {
                return ChunkedDataExtractor.this.context.end - ChunkedDataExtractor.this.currentEnd;
            }
            long estimatedChunk = 10L * ((long)ChunkedDataExtractor.this.context.scrollSize * this.getDataTimeSpread()) / this.totalHits;
            return Math.max(estimatedChunk, 60000L);
        }

        @Override
        public boolean hasData() {
            return this.totalHits > 0L;
        }
    }
}

