/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.encrypted;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.util.Objects;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.ShortBufferException;
import javax.crypto.spec.GCMParameterSpec;
import org.elasticsearch.common.util.ByteUtils;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.repositories.encrypted.ChainingInputStream;
import org.elasticsearch.repositories.encrypted.PrefixInputStream;

public final class DecryptionPacketsInputStream
extends ChainingInputStream {
    private final InputStream source;
    private final SecretKey secretKey;
    private final int packetLength;
    private final byte[] packetBuffer;
    private boolean hasNext;
    private long counter;

    public static long getDecryptionLength(long ciphertextLength, int packetLength) {
        long encryptedPacketLength = packetLength + 16 + 12;
        long completePackets = ciphertextLength / encryptedPacketLength;
        long decryptedSize = completePackets * (long)packetLength;
        if (ciphertextLength % encryptedPacketLength != 0L) {
            decryptedSize += ciphertextLength % encryptedPacketLength - 12L - 16L;
        }
        return decryptedSize;
    }

    public DecryptionPacketsInputStream(InputStream source, SecretKey secretKey, int packetLength) {
        this.source = Objects.requireNonNull(source);
        this.secretKey = Objects.requireNonNull(secretKey);
        if (packetLength <= 0 || packetLength >= 0x800000) {
            throw new IllegalArgumentException("Invalid packet length [" + packetLength + "]");
        }
        this.packetLength = packetLength;
        this.packetBuffer = new byte[packetLength + 16];
        this.hasNext = true;
        this.counter = Long.MIN_VALUE;
    }

    @Override
    InputStream nextComponent(InputStream currentComponentIn) throws IOException {
        if (currentComponentIn != null && currentComponentIn.read() != -1) {
            throw new IllegalStateException("Stream for previous packet has not been fully processed");
        }
        if (!this.hasNext) {
            return null;
        }
        PrefixInputStream packetInputStream = new PrefixInputStream(this.source, this.packetLength + 12 + 16, false);
        int currentPacketLength = this.decrypt(packetInputStream);
        if (currentPacketLength != this.packetLength) {
            this.hasNext = false;
        }
        return new ByteArrayInputStream(this.packetBuffer, 0, currentPacketLength);
    }

    @Override
    public boolean markSupported() {
        return false;
    }

    @Override
    public void mark(int readlimit) {
    }

    @Override
    public void reset() throws IOException {
        throw new IOException("Mark/reset not supported");
    }

    @Override
    public void close() throws IOException {
        IOUtils.close((Closeable[])new Closeable[]{() -> super.close(), this.source});
    }

    private int decrypt(PrefixInputStream packetInputStream) throws IOException {
        int ivLength = packetInputStream.readNBytes(this.packetBuffer, 0, 12);
        if (ivLength != 12) {
            throw new IOException("Packet heading IV error. Unexpected length [" + ivLength + "].");
        }
        long packetIvCounter = ByteUtils.readLongLE((byte[])this.packetBuffer, (int)4);
        if (packetIvCounter != this.counter) {
            throw new IOException("Packet counter mismatch. Expecting [" + this.counter + "], but got [" + packetIvCounter + "].");
        }
        ++this.counter;
        if (this.counter == Long.MIN_VALUE) {
            throw new IOException("Maximum packet count limit exceeded");
        }
        Cipher packetCipher = this.getPacketDecryptionCipher(this.packetBuffer);
        int packetLen = packetInputStream.readNBytes(this.packetBuffer, 0, this.packetBuffer.length);
        if (packetLen < 16) {
            throw new IOException("Encrypted packet is too short");
        }
        try {
            return packetCipher.doFinal(this.packetBuffer, 0, packetLen, this.packetBuffer);
        }
        catch (BadPaddingException | IllegalBlockSizeException | ShortBufferException e) {
            throw new IOException("Exception during packet decryption", e);
        }
    }

    private Cipher getPacketDecryptionCipher(byte[] packet) throws IOException {
        GCMParameterSpec gcmParameterSpec = new GCMParameterSpec(128, packet, 0, 12);
        try {
            Cipher packetCipher = Cipher.getInstance("AES/GCM/NoPadding");
            packetCipher.init(2, (Key)this.secretKey, gcmParameterSpec);
            return packetCipher;
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new IOException("Exception during packet cipher initialisation", e);
        }
    }
}

