/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.license;

import java.util.Collection;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.support.ContextPreservingActionListener;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.metadata.ClusterNameExpressionResolver;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.license.License;
import org.elasticsearch.license.LicensedFeature;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.protocol.xpack.XPackInfoRequest;
import org.elasticsearch.protocol.xpack.XPackInfoResponse;
import org.elasticsearch.protocol.xpack.license.LicenseStatus;
import org.elasticsearch.xpack.core.action.XPackInfoAction;

public final class RemoteClusterLicenseChecker {
    private static final ClusterNameExpressionResolver clusterNameExpressionResolver = new ClusterNameExpressionResolver();
    private final Client client;
    private final LicensedFeature feature;

    public RemoteClusterLicenseChecker(Client client, @Nullable LicensedFeature feature) {
        this.client = client;
        this.feature = feature;
    }

    public void checkRemoteClusterLicenses(List<String> clusterAliases, final ActionListener<LicenseCheck> listener) {
        final Iterator<String> clusterAliasesIterator = clusterAliases.iterator();
        if (!clusterAliasesIterator.hasNext()) {
            listener.onResponse((Object)LicenseCheck.success());
            return;
        }
        final AtomicReference<String> clusterAlias = new AtomicReference<String>();
        ActionListener<XPackInfoResponse> infoListener = new ActionListener<XPackInfoResponse>(){

            public void onResponse(XPackInfoResponse xPackInfoResponse) {
                XPackInfoResponse.LicenseInfo licenseInfo = xPackInfoResponse.getLicenseInfo();
                if (licenseInfo == null) {
                    listener.onFailure((Exception)new ResourceNotFoundException("license info is missing for cluster [" + (String)clusterAlias.get() + "]", new Object[0]));
                    return;
                }
                if (!RemoteClusterLicenseChecker.isActive(RemoteClusterLicenseChecker.this.feature, licenseInfo) || !RemoteClusterLicenseChecker.isAllowed(RemoteClusterLicenseChecker.this.feature, licenseInfo)) {
                    listener.onResponse((Object)LicenseCheck.failure(new RemoteClusterLicenseInfo((String)clusterAlias.get(), licenseInfo)));
                    return;
                }
                if (clusterAliasesIterator.hasNext()) {
                    clusterAlias.set((String)clusterAliasesIterator.next());
                    RemoteClusterLicenseChecker.this.remoteClusterLicense((String)clusterAlias.get(), this);
                } else {
                    listener.onResponse((Object)LicenseCheck.success());
                }
            }

            public void onFailure(Exception e) {
                String message = "could not determine the license type for cluster [" + (String)clusterAlias.get() + "]";
                listener.onFailure((Exception)new ElasticsearchException(message, (Throwable)e, new Object[0]));
            }
        };
        clusterAlias.set(clusterAliasesIterator.next());
        this.remoteClusterLicense((String)clusterAlias.get(), infoListener);
    }

    private static boolean isActive(LicensedFeature feature, XPackInfoResponse.LicenseInfo licenseInfo) {
        return feature != null && !feature.isNeedsActive() || licenseInfo.getStatus() == LicenseStatus.ACTIVE;
    }

    private static boolean isAllowed(LicensedFeature feature, XPackInfoResponse.LicenseInfo licenseInfo) {
        License.OperationMode mode = License.OperationMode.parse(licenseInfo.getMode());
        return feature == null || XPackLicenseState.isAllowedByOperationMode(mode, feature.getMinimumOperationMode());
    }

    private void remoteClusterLicense(String clusterAlias, ActionListener<XPackInfoResponse> listener) {
        ThreadContext threadContext = this.client.threadPool().getThreadContext();
        ContextPreservingActionListener contextPreservingActionListener = new ContextPreservingActionListener(threadContext.newRestorableContext(false), listener);
        try (ThreadContext.StoredContext ignore = threadContext.stashContext();){
            threadContext.markAsSystemContext();
            XPackInfoRequest request = new XPackInfoRequest();
            request.setCategories(EnumSet.of(XPackInfoRequest.Category.LICENSE));
            try {
                this.client.getRemoteClusterClient(clusterAlias).execute((ActionType)XPackInfoAction.INSTANCE, (ActionRequest)request, (ActionListener)contextPreservingActionListener);
            }
            catch (Exception e) {
                contextPreservingActionListener.onFailure(e);
            }
        }
    }

    public static boolean isRemoteIndex(String index) {
        return index.indexOf(58) != -1;
    }

    public static boolean containsRemoteIndex(Collection<String> indices) {
        return indices.stream().anyMatch(RemoteClusterLicenseChecker::isRemoteIndex);
    }

    public static List<String> remoteIndices(Collection<String> indices) {
        return indices.stream().filter(RemoteClusterLicenseChecker::isRemoteIndex).collect(Collectors.toList());
    }

    public static List<String> remoteClusterAliases(Set<String> remoteClusters, List<String> indices) {
        return indices.stream().filter(RemoteClusterLicenseChecker::isRemoteIndex).map(index -> index.substring(0, index.indexOf(58))).distinct().flatMap(clusterExpression -> clusterNameExpressionResolver.resolveClusterNames(remoteClusters, clusterExpression).stream()).distinct().collect(Collectors.toList());
    }

    public static String buildErrorMessage(LicensedFeature feature, RemoteClusterLicenseInfo remoteClusterLicenseInfo) {
        StringBuilder error = new StringBuilder();
        if (!RemoteClusterLicenseChecker.isActive(feature, remoteClusterLicenseInfo.licenseInfo())) {
            error.append(String.format(Locale.ROOT, "the license on cluster [%s] is not active", remoteClusterLicenseInfo.clusterAlias()));
        } else {
            assert (!RemoteClusterLicenseChecker.isAllowed(feature, remoteClusterLicenseInfo.licenseInfo())) : "license must be incompatible to build error message";
            String message = String.format(Locale.ROOT, "the license mode [%s] on cluster [%s] does not enable [%s]", new Object[]{License.OperationMode.parse(remoteClusterLicenseInfo.licenseInfo().getMode()), remoteClusterLicenseInfo.clusterAlias(), feature.getName()});
            error.append(message);
        }
        return error.toString();
    }

    public static final class LicenseCheck {
        private final RemoteClusterLicenseInfo remoteClusterLicenseInfo;
        private static final LicenseCheck SUCCESS = new LicenseCheck(null);

        public RemoteClusterLicenseInfo remoteClusterLicenseInfo() {
            assert (!this.isSuccess());
            return this.remoteClusterLicenseInfo;
        }

        public static LicenseCheck success() {
            return SUCCESS;
        }

        public boolean isSuccess() {
            return this == SUCCESS;
        }

        public static LicenseCheck failure(RemoteClusterLicenseInfo remoteClusterLicenseInfo) {
            return new LicenseCheck(remoteClusterLicenseInfo);
        }

        private LicenseCheck(RemoteClusterLicenseInfo remoteClusterLicenseInfo) {
            this.remoteClusterLicenseInfo = remoteClusterLicenseInfo;
        }
    }

    public static final class RemoteClusterLicenseInfo {
        private final String clusterAlias;
        private final XPackInfoResponse.LicenseInfo licenseInfo;

        public String clusterAlias() {
            return this.clusterAlias;
        }

        public XPackInfoResponse.LicenseInfo licenseInfo() {
            return this.licenseInfo;
        }

        RemoteClusterLicenseInfo(String clusterAlias, XPackInfoResponse.LicenseInfo licenseInfo) {
            this.clusterAlias = clusterAlias;
            this.licenseInfo = licenseInfo;
        }
    }
}

