/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.X509ExtendedKeyManager;
import org.elasticsearch.common.ssl.KeyStoreUtil;
import org.elasticsearch.common.ssl.PemTrustConfig;
import org.elasticsearch.common.ssl.PemUtils;
import org.elasticsearch.common.ssl.SslConfigException;
import org.elasticsearch.common.ssl.SslFileUtil;
import org.elasticsearch.common.ssl.SslKeyConfig;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.StoredCertificate;
import org.elasticsearch.core.Tuple;

public final class PemKeyConfig
implements SslKeyConfig {
    private static final String KEY_FILE_TYPE = "PEM private key";
    private static final String CERT_FILE_TYPE = "PEM certificate";
    private final String certificate;
    private final String key;
    private final char[] keyPassword;
    private final Path configBasePath;

    public PemKeyConfig(String certificatePath, String keyPath, char[] keyPassword, Path configBasePath) {
        this.certificate = Objects.requireNonNull(certificatePath, "Certificate path cannot be null");
        this.key = Objects.requireNonNull(keyPath, "Key path cannot be null");
        this.keyPassword = Objects.requireNonNull(keyPassword, "Key password cannot be null (but may be empty)");
        this.configBasePath = Objects.requireNonNull(configBasePath, "Config base path cannot be null");
    }

    @Override
    public boolean hasKeyMaterial() {
        return true;
    }

    @Override
    public Collection<Path> getDependentFiles() {
        return Arrays.asList(this.resolve(this.certificate), this.resolve(this.key));
    }

    private Path resolve(String fileName) {
        return this.configBasePath.resolve(fileName);
    }

    @Override
    public Collection<StoredCertificate> getConfiguredCertificates() {
        List<Certificate> certificates = this.getCertificates(this.resolve(this.certificate));
        ArrayList<StoredCertificate> info = new ArrayList<StoredCertificate>(certificates.size());
        boolean first = true;
        for (Certificate cert : certificates) {
            if (cert instanceof X509Certificate) {
                info.add(new StoredCertificate((X509Certificate)cert, this.certificate, "PEM", null, first));
            }
            first = false;
        }
        return info;
    }

    @Override
    public X509ExtendedKeyManager createKeyManager() {
        Path keyPath = this.resolve(this.key);
        PrivateKey privateKey = this.getPrivateKey(keyPath);
        Path certPath = this.resolve(this.certificate);
        List<Certificate> certificates = this.getCertificates(certPath);
        try {
            KeyStore keyStore = KeyStoreUtil.buildKeyStore(certificates, privateKey, this.keyPassword);
            return KeyStoreUtil.createKeyManager(keyStore, this.keyPassword, KeyManagerFactory.getDefaultAlgorithm());
        }
        catch (GeneralSecurityException e) {
            throw new SslConfigException("failed to load a KeyManager for certificate/key pair [" + certPath + "], [" + keyPath + "]", e);
        }
    }

    @Override
    public List<Tuple<PrivateKey, X509Certificate>> getKeys() {
        Path keyPath = this.resolve(this.key);
        Path certPath = this.resolve(this.certificate);
        List<Certificate> certificates = this.getCertificates(certPath);
        if (certificates.isEmpty()) {
            return List.of();
        }
        Certificate leafCertificate = certificates.get(0);
        if (leafCertificate instanceof X509Certificate) {
            return List.of(Tuple.tuple((Object)this.getPrivateKey(keyPath), (Object)((X509Certificate)leafCertificate)));
        }
        return List.of();
    }

    @Override
    public SslTrustConfig asTrustConfig() {
        return new PemTrustConfig(List.of(this.certificate), this.configBasePath);
    }

    private PrivateKey getPrivateKey(Path path) {
        try {
            PrivateKey privateKey = PemUtils.parsePrivateKey(path, () -> this.keyPassword);
            if (privateKey == null) {
                throw new SslConfigException("could not load ssl private key file [" + path + "]");
            }
            return privateKey;
        }
        catch (AccessControlException e) {
            throw SslFileUtil.accessControlFailure(KEY_FILE_TYPE, List.of(path), e, this.configBasePath);
        }
        catch (IOException e) {
            throw SslFileUtil.ioException(KEY_FILE_TYPE, List.of(path), e);
        }
        catch (GeneralSecurityException e) {
            throw SslFileUtil.securityException(KEY_FILE_TYPE, List.of(path), e);
        }
    }

    private List<Certificate> getCertificates(Path path) {
        try {
            return PemUtils.readCertificates(Collections.singleton(path));
        }
        catch (AccessControlException e) {
            throw SslFileUtil.accessControlFailure(CERT_FILE_TYPE, List.of(path), e, this.configBasePath);
        }
        catch (IOException e) {
            throw SslFileUtil.ioException(CERT_FILE_TYPE, List.of(path), e);
        }
        catch (GeneralSecurityException e) {
            throw SslFileUtil.securityException(CERT_FILE_TYPE, List.of(path), e);
        }
    }

    public String toString() {
        return "PEM-key-config{cert=" + this.certificate + " key=" + this.key + "}";
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PemKeyConfig that = (PemKeyConfig)o;
        return Objects.equals(this.certificate, that.certificate) && Objects.equals(this.key, that.key) && Arrays.equals(this.keyPassword, that.keyPassword);
    }

    public int hashCode() {
        int result = Objects.hash(this.certificate, this.key);
        result = 31 * result + Arrays.hashCode(this.keyPassword);
        return result;
    }
}

