/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.searchablesnapshots.cache.full;

import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.IntPredicate;
import java.util.function.Predicate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.core.KeywordAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexDeletionPolicy;
import org.apache.lucene.index.IndexNotFoundException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.index.KeepOnlyLastCommitDeletionPolicy;
import org.apache.lucene.index.LeafReader;
import org.apache.lucene.index.LeafReaderContext;
import org.apache.lucene.index.MergeScheduler;
import org.apache.lucene.index.SerialMergeScheduler;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreMode;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.Weight;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.io.stream.ByteBufferStreamInput;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardPath;
import org.elasticsearch.snapshots.SnapshotId;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.ByteRange;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.CacheFile;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.CacheKey;
import org.elasticsearch.xpack.searchablesnapshots.cache.full.CacheService;

public class PersistentCache
implements Closeable {
    private static final Logger logger = LogManager.getLogger(PersistentCache.class);
    private static final String NODE_VERSION_COMMIT_KEY = "node_version";
    private final NodeEnvironment nodeEnvironment;
    private final Map<String, Document> documents;
    private final List<CacheIndexWriter> writers;
    private final AtomicBoolean started;
    private final AtomicBoolean closed;
    private static final String CACHE_ID_FIELD = "cache_id";
    private static final String CACHE_PATH_FIELD = "cache_path";
    private static final String CACHE_RANGES_FIELD = "cache_ranges";
    private static final String SNAPSHOT_ID_FIELD = "snapshot_id";
    private static final String SNAPSHOT_INDEX_NAME_FIELD = "index_name";
    private static final String SHARD_INDEX_NAME_FIELD = "shard_index_name";
    private static final String SHARD_INDEX_ID_FIELD = "shard_index_id";
    private static final String SHARD_ID_FIELD = "shard_id";
    private static final String FILE_NAME_FIELD = "file_name";
    private static final String FILE_LENGTH_FIELD = "file_length";

    public PersistentCache(NodeEnvironment nodeEnvironment) {
        this.documents = Collections.synchronizedMap(PersistentCache.loadDocuments(nodeEnvironment));
        this.writers = PersistentCache.createWriters(nodeEnvironment);
        this.nodeEnvironment = nodeEnvironment;
        this.started = new AtomicBoolean();
        this.closed = new AtomicBoolean();
    }

    private void ensureOpen() {
        if (this.closed.get()) {
            throw new AlreadyClosedException("Persistent cache is already closed");
        }
    }

    private void ensureStarted() {
        if (!this.started.get()) {
            throw new IllegalStateException("Persistent cache is not started");
        }
    }

    private CacheIndexWriter getWriter(CacheFile cacheFile) {
        this.ensureOpen();
        if (this.writers.size() == 1) {
            return this.writers.get(0);
        }
        Path path = cacheFile.getFile().toAbsolutePath();
        return this.writers.stream().filter(writer -> path.startsWith(writer.nodePath().path)).findFirst().orElseThrow(() -> new PersistentCacheIndexNotFoundException(this.nodeEnvironment, cacheFile));
    }

    public void addCacheFile(CacheFile cacheFile, SortedSet<ByteRange> ranges) throws IOException {
        this.ensureStarted();
        this.getWriter(cacheFile).updateCacheFile(cacheFile, ranges);
    }

    public void removeCacheFile(CacheFile cacheFile) throws IOException {
        this.ensureStarted();
        this.getWriter(cacheFile).deleteCacheFile(cacheFile);
    }

    public long getCacheSize(ShardId shardId, SnapshotId snapshotId) {
        return this.getCacheSize(shardId, snapshotId, x$0 -> Files.exists(x$0, new LinkOption[0]));
    }

    long getCacheSize(ShardId shardId, SnapshotId snapshotId, Predicate<Path> predicate) {
        long aggregateSize = 0L;
        for (CacheIndexWriter writer : this.writers) {
            Path snapshotCacheDir = CacheService.resolveSnapshotCache(writer.nodePath().resolve(shardId)).resolve(snapshotId.getUUID());
            if (!Files.exists(snapshotCacheDir, new LinkOption[0])) continue;
            try (DirectoryReader indexReader = DirectoryReader.open((IndexWriter)writer.indexWriter);){
                IndexSearcher searcher = new IndexSearcher((IndexReader)indexReader);
                searcher.setQueryCache(null);
                Weight weight = searcher.createWeight((Query)new BooleanQuery.Builder().add((Query)new TermQuery(new Term(SNAPSHOT_ID_FIELD, snapshotId.getUUID())), BooleanClause.Occur.MUST).add((Query)new TermQuery(new Term(SHARD_INDEX_ID_FIELD, shardId.getIndex().getUUID())), BooleanClause.Occur.MUST).add((Query)new TermQuery(new Term(SHARD_ID_FIELD, String.valueOf(shardId.getId()))), BooleanClause.Occur.MUST).build(), ScoreMode.COMPLETE_NO_SCORES, 0.0f);
                for (LeafReaderContext leafReaderContext : searcher.getIndexReader().leaves()) {
                    Scorer scorer = weight.scorer(leafReaderContext);
                    if (scorer == null) continue;
                    Bits liveDocs = leafReaderContext.reader().getLiveDocs();
                    IntPredicate isLiveDoc = liveDocs == null ? i -> true : arg_0 -> ((Bits)liveDocs).get(arg_0);
                    DocIdSetIterator docIdSetIterator = scorer.iterator();
                    while (docIdSetIterator.nextDoc() != Integer.MAX_VALUE) {
                        Document document;
                        String cacheFileId;
                        if (!isLiveDoc.test(docIdSetIterator.docID()) || !predicate.test(snapshotCacheDir.resolve(cacheFileId = PersistentCache.getValue(document = leafReaderContext.reader().document(docIdSetIterator.docID()), CACHE_ID_FIELD)))) continue;
                        long size = PersistentCache.buildCacheFileRanges(document).stream().mapToLong(ByteRange::length).sum();
                        logger.trace("cache file [{}] has size [{}]", (Object)PersistentCache.getValue(document, CACHE_ID_FIELD), (Object)size);
                        aggregateSize += size;
                    }
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            if (aggregateSize <= 0L) continue;
            return aggregateSize;
        }
        return 0L;
    }

    void repopulateCache(final CacheService cacheService) {
        this.ensureOpen();
        if (this.started.compareAndSet(false, true)) {
            try {
                for (final CacheIndexWriter writer : this.writers) {
                    NodeEnvironment.NodePath nodePath = writer.nodePath();
                    logger.debug("loading persistent cache on data path [{}]", (Object)nodePath);
                    for (String indexUUID : this.nodeEnvironment.availableIndexFoldersForPath(nodePath)) {
                        for (ShardId shardId : this.nodeEnvironment.findAllShardIds(new Index("_unknown_", indexUUID))) {
                            Path shardDataPath = writer.nodePath().resolve(shardId);
                            Path shardCachePath = CacheService.getShardCachePath(new ShardPath(false, shardDataPath, shardDataPath, shardId));
                            if (!Files.isDirectory(shardCachePath, new LinkOption[0])) continue;
                            logger.trace("found snapshot cache dir at [{}], loading cache files from disk and index", (Object)shardCachePath);
                            Files.walkFileTree(shardCachePath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                                @Override
                                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                                    try {
                                        String id = PersistentCache.buildId(file);
                                        Document cacheDocument = PersistentCache.this.documents.get(id);
                                        if (cacheDocument != null) {
                                            logger.trace("indexing cache file with id [{}] in persistent cache index", (Object)id);
                                            writer.updateCacheFile(id, cacheDocument);
                                            CacheKey cacheKey = PersistentCache.buildCacheKey(cacheDocument);
                                            long fileLength = PersistentCache.getFileLength(cacheDocument);
                                            SortedSet<ByteRange> ranges = PersistentCache.buildCacheFileRanges(cacheDocument);
                                            logger.trace("adding cache file with [id={}, key={}, ranges={}]", (Object)id, (Object)cacheKey, ranges);
                                            cacheService.put(cacheKey, fileLength, file.getParent(), id, ranges);
                                        } else {
                                            logger.trace("deleting cache file [{}] (does not exist in persistent cache index)", (Object)file);
                                            Files.delete(file);
                                        }
                                    }
                                    catch (Exception e) {
                                        throw ExceptionsHelper.convertToRuntime((Exception)e);
                                    }
                                    return FileVisitResult.CONTINUE;
                                }
                            });
                        }
                    }
                }
                for (final CacheIndexWriter writer : this.writers) {
                    writer.commit();
                }
                logger.info("persistent cache index loaded");
                this.documents.clear();
            }
            catch (IOException e) {
                try {
                    this.close();
                }
                catch (Exception e2) {
                    logger.warn("failed to close persistent cache index", (Throwable)e2);
                    e.addSuppressed(e2);
                }
                throw new UncheckedIOException("Failed to load persistent cache", e);
            }
            finally {
                this.closeIfAnyIndexWriterHasTragedyOrIsClosed();
            }
        }
        assert (false) : "persistent cache is already loaded";
    }

    void commit() throws IOException {
        this.ensureOpen();
        try {
            for (CacheIndexWriter writer : this.writers) {
                writer.commit();
            }
        }
        catch (IOException e) {
            try {
                this.close();
            }
            catch (Exception e2) {
                logger.warn("failed to close persistent cache index writer", (Throwable)e2);
                e.addSuppressed(e2);
            }
            throw e;
        }
        finally {
            this.closeIfAnyIndexWriterHasTragedyOrIsClosed();
        }
    }

    private void closeIfAnyIndexWriterHasTragedyOrIsClosed() {
        if (this.writers.stream().map(writer -> writer.indexWriter).anyMatch(iw -> iw.getTragicException() != null || !iw.isOpen())) {
            try {
                this.close();
            }
            catch (Exception e) {
                logger.warn("failed to close persistent cache index", (Throwable)e);
            }
        }
    }

    public long getNumDocs() {
        this.ensureOpen();
        long count = 0L;
        for (CacheIndexWriter writer : this.writers) {
            count += writer.indexWriter.getPendingNumDocs();
        }
        return count;
    }

    @Override
    public void close() throws IOException {
        if (this.closed.compareAndSet(false, true)) {
            try {
                IOUtils.close(this.writers);
            }
            finally {
                this.documents.clear();
            }
        }
    }

    private static List<CacheIndexWriter> createWriters(NodeEnvironment nodeEnvironment) {
        ArrayList<CacheIndexWriter> writers = new ArrayList<CacheIndexWriter>();
        boolean success = false;
        try {
            NodeEnvironment.NodePath[] nodePaths;
            for (NodeEnvironment.NodePath nodePath : nodePaths = nodeEnvironment.nodePaths()) {
                writers.add(PersistentCache.createCacheIndexWriter(nodePath));
            }
            success = true;
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to create persistent cache writers", e);
        }
        finally {
            if (!success) {
                IOUtils.closeWhileHandlingException(writers);
            }
        }
        return Collections.unmodifiableList(writers);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static CacheIndexWriter createCacheIndexWriter(NodeEnvironment.NodePath nodePath) throws IOException {
        ArrayList<Object> closeables = new ArrayList<Object>();
        boolean success = false;
        try {
            Path directoryPath = PersistentCache.createCacheIndexFolder(nodePath);
            FSDirectory directory = FSDirectory.open((Path)directoryPath);
            closeables.add(directory);
            IndexWriterConfig config = new IndexWriterConfig((Analyzer)new KeywordAnalyzer());
            config.setIndexDeletionPolicy((IndexDeletionPolicy)new KeepOnlyLastCommitDeletionPolicy());
            config.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
            config.setMergeScheduler((MergeScheduler)new SerialMergeScheduler());
            config.setRAMBufferSizeMB(1.0);
            config.setCommitOnClose(false);
            IndexWriter indexWriter = new IndexWriter((Directory)directory, config);
            closeables.add(indexWriter);
            CacheIndexWriter cacheIndexWriter = new CacheIndexWriter(nodePath, (Directory)directory, indexWriter);
            success = true;
            CacheIndexWriter cacheIndexWriter2 = cacheIndexWriter;
            return cacheIndexWriter2;
        }
        finally {
            if (!success) {
                IOUtils.close(closeables);
            }
        }
    }

    static Map<String, Document> loadDocuments(NodeEnvironment nodeEnvironment) {
        HashMap<String, Document> documents = new HashMap<String, Document>();
        try {
            for (NodeEnvironment.NodePath nodePath : nodeEnvironment.nodePaths()) {
                Path directoryPath = PersistentCache.resolveCacheIndexFolder(nodePath);
                if (!Files.exists(directoryPath, new LinkOption[0])) continue;
                documents.putAll(PersistentCache.loadDocuments(directoryPath));
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to load existing documents from persistent cache index", e);
        }
        return documents;
    }

    static Map<String, Document> loadDocuments(Path directoryPath) throws IOException {
        HashMap<String, Document> documents = new HashMap<String, Document>();
        try (FSDirectory directory = FSDirectory.open((Path)directoryPath);){
            try (DirectoryReader indexReader = DirectoryReader.open((Directory)directory);){
                logger.trace("loading documents from persistent cache index [{}]", (Object)directoryPath);
                for (LeafReaderContext leafReaderContext : indexReader.leaves()) {
                    LeafReader leafReader = leafReaderContext.reader();
                    Bits liveDocs = leafReader.getLiveDocs();
                    for (int i = 0; i < leafReader.maxDoc(); ++i) {
                        if (liveDocs != null && !liveDocs.get(i)) continue;
                        Document document = leafReader.document(i);
                        logger.trace("loading document [{}]", (Object)document);
                        documents.put(PersistentCache.getValue(document, CACHE_ID_FIELD), document);
                    }
                }
            }
            catch (IndexNotFoundException e) {
                logger.debug("persistent cache index does not exist yet", (Throwable)e);
            }
        }
        return documents;
    }

    public static void cleanUp(Settings settings, NodeEnvironment nodeEnvironment) {
        boolean isDataNode = DiscoveryNode.canContainData((Settings)settings);
        if (isDataNode) {
            assert (false) : "should not be called on data nodes";
            throw new IllegalStateException("Cannot clean searchable snapshot caches: node is a data node");
        }
        try {
            for (NodeEnvironment.NodePath nodePath : nodeEnvironment.nodePaths()) {
                for (String indexUUID : nodeEnvironment.availableIndexFoldersForPath(nodePath)) {
                    for (ShardId shardId : nodeEnvironment.findAllShardIds(new Index("_unknown_", indexUUID))) {
                        Path shardDataPath = nodePath.resolve(shardId);
                        ShardPath shardPath = new ShardPath(false, shardDataPath, shardDataPath, shardId);
                        Path cacheDir = CacheService.getShardCachePath(shardPath);
                        if (!Files.isDirectory(cacheDir, new LinkOption[0])) continue;
                        logger.debug("deleting searchable snapshot shard cache directory [{}]", (Object)cacheDir);
                        IOUtils.rm((Path[])new Path[]{cacheDir});
                    }
                }
                Path cacheIndexDir = PersistentCache.resolveCacheIndexFolder(nodePath);
                if (!Files.isDirectory(cacheIndexDir, new LinkOption[0])) continue;
                logger.debug("deleting searchable snapshot lucene directory [{}]", (Object)cacheIndexDir);
                IOUtils.rm((Path[])new Path[]{cacheIndexDir});
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to clean up searchable snapshots cache", e);
        }
    }

    private static String buildId(CacheFile cacheFile) {
        return PersistentCache.buildId(cacheFile.getFile());
    }

    private static String buildId(Path path) {
        return path.getFileName().toString();
    }

    private static Term buildTerm(String cacheFileUuid) {
        return new Term(CACHE_ID_FIELD, cacheFileUuid);
    }

    private static Document buildDocument(NodeEnvironment.NodePath nodePath, CacheFile cacheFile, SortedSet<ByteRange> cacheRanges) throws IOException {
        Document document = new Document();
        document.add((IndexableField)new StringField(CACHE_ID_FIELD, PersistentCache.buildId(cacheFile), Field.Store.YES));
        document.add((IndexableField)new StringField(CACHE_PATH_FIELD, nodePath.indicesPath.relativize(cacheFile.getFile()).toString(), Field.Store.YES));
        try (BytesStreamOutput output = new BytesStreamOutput();){
            output.writeVInt(cacheRanges.size());
            for (ByteRange cacheRange : cacheRanges) {
                output.writeVLong(cacheRange.start());
                output.writeVLong(cacheRange.end());
            }
            output.flush();
            document.add((IndexableField)new StoredField(CACHE_RANGES_FIELD, output.bytes().toBytesRef()));
        }
        CacheKey cacheKey = cacheFile.getCacheKey();
        document.add((IndexableField)new StringField(FILE_NAME_FIELD, cacheKey.getFileName(), Field.Store.YES));
        document.add((IndexableField)new StringField(FILE_LENGTH_FIELD, Long.toString(cacheFile.getLength()), Field.Store.YES));
        document.add((IndexableField)new StringField(SNAPSHOT_ID_FIELD, cacheKey.getSnapshotUUID(), Field.Store.YES));
        document.add((IndexableField)new StringField(SNAPSHOT_INDEX_NAME_FIELD, cacheKey.getSnapshotIndexName(), Field.Store.YES));
        ShardId shardId = cacheKey.getShardId();
        document.add((IndexableField)new StringField(SHARD_INDEX_NAME_FIELD, shardId.getIndex().getName(), Field.Store.YES));
        document.add((IndexableField)new StringField(SHARD_INDEX_ID_FIELD, shardId.getIndex().getUUID(), Field.Store.YES));
        document.add((IndexableField)new StringField(SHARD_ID_FIELD, Integer.toString(shardId.getId()), Field.Store.YES));
        return document;
    }

    private static String getValue(Document document, String fieldName) {
        String value = document.get(fieldName);
        assert (value != null) : "no value found for field [" + fieldName + "] and document [" + document + "]";
        return value;
    }

    private static CacheKey buildCacheKey(Document document) {
        return new CacheKey(PersistentCache.getValue(document, SNAPSHOT_ID_FIELD), PersistentCache.getValue(document, SNAPSHOT_INDEX_NAME_FIELD), new ShardId(new Index(PersistentCache.getValue(document, SHARD_INDEX_NAME_FIELD), PersistentCache.getValue(document, SHARD_INDEX_ID_FIELD)), Integer.parseInt(PersistentCache.getValue(document, SHARD_ID_FIELD))), PersistentCache.getValue(document, FILE_NAME_FIELD));
    }

    private static long getFileLength(Document document) {
        String fileLength = PersistentCache.getValue(document, FILE_LENGTH_FIELD);
        assert (fileLength != null);
        return Long.parseLong(fileLength);
    }

    private static SortedSet<ByteRange> buildCacheFileRanges(Document document) throws IOException {
        BytesRef cacheRangesBytesRef = document.getBinaryValue(CACHE_RANGES_FIELD);
        assert (cacheRangesBytesRef != null);
        TreeSet<ByteRange> cacheRanges = new TreeSet<ByteRange>();
        try (ByteBufferStreamInput input = new ByteBufferStreamInput(ByteBuffer.wrap(cacheRangesBytesRef.bytes));){
            int length = input.readVInt();
            assert (length > 0) : "empty cache ranges";
            ByteRange previous = null;
            for (int i = 0; i < length; ++i) {
                ByteRange range = ByteRange.of(input.readVLong(), input.readVLong());
                assert (range.length() > 0L) : range;
                assert (range.end() <= PersistentCache.getFileLength(document));
                assert (previous == null || previous.end() < range.start());
                boolean added = cacheRanges.add(range);
                assert (added) : range + " already exist in " + cacheRanges;
                previous = range;
            }
        }
        return Collections.unmodifiableSortedSet(cacheRanges);
    }

    static Path resolveCacheIndexFolder(NodeEnvironment.NodePath nodePath) {
        return PersistentCache.resolveCacheIndexFolder(nodePath.path);
    }

    static Path resolveCacheIndexFolder(Path dataPath) {
        return CacheService.resolveSnapshotCache(dataPath);
    }

    private static Path createCacheIndexFolder(NodeEnvironment.NodePath nodePath) throws IOException {
        Path snapshotCacheRootDir = PersistentCache.resolveCacheIndexFolder(nodePath);
        if (!Files.exists(snapshotCacheRootDir, new LinkOption[0])) {
            logger.debug("creating new persistent cache index directory [{}]", (Object)snapshotCacheRootDir);
            Files.createDirectories(snapshotCacheRootDir, new FileAttribute[0]);
        }
        return snapshotCacheRootDir;
    }

    static class CacheIndexWriter
    implements Closeable {
        private final NodeEnvironment.NodePath nodePath;
        private final IndexWriter indexWriter;
        private final Directory directory;
        private static final Set<Map.Entry<String, String>> LUCENE_COMMIT_DATA = Collections.singletonMap("node_version", Integer.toString(Version.CURRENT.id)).entrySet();

        private CacheIndexWriter(NodeEnvironment.NodePath nodePath, Directory directory, IndexWriter indexWriter) {
            this.nodePath = nodePath;
            this.directory = directory;
            this.indexWriter = indexWriter;
        }

        NodeEnvironment.NodePath nodePath() {
            return this.nodePath;
        }

        void updateCacheFile(CacheFile cacheFile, SortedSet<ByteRange> cacheRanges) throws IOException {
            this.updateCacheFile(PersistentCache.buildId(cacheFile), PersistentCache.buildDocument(this.nodePath, cacheFile, cacheRanges));
        }

        void updateCacheFile(String cacheFileId, Document cacheFileDocument) throws IOException {
            Term term = PersistentCache.buildTerm(cacheFileId);
            logger.debug("updating document with term [{}]", (Object)term);
            this.indexWriter.updateDocument(term, (Iterable)cacheFileDocument);
        }

        void deleteCacheFile(CacheFile cacheFile) throws IOException {
            this.deleteCacheFile(PersistentCache.buildId(cacheFile));
        }

        void deleteCacheFile(String cacheFileId) throws IOException {
            Term term = PersistentCache.buildTerm(cacheFileId);
            logger.debug("deleting document with term [{}]", (Object)term);
            this.indexWriter.deleteDocuments(new Term[]{term});
        }

        void commit() throws IOException {
            logger.debug("committing");
            this.indexWriter.setLiveCommitData(LUCENE_COMMIT_DATA);
            this.indexWriter.commit();
        }

        @Override
        public void close() throws IOException {
            logger.debug("closing persistent cache index");
            IOUtils.close((Closeable[])new Closeable[]{this.indexWriter, this.directory});
        }

        public String toString() {
            return "[persistent cache index][" + this.nodePath + "]";
        }
    }

    static class PersistentCacheIndexNotFoundException
    extends IllegalArgumentException {
        PersistentCacheIndexNotFoundException(NodeEnvironment nodeEnvironment, CacheFile cacheFile) {
            super("Persistent cache index not found for cache file path [" + cacheFile.getFile() + "] using node paths " + Arrays.toString(nodeEnvironment.nodeDataPaths()));
        }
    }
}

