/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.job;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.xpack.ml.MachineLearning;
import org.elasticsearch.xpack.ml.autoscaling.MlAutoscalingDeciderService;
import org.elasticsearch.xpack.ml.autoscaling.NativeMemoryCapacity;
import org.elasticsearch.xpack.ml.job.NodeLoad;
import org.elasticsearch.xpack.ml.job.NodeLoadDetector;
import org.elasticsearch.xpack.ml.process.MlMemoryTracker;
import org.elasticsearch.xpack.ml.utils.NativeMemoryCalculator;

public class JobNodeSelector {
    public static final PersistentTasksCustomMetadata.Assignment AWAITING_LAZY_ASSIGNMENT = new PersistentTasksCustomMetadata.Assignment(null, "persistent task is awaiting node assignment.");
    private static final Logger logger = LogManager.getLogger(JobNodeSelector.class);
    private final String jobId;
    private final String taskName;
    private final ClusterState clusterState;
    private final Collection<DiscoveryNode> candidateNodes;
    private final MlMemoryTracker memoryTracker;
    private final Function<DiscoveryNode, String> nodeFilter;
    private final NodeLoadDetector nodeLoadDetector;
    private final int maxLazyNodes;

    private static String createReason(String job, String node, String msg, Object ... params) {
        String preamble = String.format(Locale.ROOT, "Not opening job [%s] on node [%s]. Reason: ", job, node);
        return preamble + ParameterizedMessage.format((String)msg, (Object[])params);
    }

    public JobNodeSelector(ClusterState clusterState, Collection<DiscoveryNode> candidateNodes, String jobId, String taskName, MlMemoryTracker memoryTracker, int maxLazyNodes, Function<DiscoveryNode, String> nodeFilter) {
        this.jobId = Objects.requireNonNull(jobId);
        this.taskName = Objects.requireNonNull(taskName);
        this.clusterState = Objects.requireNonNull(clusterState);
        this.candidateNodes = Objects.requireNonNull(candidateNodes);
        this.memoryTracker = Objects.requireNonNull(memoryTracker);
        this.nodeLoadDetector = new NodeLoadDetector(Objects.requireNonNull(memoryTracker));
        this.maxLazyNodes = maxLazyNodes;
        this.nodeFilter = node -> {
            if (MachineLearning.isMlNode(node)) {
                return nodeFilter != null ? (String)nodeFilter.apply((DiscoveryNode)node) : null;
            }
            return JobNodeSelector.createReason(jobId, JobNodeSelector.nodeNameOrId(node), "This node isn't a machine learning node.", new Object[0]);
        };
    }

    public Tuple<NativeMemoryCapacity, Long> perceivedCapacityAndMaxFreeMemory(int maxMachineMemoryPercent, boolean useAutoMemoryPercentage, int maxOpenJobs) {
        List<DiscoveryNode> capableNodes = this.candidateNodes.stream().filter(n -> this.nodeFilter.apply((DiscoveryNode)n) == null).collect(Collectors.toList());
        NativeMemoryCapacity currentCapacityForMl = MlAutoscalingDeciderService.currentScale(capableNodes, maxMachineMemoryPercent, useAutoMemoryPercentage);
        long mostAvailableMemory = capableNodes.stream().map(n -> this.nodeLoadDetector.detectNodeLoad(this.clusterState, (DiscoveryNode)n, maxOpenJobs, maxMachineMemoryPercent, useAutoMemoryPercentage)).filter(nl -> nl.remainingJobs() > 0).mapToLong(NodeLoad::getFreeMemory).max().orElse(0L);
        return Tuple.tuple((Object)currentCapacityForMl, (Object)mostAvailableMemory);
    }

    public PersistentTasksCustomMetadata.Assignment selectNode(int dynamicMaxOpenJobs, int maxConcurrentJobAllocations, int maxMachineMemoryPercent, long maxNodeSize, boolean useAutoMemoryPercentage) {
        Long estimatedMemoryFootprint = this.memoryTracker.getJobMemoryRequirement(this.taskName, this.jobId);
        return this.selectNode(estimatedMemoryFootprint, dynamicMaxOpenJobs, maxConcurrentJobAllocations, maxMachineMemoryPercent, maxNodeSize, useAutoMemoryPercentage);
    }

    public PersistentTasksCustomMetadata.Assignment selectNode(Long estimatedMemoryFootprint, int dynamicMaxOpenJobs, int maxConcurrentJobAllocations, int maxMachineMemoryPercent, long maxNodeSize, boolean useAutoMemoryPercentage) {
        if (estimatedMemoryFootprint == null) {
            this.memoryTracker.asyncRefresh();
            String reason = "Not opening job [" + this.jobId + "] because job memory requirements are stale - refresh requested";
            logger.debug(reason);
            return new PersistentTasksCustomMetadata.Assignment(null, reason);
        }
        TreeMap<String, String> reasons = new TreeMap<String, String>();
        long maxAvailableMemory = Long.MIN_VALUE;
        DiscoveryNode minLoadedNodeByMemory = null;
        long requiredMemoryForJob = estimatedMemoryFootprint;
        for (DiscoveryNode node : this.candidateNodes) {
            long availableMemory;
            String reason = this.nodeFilter.apply(node);
            if (reason != null) {
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            NodeLoad currentLoad = this.nodeLoadDetector.detectNodeLoad(this.clusterState, node, dynamicMaxOpenJobs, maxMachineMemoryPercent, useAutoMemoryPercentage);
            if (currentLoad.getError() != null) {
                reason = JobNodeSelector.createReason(this.jobId, JobNodeSelector.nodeNameAndMlAttributes(node), currentLoad.getError(), new Object[0]);
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            boolean canAllocateByMemory = currentLoad.isUseMemory();
            int maxNumberOfOpenJobs = currentLoad.getMaxJobs();
            if (currentLoad.getNumAllocatingJobs() >= (long)maxConcurrentJobAllocations) {
                reason = JobNodeSelector.createReason(this.jobId, JobNodeSelector.nodeNameAndMlAttributes(node), "Node exceeds [{}] the maximum number of jobs [{}] in opening state.", currentLoad.getNumAllocatingJobs(), maxConcurrentJobAllocations);
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            if (currentLoad.remainingJobs() == 0) {
                reason = JobNodeSelector.createReason(this.jobId, JobNodeSelector.nodeNameAndMlAttributes(node), "This node is full. Number of opened jobs and allocated native inference processes [{}], {} [{}].", currentLoad.getNumAssignedJobs(), MachineLearning.MAX_OPEN_JOBS_PER_NODE.getKey(), maxNumberOfOpenJobs);
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            if (!canAllocateByMemory) {
                reason = JobNodeSelector.createReason(this.jobId, JobNodeSelector.nodeNameAndMlAttributes(node), "This node is not providing accurate information to determine is load by memory.", new Object[0]);
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            if (currentLoad.getMaxMlMemory() <= 0L) {
                reason = JobNodeSelector.createReason(this.jobId, JobNodeSelector.nodeNameAndMlAttributes(node), "This node is indicating that it has no native memory for machine learning.", new Object[0]);
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            if (currentLoad.getNumAssignedJobs() == 0L) {
                requiredMemoryForJob += MachineLearning.NATIVE_EXECUTABLE_CODE_OVERHEAD.getBytes();
            }
            if (requiredMemoryForJob > (availableMemory = currentLoad.getMaxMlMemory() - currentLoad.getAssignedJobMemory())) {
                reason = JobNodeSelector.createReason(this.jobId, JobNodeSelector.nodeNameAndMlAttributes(node), "This node has insufficient available memory. Available memory for ML [{} ({})], memory required by existing jobs [{} ({})], estimated memory required for this job [{} ({})].", currentLoad.getMaxMlMemory(), ByteSizeValue.ofBytes((long)currentLoad.getMaxMlMemory()).toString(), currentLoad.getAssignedJobMemory(), ByteSizeValue.ofBytes((long)currentLoad.getAssignedJobMemory()).toString(), requiredMemoryForJob, ByteSizeValue.ofBytes((long)requiredMemoryForJob).toString());
                logger.trace(reason);
                reasons.put(node.getName(), reason);
                continue;
            }
            if (maxAvailableMemory >= availableMemory) continue;
            maxAvailableMemory = availableMemory;
            minLoadedNodeByMemory = node;
        }
        return this.createAssignment(estimatedMemoryFootprint, minLoadedNodeByMemory, reasons.values(), maxNodeSize > 0L ? NativeMemoryCalculator.allowedBytesForMl(maxNodeSize, maxMachineMemoryPercent, useAutoMemoryPercentage) : Long.MAX_VALUE);
    }

    PersistentTasksCustomMetadata.Assignment createAssignment(long estimatedMemoryUsage, DiscoveryNode minLoadedNode, Collection<String> reasons, long biggestPossibleJob) {
        if (minLoadedNode == null) {
            String explanation = String.join((CharSequence)"|", reasons);
            PersistentTasksCustomMetadata.Assignment currentAssignment = new PersistentTasksCustomMetadata.Assignment(null, explanation);
            logger.debug("no node selected for job [{}], reasons [{}]", (Object)this.jobId, (Object)explanation);
            if (MachineLearning.NATIVE_EXECUTABLE_CODE_OVERHEAD.getBytes() + estimatedMemoryUsage > biggestPossibleJob) {
                ParameterizedMessage message = new ParameterizedMessage("[{}] not waiting for node assignment as estimated job size [{}] is greater than largest possible job size [{}]", new Object[]{this.jobId, MachineLearning.NATIVE_EXECUTABLE_CODE_OVERHEAD.getBytes() + estimatedMemoryUsage, biggestPossibleJob});
                logger.info((Message)message);
                ArrayList<String> newReasons = new ArrayList<String>(reasons);
                newReasons.add(message.getFormattedMessage());
                explanation = String.join((CharSequence)"|", newReasons);
                return new PersistentTasksCustomMetadata.Assignment(null, explanation);
            }
            return this.considerLazyAssignment(currentAssignment);
        }
        logger.debug("selected node [{}] for job [{}]", (Object)minLoadedNode, (Object)this.jobId);
        return new PersistentTasksCustomMetadata.Assignment(minLoadedNode.getId(), "");
    }

    PersistentTasksCustomMetadata.Assignment considerLazyAssignment(PersistentTasksCustomMetadata.Assignment currentAssignment) {
        assert (currentAssignment.getExecutorNode() == null);
        int numMlNodes = 0;
        for (DiscoveryNode node : this.candidateNodes) {
            if (!MachineLearning.isMlNode(node)) continue;
            ++numMlNodes;
        }
        if (numMlNodes < this.maxLazyNodes) {
            return AWAITING_LAZY_ASSIGNMENT;
        }
        return currentAssignment;
    }

    static String nodeNameOrId(DiscoveryNode node) {
        String nodeNameOrID = node.getName();
        if (Strings.isNullOrEmpty((String)nodeNameOrID)) {
            nodeNameOrID = node.getId();
        }
        return nodeNameOrID;
    }

    public static String nodeNameAndVersion(DiscoveryNode node) {
        String nodeNameOrID = JobNodeSelector.nodeNameOrId(node);
        StringBuilder builder = new StringBuilder("{").append(nodeNameOrID).append('}');
        builder.append('{').append("version=").append(node.getVersion()).append('}');
        return builder.toString();
    }

    public static String nodeNameAndMlAttributes(DiscoveryNode node) {
        String nodeNameOrID = JobNodeSelector.nodeNameOrId(node);
        StringBuilder builder = new StringBuilder("{").append(nodeNameOrID).append('}');
        for (Map.Entry entry : node.getAttributes().entrySet()) {
            if (!((String)entry.getKey()).startsWith("ml.") && !((String)entry.getKey()).equals("node.ml")) continue;
            builder.append('{').append(entry).append('}');
        }
        return builder.toString();
    }
}

