/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest.geoip;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.OriginSettingClient;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.ingest.IngestService;
import org.elasticsearch.ingest.geoip.DatabaseReaderLazyLoader;
import org.elasticsearch.ingest.geoip.GeoIpCache;
import org.elasticsearch.ingest.geoip.GeoIpTaskState;
import org.elasticsearch.ingest.geoip.LocalDatabases;
import org.elasticsearch.ingest.geoip.TarInputStream;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.watcher.ResourceWatcherService;

public final class DatabaseRegistry
implements Closeable {
    private static final Logger LOGGER = LogManager.getLogger(DatabaseRegistry.class);
    private final Client client;
    private final GeoIpCache cache;
    private final Path geoipTmpBaseDirectory;
    private Path geoipTmpDirectory;
    private final LocalDatabases localDatabases;
    private final Consumer<Runnable> genericExecutor;
    private final ConcurrentMap<String, DatabaseReaderLazyLoader> databases = new ConcurrentHashMap<String, DatabaseReaderLazyLoader>();

    DatabaseRegistry(Environment environment, Client client, GeoIpCache cache, Consumer<Runnable> genericExecutor) {
        this(environment.tmpFile(), (Client)new OriginSettingClient(client, "ingest"), cache, new LocalDatabases(environment, cache), genericExecutor);
    }

    DatabaseRegistry(Path tmpDir, Client client, GeoIpCache cache, LocalDatabases localDatabases, Consumer<Runnable> genericExecutor) {
        this.client = client;
        this.cache = cache;
        this.geoipTmpBaseDirectory = tmpDir.resolve("geoip-databases");
        this.localDatabases = localDatabases;
        this.genericExecutor = genericExecutor;
    }

    public void initialize(String nodeId, ResourceWatcherService resourceWatcher, IngestService ingestService) throws IOException {
        this.localDatabases.initialize(resourceWatcher);
        this.geoipTmpDirectory = this.geoipTmpBaseDirectory.resolve(nodeId);
        Files.walkFileTree(this.geoipTmpDirectory, (FileVisitor<? super Path>)new FileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                try {
                    LOGGER.info("deleting stale file [{}]", (Object)file);
                    Files.deleteIfExists(file);
                }
                catch (IOException e) {
                    LOGGER.warn("can't delete stale file [" + file + "]", (Throwable)e);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException e) {
                if (!(e instanceof NoSuchFileException)) {
                    LOGGER.warn("can't delete stale file [" + file + "]", (Throwable)e);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                return FileVisitResult.CONTINUE;
            }
        });
        if (!Files.exists(this.geoipTmpDirectory, new LinkOption[0])) {
            Files.createDirectories(this.geoipTmpDirectory, new FileAttribute[0]);
        }
        LOGGER.info("initialized database registry, using geoip-databases directory [{}]", (Object)this.geoipTmpDirectory);
        ingestService.addIngestClusterStateListener(this::checkDatabases);
    }

    public DatabaseReaderLazyLoader getDatabase(String name, boolean fallbackUsingDefaultDatabases) {
        DatabaseReaderLazyLoader instance;
        while ((instance = this.databases.getOrDefault(name, this.localDatabases.getDatabase(name, fallbackUsingDefaultDatabases))) != null && !instance.preLookup()) {
        }
        return instance;
    }

    List<DatabaseReaderLazyLoader> getAllDatabases() {
        ArrayList<DatabaseReaderLazyLoader> all = new ArrayList<DatabaseReaderLazyLoader>(this.localDatabases.getAllDatabases());
        this.databases.forEach((key, value) -> all.add((DatabaseReaderLazyLoader)value));
        return all;
    }

    DatabaseReaderLazyLoader get(String key) {
        return (DatabaseReaderLazyLoader)this.databases.get(key);
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.databases.values());
    }

    void checkDatabases(ClusterState state) {
        DiscoveryNode localNode = state.nodes().getLocalNode();
        if (!localNode.isIngestNode()) {
            return;
        }
        PersistentTasksCustomMetadata persistentTasks = (PersistentTasksCustomMetadata)state.metadata().custom("persistent_tasks");
        if (persistentTasks == null) {
            return;
        }
        IndexRoutingTable databasesIndexRT = state.getRoutingTable().index(".geoip_databases");
        if (databasesIndexRT == null || !databasesIndexRT.allPrimaryShardsActive()) {
            return;
        }
        PersistentTasksCustomMetadata.PersistentTask task = PersistentTasksCustomMetadata.getTaskWithId((ClusterState)state, (String)"geoip-downloader");
        GeoIpTaskState taskState = task == null || task.getState() == null ? GeoIpTaskState.EMPTY : (GeoIpTaskState)task.getState();
        taskState.getDatabases().entrySet().stream().filter(e -> ((GeoIpTaskState.Metadata)e.getValue()).isValid(state.getMetadata().settings())).forEach(e -> {
            String localMd5;
            String name = (String)e.getKey();
            GeoIpTaskState.Metadata metadata = (GeoIpTaskState.Metadata)e.getValue();
            DatabaseReaderLazyLoader reference = (DatabaseReaderLazyLoader)this.databases.get(name);
            String remoteMd5 = metadata.getMd5();
            String string = localMd5 = reference != null ? reference.getMd5() : null;
            if (Objects.equals(localMd5, remoteMd5)) {
                LOGGER.debug("Current reference of [{}] is up to date [{}] with was recorded in CS [{}]", (Object)name, (Object)localMd5, (Object)remoteMd5);
                return;
            }
            try {
                this.retrieveAndUpdateDatabase(name, metadata);
            }
            catch (Exception ex) {
                LOGGER.error(() -> new ParameterizedMessage("attempt to download database [{}] failed", (Object)name), (Throwable)ex);
            }
        });
        ArrayList<String> staleEntries = new ArrayList<String>(this.databases.keySet());
        staleEntries.removeAll(taskState.getDatabases().entrySet().stream().filter(e -> ((GeoIpTaskState.Metadata)e.getValue()).isValid(state.getMetadata().settings())).map(Map.Entry::getKey).collect(Collectors.toSet()));
        this.removeStaleEntries(staleEntries);
    }

    void retrieveAndUpdateDatabase(String databaseName, GeoIpTaskState.Metadata metadata) throws IOException {
        Path databaseTmpGzFile;
        String recordedMd5 = metadata.getMd5();
        try {
            databaseTmpGzFile = Files.createFile(this.geoipTmpDirectory.resolve(databaseName + ".tmp.gz"), new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            LOGGER.debug("database update [{}] already in progress, skipping...", (Object)databaseName);
            return;
        }
        DatabaseReaderLazyLoader lazyLoader = (DatabaseReaderLazyLoader)this.databases.get(databaseName);
        if (lazyLoader != null && recordedMd5.equals(lazyLoader.getMd5())) {
            LOGGER.debug("deleting tmp file because database [{}] has already been updated.", (Object)databaseName);
            Files.delete(databaseTmpGzFile);
            return;
        }
        Path databaseTmpFile = Files.createFile(this.geoipTmpDirectory.resolve(databaseName + ".tmp"), new FileAttribute[0]);
        LOGGER.info("downloading geoip database [{}] to [{}]", (Object)databaseName, (Object)databaseTmpGzFile);
        this.retrieveDatabase(databaseName, recordedMd5, metadata, (CheckedConsumer<byte[], IOException>)((CheckedConsumer)bytes -> Files.write(databaseTmpGzFile, bytes, StandardOpenOption.APPEND)), (CheckedRunnable<Exception>)((CheckedRunnable)() -> {
            LOGGER.debug("decompressing [{}]", (Object)databaseTmpGzFile.getFileName());
            Path databaseFile = this.geoipTmpDirectory.resolve(databaseName);
            try (TarInputStream is = new TarInputStream(new GZIPInputStream((InputStream)new BufferedInputStream(Files.newInputStream(databaseTmpGzFile, new OpenOption[0])), 8192));){
                TarInputStream.TarEntry entry;
                while ((entry = is.getNextEntry()) != null) {
                    if (entry.isNotFile()) continue;
                    String name = entry.getName().substring(entry.getName().lastIndexOf(47) + 1);
                    if (name.startsWith(databaseName)) {
                        Files.copy(is, databaseTmpFile, StandardCopyOption.REPLACE_EXISTING);
                        continue;
                    }
                    Files.copy(is, this.geoipTmpDirectory.resolve(databaseName + "_" + name), StandardCopyOption.REPLACE_EXISTING);
                }
            }
            LOGGER.debug("moving database from [{}] to [{}]", (Object)databaseTmpFile, (Object)databaseFile);
            Files.move(databaseTmpFile, databaseFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            this.updateDatabase(databaseName, recordedMd5, databaseFile);
            Files.delete(databaseTmpGzFile);
        }), failure -> {
            LOGGER.error(() -> new ParameterizedMessage("failed to download database [{}]", (Object)databaseName), (Throwable)failure);
            try {
                Files.deleteIfExists(databaseTmpFile);
                Files.deleteIfExists(databaseTmpGzFile);
            }
            catch (IOException ioe) {
                ioe.addSuppressed((Throwable)failure);
                LOGGER.error("Unable to delete tmp database file after failure", (Throwable)ioe);
            }
        });
    }

    void updateDatabase(String databaseFileName, String recordedMd5, Path file) {
        try {
            LOGGER.debug("starting reload of changed geoip database file [{}]", (Object)file);
            DatabaseReaderLazyLoader loader = new DatabaseReaderLazyLoader(this.cache, file, recordedMd5);
            DatabaseReaderLazyLoader existing = this.databases.put(databaseFileName, loader);
            if (existing != null) {
                existing.close();
            }
            LOGGER.info("successfully reloaded changed geoip database file [{}]", (Object)file);
        }
        catch (Exception e) {
            LOGGER.error(() -> new ParameterizedMessage("failed to update database [{}]", (Object)databaseFileName), (Throwable)e);
        }
    }

    void removeStaleEntries(Collection<String> staleEntries) {
        for (String staleEntry : staleEntries) {
            try {
                LOGGER.info("database [{}] no longer exists, cleaning up...", (Object)staleEntry);
                DatabaseReaderLazyLoader existing = (DatabaseReaderLazyLoader)this.databases.remove(staleEntry);
                assert (existing != null);
                existing.close(true);
            }
            catch (Exception e) {
                LOGGER.error(() -> new ParameterizedMessage("failed to clean database [{}]", (Object)staleEntry), (Throwable)e);
            }
        }
    }

    void retrieveDatabase(String databaseName, String expectedMd5, GeoIpTaskState.Metadata metadata, CheckedConsumer<byte[], IOException> chunkConsumer, CheckedRunnable<Exception> completedHandler, Consumer<Exception> failureHandler) {
        this.genericExecutor.accept(() -> {
            MessageDigest md = MessageDigests.md5();
            int firstChunk = metadata.getFirstChunk();
            int lastChunk = metadata.getLastChunk();
            try {
                for (int chunk = firstChunk; chunk <= lastChunk; ++chunk) {
                    SearchRequest searchRequest = new SearchRequest(new String[]{".geoip_databases"});
                    String id = String.format(Locale.ROOT, "%s_%d_%d", databaseName, chunk, metadata.getLastUpdate());
                    searchRequest.source().query((QueryBuilder)new TermQueryBuilder("_id", id));
                    SearchResponse searchResponse = (SearchResponse)this.client.search(searchRequest).actionGet();
                    SearchHit[] hits = searchResponse.getHits().getHits();
                    if (searchResponse.getHits().getHits().length == 0) {
                        failureHandler.accept((Exception)new ResourceNotFoundException("chunk document with id [" + id + "] not found", new Object[0]));
                        return;
                    }
                    byte[] data = (byte[])hits[0].getSourceAsMap().get("data");
                    md.update(data);
                    chunkConsumer.accept((Object)data);
                }
                String actualMd5 = MessageDigests.toHexString((byte[])md.digest());
                if (Objects.equals(expectedMd5, actualMd5)) {
                    completedHandler.run();
                } else {
                    failureHandler.accept(new RuntimeException("expected md5 hash [" + expectedMd5 + "], but got md5 hash [" + actualMd5 + "]"));
                }
            }
            catch (Exception e) {
                failureHandler.accept(e);
            }
        });
    }

    public Set<String> getAvailableDatabases() {
        return org.elasticsearch.core.Set.copyOf(this.databases.keySet());
    }

    public Set<String> getFilesInTemp() {
        Set<String> set;
        block8: {
            Stream<Path> files = Files.list(this.geoipTmpDirectory);
            try {
                set = files.map(Path::getFileName).map(Path::toString).collect(Collectors.toSet());
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            files.close();
        }
        return set;
    }
}

