/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.ssl;

import java.io.IOException;
import java.nio.file.Path;
import java.security.AccessControlException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.env.Environment;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.ssl.SSLConfiguration;
import org.elasticsearch.xpack.core.ssl.SSLService;

public final class SSLConfigurationReloader {
    private static final Logger logger = LogManager.getLogger(SSLConfigurationReloader.class);
    private final CompletableFuture<SSLService> sslServiceFuture = new CompletableFuture();

    public SSLConfigurationReloader(Environment environment, ResourceWatcherService resourceWatcherService, Collection<SSLConfiguration> sslConfigurations) {
        SSLConfigurationReloader.startWatching(environment, SSLConfigurationReloader.reloadConsumer(this.sslServiceFuture), resourceWatcherService, sslConfigurations);
    }

    SSLConfigurationReloader(Environment environment, Consumer<SSLConfiguration> reloadConsumer, ResourceWatcherService resourceWatcherService, Collection<SSLConfiguration> sslConfigurations) {
        SSLConfigurationReloader.startWatching(environment, reloadConsumer, resourceWatcherService, sslConfigurations);
    }

    public void setSSLService(SSLService sslService) {
        boolean completed = this.sslServiceFuture.complete(sslService);
        if (!completed) {
            throw new IllegalStateException("ssl service future was already completed!");
        }
    }

    private static Consumer<SSLConfiguration> reloadConsumer(CompletableFuture<SSLService> future) {
        return sslConfiguration -> {
            try {
                SSLService sslService = (SSLService)future.get();
                logger.debug("reloading ssl configuration [{}]", sslConfiguration);
                sslService.reloadSSLContext((SSLConfiguration)sslConfiguration);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (ExecutionException e) {
                throw new ElasticsearchException("failed to obtain ssl service", (Throwable)e, new Object[0]);
            }
        };
    }

    private static void startWatching(Environment environment, Consumer<SSLConfiguration> reloadConsumer, ResourceWatcherService resourceWatcherService, Collection<SSLConfiguration> sslConfigurations) {
        HashMap<Path, List> pathToConfigurationsMap = new HashMap<Path, List>();
        for (SSLConfiguration sslConfiguration : sslConfigurations) {
            List<Path> filesToMonitor = sslConfiguration.filesToMonitor(environment);
            for (Path directory : SSLConfigurationReloader.directoriesToMonitor(filesToMonitor)) {
                pathToConfigurationsMap.compute(directory, (path, list) -> {
                    if (list == null) {
                        list = new ArrayList<SSLConfiguration>();
                    }
                    list.add(sslConfiguration);
                    return list;
                });
            }
        }
        pathToConfigurationsMap.forEach((path, configurations) -> {
            ChangeListener changeListener = new ChangeListener(environment, Collections.unmodifiableList(configurations), reloadConsumer);
            FileWatcher fileWatcher = new FileWatcher(path);
            fileWatcher.addListener((Object)changeListener);
            try {
                resourceWatcherService.add((ResourceWatcher)fileWatcher, ResourceWatcherService.Frequency.HIGH);
            }
            catch (IOException | AccessControlException e) {
                logger.error("failed to start watching directory [{}] for ssl configurations [{}] - {}", path, configurations, (Object)e);
            }
        });
    }

    private static Set<Path> directoriesToMonitor(List<Path> filePaths) {
        HashSet<Path> paths = new HashSet<Path>();
        for (Path path : filePaths) {
            paths.add(path.getParent());
        }
        return paths;
    }

    private static class ChangeListener
    implements FileChangesListener {
        private final Environment environment;
        private final List<SSLConfiguration> sslConfigurations;
        private final Consumer<SSLConfiguration> reloadConsumer;

        private ChangeListener(Environment environment, List<SSLConfiguration> sslConfigurations, Consumer<SSLConfiguration> reloadConsumer) {
            this.environment = environment;
            this.sslConfigurations = sslConfigurations;
            this.reloadConsumer = reloadConsumer;
        }

        public void onFileCreated(Path file) {
            this.onFileChanged(file);
        }

        public void onFileDeleted(Path file) {
            this.onFileChanged(file);
        }

        public void onFileChanged(Path file) {
            boolean reloaded = false;
            for (SSLConfiguration sslConfiguration : this.sslConfigurations) {
                if (!sslConfiguration.filesToMonitor(this.environment).contains(file)) continue;
                this.reloadConsumer.accept(sslConfiguration);
                reloaded = true;
            }
            if (reloaded) {
                logger.info("reloaded [{}] and updated ssl contexts using this file", (Object)file);
            }
        }
    }
}

