/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.datafeed.extractor.aggregation;

import java.io.IOException;
import java.io.OutputStream;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.common.xcontent.XContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.MultiBucketsAggregation;
import org.elasticsearch.search.aggregations.bucket.SingleBucketAggregation;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregation;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.metrics.GeoCentroid;
import org.elasticsearch.search.aggregations.metrics.Max;
import org.elasticsearch.search.aggregations.metrics.NumericMetricsAggregation;
import org.elasticsearch.search.aggregations.metrics.Percentile;
import org.elasticsearch.search.aggregations.metrics.Percentiles;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;

class AggregationToJsonProcessor {
    private static final Logger LOGGER = LogManager.getLogger(AggregationToJsonProcessor.class);
    private final String timeField;
    private final Set<String> fields;
    private final boolean includeDocCount;
    private final LinkedHashMap<String, Object> keyValuePairs;
    private long keyValueWrittenCount;
    private final SortedMap<Long, List<Map<String, Object>>> docsByBucketTimestamp;
    private final long startTime;
    private final String compositeAggDateValueSourceName;

    AggregationToJsonProcessor(String timeField, Set<String> fields, boolean includeDocCount, long startTime, @Nullable String compositeAggDateValueSourceName) {
        this.timeField = Objects.requireNonNull(timeField);
        this.fields = Objects.requireNonNull(fields);
        this.includeDocCount = includeDocCount;
        this.keyValuePairs = new LinkedHashMap();
        this.docsByBucketTimestamp = new TreeMap<Long, List<Map<String, Object>>>();
        this.keyValueWrittenCount = 0L;
        this.startTime = startTime;
        this.compositeAggDateValueSourceName = compositeAggDateValueSourceName;
    }

    public void process(Aggregations aggs) throws IOException {
        this.processAggs(0L, aggs.asList());
    }

    private void processAggs(long docCount, List<Aggregation> aggregations) throws IOException {
        if (aggregations.isEmpty()) {
            this.queueDocToWrite(this.keyValuePairs, docCount);
            return;
        }
        ArrayList<Aggregation> leafAggregations = new ArrayList<Aggregation>();
        ArrayList<MultiBucketsAggregation> bucketAggregations = new ArrayList<MultiBucketsAggregation>();
        ArrayList<Object> singleBucketAggregations = new ArrayList<Object>();
        for (Aggregation agg : aggregations) {
            if (agg instanceof MultiBucketsAggregation) {
                bucketAggregations.add((MultiBucketsAggregation)agg);
                continue;
            }
            if (agg instanceof SingleBucketAggregation) {
                SingleBucketAggregation singleBucketAggregation = (SingleBucketAggregation)agg;
                for (Aggregation aggregation2 : singleBucketAggregation.getAggregations()) {
                    if (aggregation2 instanceof MultiBucketsAggregation || aggregation2 instanceof SingleBucketAggregation) {
                        singleBucketAggregations.add(singleBucketAggregation);
                        continue;
                    }
                    leafAggregations.add(aggregation2);
                }
                continue;
            }
            leafAggregations.add(agg);
        }
        int bucketAggLevelCount = Math.max(bucketAggregations.size(), (int)singleBucketAggregations.stream().flatMap(s -> AggregationToJsonProcessor.asList(s.getAggregations()).stream()).filter(MultiBucketsAggregation.class::isInstance).count());
        if (bucketAggLevelCount > 1) {
            throw new IllegalArgumentException("Multiple bucket aggregations at the same level are not supported");
        }
        ArrayList<String> addedLeafKeys = new ArrayList<String>();
        for (Aggregation leafAgg : leafAggregations) {
            boolean bl;
            if (this.timeField.equals(leafAgg.getName())) {
                this.processTimeField(leafAgg);
                continue;
            }
            if (!this.fields.contains(leafAgg.getName()) || !(bl = this.processLeaf(leafAgg))) continue;
            addedLeafKeys.add(leafAgg.getName());
        }
        boolean noMoreBucketsToProcess = bucketAggregations.isEmpty();
        if (!noMoreBucketsToProcess) {
            MultiBucketsAggregation bucketAgg = (MultiBucketsAggregation)bucketAggregations.get(0);
            if (bucketAgg instanceof Histogram) {
                this.processDateHistogram((Histogram)bucketAgg);
            } else if (bucketAgg instanceof CompositeAggregation) {
                this.processCompositeAgg((CompositeAggregation)bucketAgg);
            } else if (this.bucketAggContainsRequiredAgg(bucketAgg)) {
                this.processBucket(bucketAgg, this.fields.contains(bucketAgg.getName()));
            } else {
                noMoreBucketsToProcess = true;
            }
        }
        noMoreBucketsToProcess = singleBucketAggregations.isEmpty() && noMoreBucketsToProcess;
        for (SingleBucketAggregation singleBucketAggregation : singleBucketAggregations) {
            this.processAggs(singleBucketAggregation.getDocCount(), AggregationToJsonProcessor.asList(singleBucketAggregation.getAggregations()).stream().filter(aggregation -> aggregation instanceof MultiBucketsAggregation || aggregation instanceof SingleBucketAggregation).collect(Collectors.toList()));
        }
        if (noMoreBucketsToProcess) {
            this.queueDocToWrite(this.keyValuePairs, docCount);
        }
        addedLeafKeys.forEach(k -> this.keyValuePairs.remove(k));
    }

    private void processDateHistogram(Histogram agg) throws IOException {
        if (this.keyValuePairs.containsKey(this.timeField)) {
            throw new IllegalArgumentException("More than one composite or date_histogram cannot be used in the aggregation. [" + agg.getName() + "] is another instance of a composite or date_histogram aggregation");
        }
        boolean checkBucketTime = true;
        for (Histogram.Bucket bucket : agg.getBuckets()) {
            if (checkBucketTime) {
                long bucketTime = this.toHistogramKeyToEpoch(bucket.getKey());
                if (bucketTime < this.startTime) {
                    LOGGER.debug("Skipping bucket at [{}], startTime is [{}]", (Object)bucketTime, (Object)this.startTime);
                    continue;
                }
                checkBucketTime = false;
            }
            List childAggs = bucket.getAggregations().asList();
            this.processAggs(bucket.getDocCount(), childAggs);
            this.keyValuePairs.remove(this.timeField);
        }
    }

    private void processCompositeAgg(CompositeAggregation agg) throws IOException {
        if (this.keyValuePairs.containsKey(this.timeField)) {
            throw new IllegalArgumentException("More than one composite or date_histogram cannot be used in the aggregation. [" + agg.getName() + "] is another instance of a composite or date_histogram aggregation");
        }
        if (this.compositeAggDateValueSourceName == null) {
            throw new IllegalArgumentException("attempted to process composite agg [" + agg.getName() + "] but does not contain date_histogram value source");
        }
        boolean checkBucketTime = true;
        for (CompositeAggregation.Bucket bucket : agg.getBuckets()) {
            if (checkBucketTime) {
                long bucketTime = this.toHistogramKeyToEpoch(bucket.getKey().get(this.compositeAggDateValueSourceName));
                if (bucketTime < this.startTime) {
                    LOGGER.debug(() -> new ParameterizedMessage("Skipping bucket at [{}], startTime is [{}]", (Object)bucketTime, (Object)this.startTime));
                    continue;
                }
                checkBucketTime = false;
            }
            Collection<String> addedFields = this.processCompositeAggBucketKeys(bucket.getKey());
            List childAggs = bucket.getAggregations().asList();
            this.processAggs(bucket.getDocCount(), childAggs);
            this.keyValuePairs.remove(this.timeField);
            for (String fieldName : addedFields) {
                this.keyValuePairs.remove(fieldName);
            }
        }
    }

    private Collection<String> processCompositeAggBucketKeys(Map<String, Object> bucketKeys) {
        ArrayList<String> addedFieldValues = new ArrayList<String>();
        for (Map.Entry<String, Object> bucketKey : bucketKeys.entrySet()) {
            if (bucketKey.getKey().equals(this.compositeAggDateValueSourceName) || !this.fields.contains(bucketKey.getKey())) continue;
            this.keyValuePairs.put(bucketKey.getKey(), bucketKey.getValue());
            addedFieldValues.add(bucketKey.getKey());
        }
        return addedFieldValues;
    }

    private long toHistogramKeyToEpoch(Object key) {
        if (key instanceof ZonedDateTime) {
            return ((ZonedDateTime)key).toInstant().toEpochMilli();
        }
        if (key instanceof Double) {
            return ((Double)key).longValue();
        }
        if (key instanceof Long) {
            return (Long)key;
        }
        throw new IllegalStateException("Histogram key [" + key + "] cannot be converted to a timestamp");
    }

    private void processTimeField(Aggregation agg) {
        if (!(agg instanceof Max)) {
            throw new IllegalArgumentException(Messages.getMessage((String)"Missing max aggregation for time_field [{0}]", (Object[])new Object[]{this.timeField}));
        }
        long timestamp = (long)((Max)agg).value();
        this.keyValuePairs.put(this.timeField, timestamp);
    }

    boolean bucketAggContainsRequiredAgg(MultiBucketsAggregation aggregation) {
        if (this.fields.contains(aggregation.getName())) {
            return true;
        }
        if (aggregation instanceof CompositeAggregation && Sets.haveNonEmptyIntersection(((CompositeAggregation)aggregation).afterKey().keySet(), this.fields)) {
            return true;
        }
        if (aggregation.getBuckets().isEmpty()) {
            return false;
        }
        boolean foundRequiredAgg = false;
        List<Aggregation> aggs = AggregationToJsonProcessor.asList(((MultiBucketsAggregation.Bucket)aggregation.getBuckets().get(0)).getAggregations());
        for (Aggregation agg : aggs) {
            if (this.fields.contains(agg.getName())) {
                foundRequiredAgg = true;
                break;
            }
            if (!(agg instanceof MultiBucketsAggregation) || !(foundRequiredAgg = this.bucketAggContainsRequiredAgg((MultiBucketsAggregation)agg))) continue;
            break;
        }
        return foundRequiredAgg;
    }

    private void processBucket(MultiBucketsAggregation bucketAgg, boolean addField) throws IOException {
        for (MultiBucketsAggregation.Bucket bucket : bucketAgg.getBuckets()) {
            ArrayList<String> addedFields = new ArrayList<String>();
            if (addField) {
                addedFields.add(bucketAgg.getName());
                this.keyValuePairs.put(bucketAgg.getName(), bucket.getKey());
            }
            if (bucket instanceof CompositeAggregation.Bucket) {
                addedFields.addAll(this.processCompositeAggBucketKeys(((CompositeAggregation.Bucket)bucket).getKey()));
            }
            this.processAggs(bucket.getDocCount(), AggregationToJsonProcessor.asList(bucket.getAggregations()));
            for (String fieldName : addedFields) {
                this.keyValuePairs.remove(fieldName);
            }
        }
    }

    private boolean processLeaf(Aggregation agg) {
        if (agg instanceof NumericMetricsAggregation.SingleValue) {
            return this.processSingleValue((NumericMetricsAggregation.SingleValue)agg);
        }
        if (agg instanceof Percentiles) {
            return this.processPercentiles((Percentiles)agg);
        }
        if (agg instanceof GeoCentroid) {
            return this.processGeoCentroid((GeoCentroid)agg);
        }
        throw new IllegalArgumentException("Unsupported aggregation type [" + agg.getName() + "]");
    }

    private boolean processSingleValue(NumericMetricsAggregation.SingleValue singleValue) {
        return this.addMetricIfFinite(singleValue.getName(), singleValue.value());
    }

    private boolean addMetricIfFinite(String key, double value) {
        if (Double.isFinite(value)) {
            this.keyValuePairs.put(key, value);
            return true;
        }
        return false;
    }

    private boolean processGeoCentroid(GeoCentroid agg) {
        if (agg.count() > 0L) {
            this.keyValuePairs.put(agg.getName(), agg.centroid().getLat() + "," + agg.centroid().getLon());
            return true;
        }
        return false;
    }

    private boolean processPercentiles(Percentiles percentiles) {
        Iterator percentileIterator = percentiles.iterator();
        boolean aggregationAdded = this.addMetricIfFinite(percentiles.getName(), ((Percentile)percentileIterator.next()).getValue());
        if (percentileIterator.hasNext()) {
            throw new IllegalArgumentException("Multi-percentile aggregation [" + percentiles.getName() + "] is not supported");
        }
        return aggregationAdded;
    }

    private void queueDocToWrite(Map<String, Object> doc, long docCount) {
        if (docCount > 0L) {
            Long timeStamp;
            LinkedHashMap<String, Object> copy = new LinkedHashMap<String, Object>(doc);
            if (this.includeDocCount) {
                copy.put("doc_count", docCount);
            }
            if ((timeStamp = (Long)copy.get(this.timeField)) == null) {
                throw new IllegalArgumentException(Messages.getMessage((String)"Missing max aggregation for time_field [{0}]", (Object[])new Object[]{this.timeField}));
            }
            this.docsByBucketTimestamp.computeIfAbsent(timeStamp, t -> new ArrayList()).add(copy);
        }
    }

    boolean writeAllDocsCancellable(Predicate<Long> shouldCancel, OutputStream outputStream) throws IOException {
        if (this.docsByBucketTimestamp.isEmpty()) {
            return true;
        }
        try (XContentBuilder jsonBuilder = new XContentBuilder((XContent)JsonXContent.jsonXContent, outputStream);){
            Iterator<Map.Entry<Long, List<Map<String, Object>>>> iterator = this.docsByBucketTimestamp.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<Long, List<Map<String, Object>>> entry = iterator.next();
                if (shouldCancel.test(entry.getKey())) {
                    boolean bl = true;
                    return bl;
                }
                for (Map<String, Object> map : entry.getValue()) {
                    this.writeJsonObject(jsonBuilder, map);
                }
                iterator.remove();
            }
        }
        return false;
    }

    private void writeJsonObject(XContentBuilder jsonBuilder, Map<String, Object> record) throws IOException {
        jsonBuilder.startObject();
        for (Map.Entry<String, Object> keyValue : record.entrySet()) {
            jsonBuilder.field(keyValue.getKey(), keyValue.getValue());
            ++this.keyValueWrittenCount;
        }
        jsonBuilder.endObject();
    }

    public long getKeyValueCount() {
        return this.keyValueWrittenCount;
    }

    private static List<Aggregation> asList(@Nullable Aggregations aggs) {
        return aggs == null ? Collections.emptyList() : aggs.asList();
    }
}

