/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.searchablesnapshots.cache.blob;

import java.time.Instant;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.index.IndexFileNames;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.action.support.TransportActions;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.OriginSettingClient;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.store.LuceneFilesExtensions;
import org.elasticsearch.node.NodeClosedException;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.xpack.searchablesnapshots.cache.blob.CachedBlob;
import org.elasticsearch.xpack.searchablesnapshots.cache.common.ByteRange;

public class BlobStoreCacheService
extends AbstractLifecycleComponent {
    private static final Logger logger = LogManager.getLogger(BlobStoreCacheService.class);
    private static final Version OLD_CACHED_BLOB_SIZE_VERSION = Version.V_7_12_0;
    public static final int DEFAULT_CACHED_BLOB_SIZE = ByteSizeUnit.KB.toIntBytes(1L);
    private static final Cache<String, String> LOG_EXCEEDING_FILES_CACHE = CacheBuilder.builder().setExpireAfterAccess(TimeValue.timeValueMinutes((long)60L)).build();
    static final int MAX_IN_FLIGHT_CACHE_FILLS = Integer.MAX_VALUE;
    private final ClusterService clusterService;
    private final Semaphore inFlightCacheFills;
    private final Supplier<Long> timeSupplier;
    private final AtomicBoolean closed;
    private final Client client;
    private final String index;

    public BlobStoreCacheService(ClusterService clusterService, Client client, String index, Supplier<Long> timeSupplier) {
        this.client = new OriginSettingClient(client, "searchable_snapshots");
        this.inFlightCacheFills = new Semaphore(Integer.MAX_VALUE);
        this.closed = new AtomicBoolean(false);
        this.clusterService = clusterService;
        this.timeSupplier = timeSupplier;
        this.index = index;
    }

    protected void doStart() {
    }

    protected void doStop() {
        if (this.closed.compareAndSet(false, true)) {
            logger.debug("blob cache service is stopped");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean waitForInFlightCacheFillsToComplete(long timeout, TimeUnit unit) {
        boolean acquired = false;
        try {
            logger.debug("waiting for in-flight blob cache fills to complete");
            acquired = this.inFlightCacheFills.tryAcquire(Integer.MAX_VALUE, timeout, unit);
        }
        catch (InterruptedException e) {
            logger.warn("interrupted while waiting for in-flight blob cache fills to complete", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        finally {
            if (acquired) {
                this.inFlightCacheFills.release(Integer.MAX_VALUE);
            }
        }
        return acquired;
    }

    int getInFlightCacheFills() {
        return Integer.MAX_VALUE - this.inFlightCacheFills.availablePermits();
    }

    protected void doClose() {
    }

    public CachedBlob get(String repository, String name, String path, long offset) {
        assert (!Thread.currentThread().getName().contains("[system_read]")) : "must not block [" + Thread.currentThread().getName() + "] for a cache read";
        PlainActionFuture future = PlainActionFuture.newFuture();
        this.getAsync(repository, name, path, offset, (ActionListener<CachedBlob>)future);
        try {
            return (CachedBlob)future.actionGet(5L, TimeUnit.SECONDS);
        }
        catch (ElasticsearchTimeoutException e) {
            if (logger.isDebugEnabled()) {
                logger.debug(() -> new ParameterizedMessage("get from cache index timed out after [5s], retrieving from blob store instead [id={}]", (Object)CachedBlob.generateId(repository, name, path, offset)), (Throwable)e);
            } else {
                logger.warn("get from cache index timed out after [5s], retrieving from blob store instead");
            }
            return CachedBlob.CACHE_NOT_READY;
        }
    }

    protected void getAsync(String repository, String name, String path, long offset, final ActionListener<CachedBlob> listener) {
        if (this.closed.get()) {
            logger.debug("failed to retrieve cached blob from system index [{}], service is closed", (Object)this.index);
            listener.onResponse((Object)CachedBlob.CACHE_NOT_READY);
            return;
        }
        final GetRequest request = new GetRequest(this.index).id(CachedBlob.generateId(repository, name, path, offset));
        this.client.get(request, (ActionListener)new ActionListener<GetResponse>(){

            public void onResponse(GetResponse response) {
                if (response.isExists()) {
                    logger.debug("cache hit : [{}]", (Object)request.id());
                    assert (!response.isSourceEmpty());
                    CachedBlob cachedBlob = CachedBlob.fromSource(response.getSource());
                    assert (response.getId().equals(cachedBlob.generatedId()));
                    listener.onResponse((Object)cachedBlob);
                } else {
                    logger.debug("cache miss: [{}]", (Object)request.id());
                    listener.onResponse((Object)CachedBlob.CACHE_MISS);
                }
            }

            public void onFailure(Exception e) {
                if (BlobStoreCacheService.isExpectedCacheGetException(e)) {
                    logger.debug(() -> new ParameterizedMessage("failed to retrieve cached blob from system index [{}]", (Object)BlobStoreCacheService.this.index), (Throwable)e);
                } else {
                    logger.warn(() -> new ParameterizedMessage("failed to retrieve cached blob from system index [{}]", (Object)BlobStoreCacheService.this.index), (Throwable)e);
                    assert (false) : e;
                }
                listener.onResponse((Object)CachedBlob.CACHE_NOT_READY);
            }
        });
    }

    private static boolean isExpectedCacheGetException(Exception e) {
        if (TransportActions.isShardNotAvailableException((Throwable)e) || e instanceof ConnectTransportException || e instanceof ClusterBlockException) {
            return true;
        }
        Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
        return cause instanceof NodeClosedException || cause instanceof ConnectTransportException;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void putAsync(String repository, String name, String path, long offset, BytesReference content, ActionListener<Void> listener) {
        try {
            CachedBlob cachedBlob = new CachedBlob(Instant.ofEpochMilli(this.timeSupplier.get()), Version.CURRENT, repository, name, path, content, offset);
            final IndexRequest request = new IndexRequest(this.index).id(cachedBlob.generatedId());
            try (XContentBuilder builder = XContentFactory.jsonBuilder();){
                request.source(cachedBlob.toXContent(builder, ToXContent.EMPTY_PARAMS));
            }
            RunOnce release = new RunOnce(() -> {
                int availablePermits = this.inFlightCacheFills.availablePermits();
                assert (availablePermits > 0) : "in-flight available permits should be greater than 0 but got: " + availablePermits;
                this.inFlightCacheFills.release();
            });
            boolean submitted = false;
            this.inFlightCacheFills.acquire();
            try {
                if (this.closed.get()) {
                    listener.onFailure((Exception)new IllegalStateException("Blob cache service is closed"));
                    return;
                }
                final ActionListener wrappedListener = ActionListener.runAfter(listener, (Runnable)release);
                this.client.index(request, (ActionListener)new ActionListener<IndexResponse>(){

                    public void onResponse(IndexResponse indexResponse) {
                        logger.trace("cache fill ({}): [{}]", (Object)indexResponse.status(), (Object)request.id());
                        wrappedListener.onResponse(null);
                    }

                    public void onFailure(Exception e) {
                        logger.debug((Message)new ParameterizedMessage("failure in cache fill: [{}]", (Object)request.id()), (Throwable)e);
                        wrappedListener.onFailure(e);
                    }
                });
                submitted = true;
            }
            finally {
                if (!submitted) {
                    release.run();
                }
            }
        }
        catch (Exception e) {
            logger.warn((Message)new ParameterizedMessage("cache fill failure: [{}]", (Object)CachedBlob.generateId(repository, name, path, offset)), (Throwable)e);
            listener.onFailure(e);
        }
    }

    public ByteRange computeBlobCacheByteRange(String fileName, long fileLength, ByteSizeValue maxMetadataLength) {
        LuceneFilesExtensions fileExtension = LuceneFilesExtensions.fromExtension((String)IndexFileNames.getExtension((String)fileName));
        if (this.useLegacyCachedBlobSizes()) {
            if (fileLength <= ByteSizeUnit.KB.toBytes(8L)) {
                return ByteRange.of(0L, fileLength);
            }
            return ByteRange.of(0L, ByteSizeUnit.KB.toBytes(4L));
        }
        if (fileExtension != null && fileExtension.isMetadata()) {
            long maxAllowedLengthInBytes = maxMetadataLength.getBytes();
            if (fileLength > maxAllowedLengthInBytes) {
                BlobStoreCacheService.logExceedingFile(fileExtension, fileLength, maxMetadataLength);
            }
            return ByteRange.of(0L, Math.min(fileLength, maxAllowedLengthInBytes));
        }
        return ByteRange.of(0L, Math.min(fileLength, (long)DEFAULT_CACHED_BLOB_SIZE));
    }

    protected boolean useLegacyCachedBlobSizes() {
        Version minNodeVersion = this.clusterService.state().nodes().getMinNodeVersion();
        return minNodeVersion.before(OLD_CACHED_BLOB_SIZE_VERSION);
    }

    private static void logExceedingFile(LuceneFilesExtensions extension, long length, ByteSizeValue maxAllowedLength) {
        if (logger.isWarnEnabled()) {
            try {
                LOG_EXCEEDING_FILES_CACHE.computeIfAbsent((Object)extension.getExtension(), key -> {
                    logger.warn("file with extension [{}] is larger ([{}]) than the max. length allowed [{}] to cache metadata files in blob cache", (Object)extension, (Object)length, (Object)maxAllowedLength);
                    return key;
                });
            }
            catch (ExecutionException e) {
                logger.warn(() -> new ParameterizedMessage("Failed to log information about exceeding file type [{}] with length [{}]", (Object)extension, (Object)length), (Throwable)e);
            }
        }
    }
}

