/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexFormatTooNewException;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.elasticsearch.Assertions;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.flush.FlushRequest;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.bytes.ReleasableBytesReference;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.util.CancellableThreads;
import org.elasticsearch.core.AbstractRefCounted;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.mapper.MapperException;
import org.elasticsearch.index.seqno.ReplicationTracker;
import org.elasticsearch.index.seqno.RetentionLeases;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardNotRecoveringException;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetadata;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.recovery.MultiFileWriter;
import org.elasticsearch.indices.recovery.PeerRecoveryTargetService;
import org.elasticsearch.indices.recovery.RecoveryFailedException;
import org.elasticsearch.indices.recovery.RecoveryRequestTracker;
import org.elasticsearch.indices.recovery.RecoveryState;
import org.elasticsearch.indices.recovery.RecoveryTargetHandler;

public class RecoveryTarget
extends AbstractRefCounted
implements RecoveryTargetHandler {
    private final Logger logger;
    private static final AtomicLong idGenerator = new AtomicLong();
    private static final String RECOVERY_PREFIX = "recovery.";
    private final ShardId shardId;
    private final long recoveryId;
    private final IndexShard indexShard;
    private final DiscoveryNode sourceNode;
    private final MultiFileWriter multiFileWriter;
    private final RecoveryRequestTracker requestTracker = new RecoveryRequestTracker();
    private final Store store;
    private final PeerRecoveryTargetService.RecoveryListener listener;
    private final AtomicBoolean finished = new AtomicBoolean();
    private final CancellableThreads cancellableThreads;
    private volatile long lastAccessTime = System.nanoTime();
    private volatile boolean recoveryMonitorEnabled = true;
    private final CountDownLatch closedLatch = new CountDownLatch(1);

    public RecoveryTarget(IndexShard indexShard, DiscoveryNode sourceNode, PeerRecoveryTargetService.RecoveryListener listener) {
        super("recovery_status");
        this.cancellableThreads = new CancellableThreads();
        this.recoveryId = idGenerator.incrementAndGet();
        this.listener = listener;
        this.logger = Loggers.getLogger(this.getClass(), indexShard.shardId(), new String[0]);
        this.indexShard = indexShard;
        this.sourceNode = sourceNode;
        this.shardId = indexShard.shardId();
        String tempFilePrefix = RECOVERY_PREFIX + UUIDs.randomBase64UUID() + ".";
        this.multiFileWriter = new MultiFileWriter(indexShard.store(), indexShard.recoveryState().getIndex(), tempFilePrefix, this.logger, this::ensureRefCount);
        this.store = indexShard.store();
        this.store.incRef();
        indexShard.recoveryStats().incCurrentAsTarget();
    }

    public RecoveryTarget retryCopy() {
        return new RecoveryTarget(this.indexShard, this.sourceNode, this.listener);
    }

    @Nullable
    public ActionListener<Void> markRequestReceivedAndCreateListener(long requestSeqNo, ActionListener<Void> listener) {
        return this.requestTracker.markReceivedAndCreateListener(requestSeqNo, listener);
    }

    public long recoveryId() {
        return this.recoveryId;
    }

    public ShardId shardId() {
        return this.shardId;
    }

    public IndexShard indexShard() {
        this.ensureRefCount();
        return this.indexShard;
    }

    public DiscoveryNode sourceNode() {
        return this.sourceNode;
    }

    public RecoveryState state() {
        return this.indexShard.recoveryState();
    }

    public CancellableThreads cancellableThreads() {
        return this.cancellableThreads;
    }

    public long lastAccessTime() {
        if (this.recoveryMonitorEnabled) {
            return this.lastAccessTime;
        }
        return System.nanoTime();
    }

    public void setLastAccessTime() {
        this.lastAccessTime = System.nanoTime();
    }

    public Releasable disableRecoveryMonitor() {
        assert (this.recoveryMonitorEnabled) : "recovery monitor already disabled";
        this.recoveryMonitorEnabled = false;
        return () -> {
            this.setLastAccessTime();
            this.recoveryMonitorEnabled = true;
        };
    }

    public Store store() {
        this.ensureRefCount();
        return this.store;
    }

    boolean resetRecovery(CancellableThreads newTargetCancellableThreads) throws IOException {
        if (this.finished.compareAndSet(false, true)) {
            try {
                this.logger.debug("reset of recovery with shard {} and id [{}]", (Object)this.shardId, (Object)this.recoveryId);
            }
            finally {
                this.decRef();
            }
            try {
                newTargetCancellableThreads.execute(this.closedLatch::await);
            }
            catch (CancellableThreads.ExecutionCancelledException e) {
                this.logger.trace("new recovery target cancelled for shard {} while waiting on old recovery target with id [{}] to close", (Object)this.shardId, (Object)this.recoveryId);
                return false;
            }
            RecoveryState.Stage stage = this.indexShard.recoveryState().getStage();
            if (this.indexShard.recoveryState().getPrimary() && (stage == RecoveryState.Stage.FINALIZE || stage == RecoveryState.Stage.DONE)) {
                assert (stage != RecoveryState.Stage.DONE) : "recovery should not have completed when it's being reset";
                throw new IllegalStateException("cannot reset recovery as previous attempt made it past finalization step");
            }
            this.indexShard.performRecoveryRestart();
            return true;
        }
        return false;
    }

    public void cancel(String reason) {
        if (this.finished.compareAndSet(false, true)) {
            try {
                this.logger.debug("recovery canceled (reason: [{}])", (Object)reason);
                this.cancellableThreads.cancel(reason);
            }
            finally {
                this.decRef();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void fail(RecoveryFailedException e, boolean sendShardFailure) {
        if (this.finished.compareAndSet(false, true)) {
            try {
                this.notifyListener(e, sendShardFailure);
            }
            finally {
                try {
                    this.cancellableThreads.cancel("failed recovery [" + ExceptionsHelper.stackTrace(e) + "]");
                }
                finally {
                    this.decRef();
                }
            }
        }
    }

    public void notifyListener(RecoveryFailedException e, boolean sendShardFailure) {
        this.listener.onRecoveryFailure(this.state(), e, sendShardFailure);
    }

    public void markAsDone() {
        if (this.finished.compareAndSet(false, true)) {
            assert (this.multiFileWriter.tempFileNames.isEmpty()) : "not all temporary files are renamed";
            try {
                this.indexShard.postRecovery("peer recovery done");
            }
            finally {
                this.decRef();
            }
            this.listener.onRecoveryDone(this.state(), this.indexShard.getTimestampRange());
        }
    }

    protected void closeInternal() {
        try {
            this.multiFileWriter.close();
        }
        finally {
            this.store.decRef();
            this.indexShard.recoveryStats().decCurrentAsTarget();
            this.closedLatch.countDown();
        }
    }

    public String toString() {
        return this.shardId + " [" + this.recoveryId + "]";
    }

    private void ensureRefCount() {
        if (this.refCount() <= 0) {
            throw new ElasticsearchException("RecoveryStatus is used but it's refcount is 0. Probably a mismatch between incRef/decRef calls", new Object[0]);
        }
    }

    @Override
    public void prepareForTranslogOperations(int totalTranslogOps, ActionListener<Void> listener) {
        ActionListener.completeWith(listener, () -> {
            this.state().getIndex().setFileDetailsComplete();
            this.state().getTranslog().totalOperations(totalTranslogOps);
            this.indexShard().openEngineAndSkipTranslogRecovery();
            return null;
        });
    }

    @Override
    public void finalizeRecovery(long globalCheckpoint, long trimAboveSeqNo, ActionListener<Void> listener) {
        ActionListener.completeWith(listener, () -> {
            this.indexShard.updateGlobalCheckpointOnReplica(globalCheckpoint, "finalizing recovery");
            this.indexShard.sync();
            this.indexShard.persistRetentionLeases();
            if (trimAboveSeqNo != -2L) {
                this.indexShard.rollTranslogGeneration();
                this.indexShard.afterWriteOperation();
                this.indexShard.trimOperationOfPreviousPrimaryTerms(trimAboveSeqNo);
            }
            if (this.hasUncommittedOperations()) {
                this.indexShard.flush(new FlushRequest(new String[0]).force(true).waitIfOngoing(true));
            }
            this.indexShard.finalizeRecovery();
            return null;
        });
    }

    private boolean hasUncommittedOperations() throws IOException {
        long localCheckpointOfCommit = Long.parseLong(this.indexShard.commitStats().getUserData().get("local_checkpoint"));
        return this.indexShard.estimateNumberOfHistoryOperations("peer-recovery", this.indexShard.indexSettings().isSoftDeleteEnabled() ? Engine.HistorySource.INDEX : Engine.HistorySource.TRANSLOG, localCheckpointOfCommit + 1L) > 0;
    }

    @Override
    public void handoffPrimaryContext(ReplicationTracker.PrimaryContext primaryContext, ActionListener<Void> listener) {
        ActionListener.completeWith(listener, () -> {
            this.indexShard.activateWithPrimaryContext(primaryContext);
            return null;
        });
    }

    @Override
    public void indexTranslogOperations(List<Translog.Operation> operations, int totalTranslogOps, long maxSeenAutoIdTimestampOnPrimary, long maxSeqNoOfDeletesOrUpdatesOnPrimary, RetentionLeases retentionLeases, long mappingVersionOnPrimary, ActionListener<Long> listener) {
        ActionListener.completeWith(listener, () -> {
            RecoveryState.Translog translog = this.state().getTranslog();
            translog.totalOperations(totalTranslogOps);
            assert (this.indexShard().recoveryState() == this.state());
            if (this.indexShard().state() != IndexShardState.RECOVERING) {
                throw new IndexShardNotRecoveringException(this.shardId, this.indexShard().state());
            }
            this.indexShard().updateMaxUnsafeAutoIdTimestamp(maxSeenAutoIdTimestampOnPrimary);
            this.indexShard().advanceMaxSeqNoOfUpdatesOrDeletes(maxSeqNoOfDeletesOrUpdatesOnPrimary);
            this.indexShard().updateRetentionLeasesOnReplica(retentionLeases);
            for (Translog.Operation operation : operations) {
                Engine.Result result = this.indexShard().applyTranslogOperation(operation, Engine.Operation.Origin.PEER_RECOVERY);
                if (result.getResultType() == Engine.Result.Type.MAPPING_UPDATE_REQUIRED) {
                    throw new MapperException("mapping updates are not allowed [" + operation + "]");
                }
                if (result.getFailure() == null) continue;
                if (Assertions.ENABLED && !(result.getFailure() instanceof MapperException)) {
                    throw new AssertionError("unexpected failure while replicating translog entry", result.getFailure());
                }
                ExceptionsHelper.reThrowIfNotNull(result.getFailure());
            }
            translog.incrementRecoveredOperations(operations.size());
            this.indexShard().sync();
            this.indexShard().afterWriteOperation();
            return this.indexShard().getLocalCheckpoint();
        });
    }

    @Override
    public void receiveFileInfo(List<String> phase1FileNames, List<Long> phase1FileSizes, List<String> phase1ExistingFileNames, List<Long> phase1ExistingFileSizes, int totalTranslogOps, ActionListener<Void> listener) {
        ActionListener.completeWith(listener, () -> {
            int i;
            this.indexShard.resetRecoveryStage();
            this.indexShard.prepareForIndexRecovery();
            RecoveryState.Index index = this.state().getIndex();
            for (i = 0; i < phase1ExistingFileNames.size(); ++i) {
                index.addFileDetail((String)phase1ExistingFileNames.get(i), (Long)phase1ExistingFileSizes.get(i), true);
            }
            for (i = 0; i < phase1FileNames.size(); ++i) {
                index.addFileDetail((String)phase1FileNames.get(i), (Long)phase1FileSizes.get(i), false);
            }
            index.setFileDetailsComplete();
            this.state().getTranslog().totalOperations(totalTranslogOps);
            this.state().getTranslog().totalOperationsOnStart(totalTranslogOps);
            return null;
        });
    }

    @Override
    public void cleanFiles(int totalTranslogOps, long globalCheckpoint, Store.MetadataSnapshot sourceMetadata, ActionListener<Void> listener) {
        ActionListener.completeWith(listener, () -> {
            this.state().getTranslog().totalOperations(totalTranslogOps);
            this.multiFileWriter.renameAllTempFiles();
            Store store = this.store();
            store.incRef();
            try {
                store.cleanupAndVerify("recovery CleanFilesRequestHandler", sourceMetadata);
                if (this.indexShard.indexSettings().getIndexVersionCreated().before(Version.V_6_0_0_rc1)) {
                    store.ensureIndexHasHistoryUUID();
                }
                String translogUUID = Translog.createEmptyTranslog(this.indexShard.shardPath().resolveTranslog(), globalCheckpoint, this.shardId, this.indexShard.getPendingPrimaryTerm());
                store.associateIndexWithNewTranslog(translogUUID);
                if (this.indexShard.getRetentionLeases().leases().isEmpty()) {
                    this.indexShard.persistRetentionLeases();
                    assert (this.indexShard.loadRetentionLeases().leases().isEmpty());
                } else assert (this.indexShard.assertRetentionLeasesPersisted());
                this.indexShard.maybeCheckIndex();
                this.state().setRemoteTranslogStage();
            }
            catch (CorruptIndexException | IndexFormatTooNewException | IndexFormatTooOldException ex) {
                try {
                    try {
                        store.removeCorruptionMarker();
                    }
                    finally {
                        Lucene.cleanLuceneIndex(store.directory());
                    }
                }
                catch (Exception e) {
                    this.logger.debug("Failed to clean lucene index", (Throwable)e);
                    ex.addSuppressed(e);
                }
                RecoveryFailedException rfe = new RecoveryFailedException(this.state(), "failed to clean after recovery", ex);
                this.fail(rfe, true);
                throw rfe;
            }
            catch (Exception ex) {
                RecoveryFailedException rfe = new RecoveryFailedException(this.state(), "failed to clean after recovery", (Throwable)ex);
                this.fail(rfe, true);
                throw rfe;
            }
            finally {
                store.decRef();
            }
            return null;
        });
    }

    @Override
    public void writeFileChunk(StoreFileMetadata fileMetadata, long position, ReleasableBytesReference content, boolean lastChunk, int totalTranslogOps, ActionListener<Void> listener) {
        try {
            this.state().getTranslog().totalOperations(totalTranslogOps);
            this.multiFileWriter.writeFileChunk(fileMetadata, position, content, lastChunk);
            listener.onResponse(null);
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    public String getTempNameForFile(String origFile) {
        return this.multiFileWriter.getTempNameForFile(origFile);
    }

    Path translogLocation() {
        return this.indexShard().shardPath().resolveTranslog();
    }
}

