/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.transport.nio;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import org.elasticsearch.nio.InboundChannelBuffer;
import org.elasticsearch.nio.utils.ExceptionsHelper;

public class SSLDriver
implements AutoCloseable {
    private static final ByteBuffer[] EMPTY_BUFFER_ARRAY = new ByteBuffer[0];
    private final SSLEngine engine;
    private final boolean isClientMode;
    private Mode currentMode = new HandshakeMode();
    private ByteBuffer networkWriteBuffer;
    private ByteBuffer networkReadBuffer;

    public SSLDriver(SSLEngine engine, boolean isClientMode) {
        this.engine = engine;
        this.isClientMode = isClientMode;
        SSLSession session = engine.getSession();
        this.networkReadBuffer = ByteBuffer.allocate(session.getPacketBufferSize());
        this.networkWriteBuffer = ByteBuffer.allocate(session.getPacketBufferSize());
        this.networkWriteBuffer.position(this.networkWriteBuffer.limit());
    }

    public void init() throws SSLException {
        this.engine.setUseClientMode(this.isClientMode);
        if (!this.currentMode.isHandshake()) {
            throw new AssertionError((Object)("Attempted to init outside from non-handshaking mode: " + this.currentMode.modeName()));
        }
        this.engine.beginHandshake();
        ((HandshakeMode)this.currentMode).startHandshake();
    }

    public void renegotiate() throws SSLException {
        if (!this.currentMode.isApplication()) {
            throw new IllegalStateException("Attempted to renegotiate while in invalid mode: " + this.currentMode.modeName());
        }
        this.currentMode = new HandshakeMode();
        this.engine.beginHandshake();
        ((HandshakeMode)this.currentMode).startHandshake();
    }

    public SSLEngine getSSLEngine() {
        return this.engine;
    }

    public boolean hasFlushPending() {
        return this.networkWriteBuffer.hasRemaining();
    }

    public boolean isHandshaking() {
        return this.currentMode.isHandshake();
    }

    public ByteBuffer getNetworkWriteBuffer() {
        return this.networkWriteBuffer;
    }

    public ByteBuffer getNetworkReadBuffer() {
        return this.networkReadBuffer;
    }

    public void read(InboundChannelBuffer buffer) throws SSLException {
        Mode modePriorToRead;
        do {
            modePriorToRead = this.currentMode;
            this.currentMode.read(buffer);
        } while (modePriorToRead != this.currentMode);
    }

    public boolean readyForApplicationWrites() {
        return this.currentMode.isApplication();
    }

    public boolean needsNonApplicationWrite() {
        return this.currentMode.needsNonApplicationWrite();
    }

    public int applicationWrite(ByteBuffer[] buffers) throws SSLException {
        assert (this.readyForApplicationWrites()) : "Should not be called if driver is not ready for application writes";
        return this.currentMode.write(buffers);
    }

    public void nonApplicationWrite() throws SSLException {
        assert (!this.currentMode.isApplication()) : "Should not be called if driver is in application mode";
        if (this.currentMode.isApplication()) {
            throw new AssertionError((Object)("Attempted to non-application write from invalid mode: " + this.currentMode.modeName()));
        }
        this.currentMode.write(EMPTY_BUFFER_ARRAY);
    }

    public void initiateClose() {
        this.closingInternal();
    }

    public boolean isClosed() {
        return this.currentMode.isClose() && ((CloseMode)this.currentMode).isCloseDone();
    }

    @Override
    public void close() throws SSLException {
        ArrayList<SSLException> closingExceptions = new ArrayList<SSLException>(2);
        this.closingInternal();
        CloseMode closeMode = (CloseMode)this.currentMode;
        if (closeMode.needToSendClose) {
            closingExceptions.add(new SSLException("Closed engine without completely sending the close alert message."));
            this.engine.closeOutbound();
        }
        if (closeMode.needToReceiveClose) {
            closingExceptions.add(new SSLException("Closed engine without receiving the close alert message."));
            closeMode.closeInboundAndSwallowPeerDidNotCloseException();
        }
        ExceptionsHelper.rethrowAndSuppress(closingExceptions);
    }

    private SSLEngineResult unwrap(InboundChannelBuffer buffer) throws SSLException {
        SSLEngineResult result;
        block6: while (true) {
            result = this.engine.unwrap(this.networkReadBuffer, buffer.sliceBuffersFrom(buffer.getIndex()));
            buffer.incrementIndex((long)result.bytesProduced());
            switch (result.getStatus()) {
                case OK: {
                    this.networkReadBuffer.compact();
                    return result;
                }
                case BUFFER_UNDERFLOW: {
                    int currentCapacity = this.networkReadBuffer.capacity();
                    this.ensureNetworkReadBufferSize();
                    if (currentCapacity == this.networkReadBuffer.capacity()) {
                        this.networkReadBuffer.compact();
                    }
                    return result;
                }
                case BUFFER_OVERFLOW: {
                    this.ensureApplicationBufferSize(buffer);
                    continue block6;
                }
                case CLOSED: {
                    assert (this.engine.isInboundDone()) : "We received close_notify so read should be done";
                    this.closingInternal();
                    return result;
                }
            }
            break;
        }
        throw new IllegalStateException("Unexpected UNWRAP result: " + (Object)((Object)result.getStatus()));
    }

    private SSLEngineResult wrap(ByteBuffer[] buffers) throws SSLException {
        SSLEngineResult result;
        assert (!this.hasFlushPending()) : "Should never called with pending writes";
        this.networkWriteBuffer.clear();
        block8: while (true) {
            try {
                result = buffers.length == 1 ? this.engine.wrap(buffers[0], this.networkWriteBuffer) : this.engine.wrap(buffers, this.networkWriteBuffer);
            }
            catch (SSLException e) {
                this.networkWriteBuffer.position(this.networkWriteBuffer.limit());
                throw e;
            }
            switch (result.getStatus()) {
                case OK: {
                    this.networkWriteBuffer.flip();
                    return result;
                }
                case BUFFER_UNDERFLOW: {
                    throw new IllegalStateException("Should not receive BUFFER_UNDERFLOW on WRAP");
                }
                case BUFFER_OVERFLOW: {
                    int currentCapacity = this.networkWriteBuffer.capacity();
                    this.ensureNetworkWriteBufferSize();
                    if (currentCapacity != this.networkWriteBuffer.capacity()) continue block8;
                    return result;
                }
                case CLOSED: {
                    if (result.bytesProduced() > 0) {
                        this.networkWriteBuffer.flip();
                    } else assert (false) : "WRAP during close processing should produce close message.";
                    return result;
                }
            }
            break;
        }
        throw new IllegalStateException("Unexpected WRAP result: " + (Object)((Object)result.getStatus()));
    }

    private void closingInternal() {
        if (!this.currentMode.isClose()) {
            this.currentMode = new CloseMode(this.currentMode.isHandshake());
        }
    }

    private void ensureApplicationBufferSize(InboundChannelBuffer applicationBuffer) {
        int applicationBufferSize = this.engine.getSession().getApplicationBufferSize();
        if (applicationBuffer.getRemaining() < (long)applicationBufferSize) {
            applicationBuffer.ensureCapacity(applicationBuffer.getIndex() + (long)this.engine.getSession().getApplicationBufferSize());
        }
    }

    private void ensureNetworkWriteBufferSize() {
        this.networkWriteBuffer = this.ensureNetBufferSize(this.networkWriteBuffer);
    }

    private void ensureNetworkReadBufferSize() {
        this.networkReadBuffer = this.ensureNetBufferSize(this.networkReadBuffer);
    }

    private ByteBuffer ensureNetBufferSize(ByteBuffer current) {
        int networkPacketSize = this.engine.getSession().getPacketBufferSize();
        if (current.capacity() < networkPacketSize) {
            ByteBuffer newBuffer = ByteBuffer.allocate(networkPacketSize);
            current.flip();
            newBuffer.put(current);
            return newBuffer;
        }
        return current;
    }

    private class CloseMode
    implements Mode {
        private boolean needToSendClose = true;
        private boolean needToReceiveClose = true;

        private CloseMode(boolean isHandshaking) {
            if (isHandshaking && !SSLDriver.this.engine.isInboundDone()) {
                this.needToReceiveClose = false;
            } else if (SSLDriver.this.engine.isInboundDone()) {
                this.needToReceiveClose = false;
            }
            if (SSLDriver.this.engine.isOutboundDone()) {
                this.needToSendClose = false;
            } else {
                SSLDriver.this.engine.closeOutbound();
            }
        }

        @Override
        public void read(InboundChannelBuffer buffer) throws SSLException {
            if (!this.needToReceiveClose) {
                return;
            }
            SSLDriver.this.ensureApplicationBufferSize(buffer);
            boolean continueUnwrap = true;
            while (continueUnwrap && SSLDriver.this.networkReadBuffer.position() > 0) {
                SSLDriver.this.networkReadBuffer.flip();
                SSLEngineResult result = SSLDriver.this.unwrap(buffer);
                continueUnwrap = result.bytesProduced() > 0 || result.bytesConsumed() > 0;
            }
            if (SSLDriver.this.engine.isInboundDone()) {
                this.needToReceiveClose = false;
            }
        }

        @Override
        public int write(ByteBuffer[] buffers) throws SSLException {
            if (!SSLDriver.this.hasFlushPending() && SSLDriver.this.engine.isOutboundDone()) {
                this.needToSendClose = false;
                if (!this.needToReceiveClose && !SSLDriver.this.engine.isInboundDone()) {
                    this.closeInboundAndSwallowPeerDidNotCloseException();
                }
            } else {
                SSLDriver.this.wrap(EMPTY_BUFFER_ARRAY);
                assert (SSLDriver.this.hasFlushPending()) : "Should have produced close message";
            }
            return 0;
        }

        @Override
        public boolean needsNonApplicationWrite() {
            return this.needToSendClose;
        }

        @Override
        public boolean isHandshake() {
            return false;
        }

        @Override
        public boolean isApplication() {
            return false;
        }

        @Override
        public boolean isClose() {
            return true;
        }

        @Override
        public String modeName() {
            return "CLOSE";
        }

        private boolean isCloseDone() {
            return !this.needToSendClose && !this.needToReceiveClose;
        }

        private void closeInboundAndSwallowPeerDidNotCloseException() throws SSLException {
            block2: {
                try {
                    SSLDriver.this.engine.closeInbound();
                }
                catch (SSLException e) {
                    if (e.getMessage().contains("before receiving peer's close_notify")) break block2;
                    throw e;
                }
            }
        }
    }

    private class ApplicationMode
    implements Mode {
        private ApplicationMode() {
        }

        @Override
        public void read(InboundChannelBuffer buffer) throws SSLException {
            SSLDriver.this.ensureApplicationBufferSize(buffer);
            boolean continueUnwrap = true;
            while (continueUnwrap && SSLDriver.this.networkReadBuffer.position() > 0) {
                SSLDriver.this.networkReadBuffer.flip();
                SSLEngineResult result = SSLDriver.this.unwrap(buffer);
                boolean renegotiationRequested = result.getStatus() != SSLEngineResult.Status.CLOSED && this.maybeRenegotiation(result.getHandshakeStatus());
                continueUnwrap = result.bytesProduced() > 0 && !renegotiationRequested;
            }
        }

        @Override
        public int write(ByteBuffer[] buffers) throws SSLException {
            SSLEngineResult result = SSLDriver.this.wrap(buffers);
            this.maybeRenegotiation(result.getHandshakeStatus());
            return result.bytesConsumed();
        }

        private boolean maybeRenegotiation(SSLEngineResult.HandshakeStatus newStatus) throws SSLException {
            if (newStatus != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING && newStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
                SSLDriver.this.renegotiate();
                return true;
            }
            return false;
        }

        @Override
        public boolean needsNonApplicationWrite() {
            return false;
        }

        @Override
        public boolean isHandshake() {
            return false;
        }

        @Override
        public boolean isApplication() {
            return true;
        }

        @Override
        public boolean isClose() {
            return false;
        }

        @Override
        public String modeName() {
            return "APPLICATION";
        }
    }

    private class HandshakeMode
    implements Mode {
        private SSLEngineResult.HandshakeStatus handshakeStatus;

        private HandshakeMode() {
        }

        private void startHandshake() throws SSLException {
            this.handshakeStatus = SSLDriver.this.engine.getHandshakeStatus();
            if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.NEED_UNWRAP && this.handshakeStatus != SSLEngineResult.HandshakeStatus.NEED_WRAP) {
                try {
                    this.handshake();
                }
                catch (SSLException e) {
                    SSLDriver.this.closingInternal();
                    throw e;
                }
            }
        }

        private void handshake() throws SSLException {
            boolean continueHandshaking = true;
            while (continueHandshaking) {
                switch (this.handshakeStatus) {
                    case NEED_UNWRAP: {
                        continueHandshaking = false;
                        break;
                    }
                    case NEED_WRAP: {
                        if (!SSLDriver.this.hasFlushPending()) {
                            this.handshakeStatus = SSLDriver.this.wrap(EMPTY_BUFFER_ARRAY).getHandshakeStatus();
                        }
                        if (this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_TASK) break;
                        continueHandshaking = false;
                        break;
                    }
                    case NEED_TASK: {
                        this.runTasks();
                        this.handshakeStatus = SSLDriver.this.engine.getHandshakeStatus();
                        break;
                    }
                    case NOT_HANDSHAKING: {
                        this.maybeFinishHandshake();
                        continueHandshaking = false;
                        break;
                    }
                    case FINISHED: {
                        this.maybeFinishHandshake();
                        continueHandshaking = false;
                    }
                }
            }
        }

        @Override
        public void read(InboundChannelBuffer buffer) throws SSLException {
            SSLDriver.this.ensureApplicationBufferSize(buffer);
            boolean continueUnwrap = true;
            while (continueUnwrap && SSLDriver.this.networkReadBuffer.position() > 0) {
                SSLDriver.this.networkReadBuffer.flip();
                try {
                    SSLEngineResult result = SSLDriver.this.unwrap(buffer);
                    this.handshakeStatus = result.getHandshakeStatus();
                    this.handshake();
                    continueUnwrap = result.bytesConsumed() > 0 && SSLDriver.this.currentMode.isHandshake();
                }
                catch (SSLException e) {
                    SSLDriver.this.closingInternal();
                    throw e;
                }
            }
        }

        @Override
        public int write(ByteBuffer[] buffers) throws SSLException {
            try {
                this.handshake();
            }
            catch (SSLException e) {
                SSLDriver.this.closingInternal();
                throw e;
            }
            return 0;
        }

        @Override
        public boolean needsNonApplicationWrite() {
            return this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_WRAP || this.handshakeStatus == SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING || this.handshakeStatus == SSLEngineResult.HandshakeStatus.FINISHED;
        }

        @Override
        public boolean isHandshake() {
            return true;
        }

        @Override
        public boolean isApplication() {
            return false;
        }

        @Override
        public boolean isClose() {
            return false;
        }

        @Override
        public String modeName() {
            return "HANDSHAKE";
        }

        private void runTasks() {
            Runnable delegatedTask;
            while ((delegatedTask = SSLDriver.this.engine.getDelegatedTask()) != null) {
                delegatedTask.run();
            }
        }

        /*
         * Enabled aggressive block sorting
         */
        private void maybeFinishHandshake() {
            if (SSLDriver.this.engine.isOutboundDone() || SSLDriver.this.engine.isInboundDone()) {
                if (SSLDriver.this.currentMode.isHandshake()) {
                    SSLDriver.this.currentMode = new CloseMode(true);
                    return;
                }
                String message = "Expected to be in handshaking mode. Instead in non-handshaking mode: " + SSLDriver.this.currentMode;
                throw new AssertionError((Object)message);
            }
            if (SSLDriver.this.hasFlushPending()) return;
            if (SSLDriver.this.currentMode.isHandshake()) {
                SSLDriver.this.currentMode = new ApplicationMode();
                return;
            }
            String message = "Attempted to transition to application mode from non-handshaking mode: " + SSLDriver.this.currentMode;
            throw new AssertionError((Object)message);
        }
    }

    private static interface Mode {
        public void read(InboundChannelBuffer var1) throws SSLException;

        public int write(ByteBuffer[] var1) throws SSLException;

        public boolean needsNonApplicationWrite();

        public boolean isHandshake();

        public boolean isApplication();

        public boolean isClose();

        public String modeName();
    }
}

