/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.indexlifecycle;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.ToXContentObject;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.index.Index;
import org.elasticsearch.xpack.core.indexlifecycle.AsyncActionStep;
import org.elasticsearch.xpack.core.indexlifecycle.AsyncWaitStep;
import org.elasticsearch.xpack.core.indexlifecycle.ClusterStateActionStep;
import org.elasticsearch.xpack.core.indexlifecycle.ClusterStateWaitStep;
import org.elasticsearch.xpack.core.indexlifecycle.ErrorStep;
import org.elasticsearch.xpack.core.indexlifecycle.IndexLifecycleMetadata;
import org.elasticsearch.xpack.core.indexlifecycle.LifecycleExecutionState;
import org.elasticsearch.xpack.core.indexlifecycle.LifecyclePolicyMetadata;
import org.elasticsearch.xpack.core.indexlifecycle.LifecycleSettings;
import org.elasticsearch.xpack.core.indexlifecycle.Phase;
import org.elasticsearch.xpack.core.indexlifecycle.PhaseCompleteStep;
import org.elasticsearch.xpack.core.indexlifecycle.PhaseExecutionInfo;
import org.elasticsearch.xpack.core.indexlifecycle.RolloverAction;
import org.elasticsearch.xpack.core.indexlifecycle.Step;
import org.elasticsearch.xpack.core.indexlifecycle.TerminalPolicyStep;
import org.elasticsearch.xpack.indexlifecycle.ExecuteStepsUpdateTask;
import org.elasticsearch.xpack.indexlifecycle.MoveToErrorStepUpdateTask;
import org.elasticsearch.xpack.indexlifecycle.MoveToNextStepUpdateTask;
import org.elasticsearch.xpack.indexlifecycle.PolicyStepsRegistry;
import org.elasticsearch.xpack.indexlifecycle.SetStepInfoUpdateTask;

public class IndexLifecycleRunner {
    private static final Logger logger = LogManager.getLogger(IndexLifecycleRunner.class);
    private static final ToXContent.Params STACKTRACE_PARAMS = new ToXContent.MapParams(Collections.singletonMap("rest.exception.stacktrace.skip", "false"));
    private PolicyStepsRegistry stepRegistry;
    private ClusterService clusterService;
    private LongSupplier nowSupplier;

    public IndexLifecycleRunner(PolicyStepsRegistry stepRegistry, ClusterService clusterService, LongSupplier nowSupplier) {
        this.stepRegistry = stepRegistry;
        this.clusterService = clusterService;
        this.nowSupplier = nowSupplier;
    }

    boolean isReadyToTransitionToThisPhase(String policy, IndexMetaData indexMetaData, String phase) {
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)indexMetaData);
        if (lifecycleState.getLifecycleDate() == null) {
            logger.trace("no index creation date has been set yet");
            return true;
        }
        Long lifecycleDate = lifecycleState.getLifecycleDate();
        assert (lifecycleDate != null && lifecycleDate >= 0L) : "expected index to have a lifecycle date but it did not";
        TimeValue after = this.stepRegistry.getIndexAgeForPhase(policy, phase);
        long now = this.nowSupplier.getAsLong();
        TimeValue age = new TimeValue(now - lifecycleDate);
        if (logger.isTraceEnabled()) {
            logger.trace("[{}] checking for index age to be at least [{}] before performing actions in the \"{}\" phase. Now: {}, lifecycle date: {}, age: [{}/{}s]", (Object)indexMetaData.getIndex().getName(), (Object)after, (Object)phase, (Object)new TimeValue(now).seconds(), (Object)new TimeValue(lifecycleDate.longValue()).seconds(), (Object)age, (Object)age.seconds());
        }
        return now >= lifecycleDate + after.getMillis();
    }

    public void runPeriodicStep(final String policy, final IndexMetaData indexMetaData) {
        final String index = indexMetaData.getIndex().getName();
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)indexMetaData);
        final Step currentStep = IndexLifecycleRunner.getCurrentStep(this.stepRegistry, policy, indexMetaData, lifecycleState);
        if (currentStep == null) {
            if (!this.stepRegistry.policyExists(policy)) {
                this.markPolicyDoesNotExist(policy, indexMetaData.getIndex(), lifecycleState);
                return;
            }
            logger.error("current step [{}] for index [{}] with policy [{}] is not recognized", (Object)IndexLifecycleRunner.getCurrentStepKey(lifecycleState), (Object)index, (Object)policy);
            return;
        }
        if (currentStep instanceof TerminalPolicyStep) {
            logger.debug("policy [{}] for index [{}] complete, skipping execution", (Object)policy, (Object)index);
            return;
        }
        if (currentStep instanceof ErrorStep) {
            logger.debug("policy [{}] for index [{}] on an error step, skipping execution", (Object)policy, (Object)index);
            return;
        }
        logger.trace("[{}] maybe running periodic step ({}) with current step {}", (Object)index, (Object)currentStep.getClass().getSimpleName(), (Object)currentStep.getKey());
        if (currentStep instanceof PhaseCompleteStep) {
            if (this.isReadyToTransitionToThisPhase(policy, indexMetaData, currentStep.getNextStepKey().getPhase())) {
                this.moveToStep(indexMetaData.getIndex(), policy, currentStep.getKey(), currentStep.getNextStepKey());
            }
        } else if (currentStep instanceof AsyncWaitStep) {
            logger.debug("[{}] running periodic policy with current-step [{}]", (Object)index, (Object)currentStep.getKey());
            ((AsyncWaitStep)currentStep).evaluateCondition(indexMetaData, new AsyncWaitStep.Listener(){

                public void onResponse(boolean conditionMet, ToXContentObject stepInfo) {
                    logger.trace("cs-change-async-wait-callback, [{}] current-step: {}", (Object)index, (Object)currentStep.getKey());
                    if (conditionMet) {
                        IndexLifecycleRunner.this.moveToStep(indexMetaData.getIndex(), policy, currentStep.getKey(), currentStep.getNextStepKey());
                    } else if (stepInfo != null) {
                        IndexLifecycleRunner.this.setStepInfo(indexMetaData.getIndex(), policy, currentStep.getKey(), stepInfo);
                    }
                }

                public void onFailure(Exception e) {
                    IndexLifecycleRunner.this.moveToErrorStep(indexMetaData.getIndex(), policy, currentStep.getKey(), e);
                }
            });
        } else {
            logger.trace("[{}] ignoring non periodic step execution from step transition [{}]", (Object)index, (Object)currentStep.getKey());
        }
    }

    public void maybeRunAsyncAction(ClusterState currentState, final IndexMetaData indexMetaData, final String policy, Step.StepKey expectedStepKey) {
        final String index = indexMetaData.getIndex().getName();
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)indexMetaData);
        final Step currentStep = IndexLifecycleRunner.getCurrentStep(this.stepRegistry, policy, indexMetaData, lifecycleState);
        if (currentStep == null) {
            logger.warn("current step [{}] for index [{}] with policy [{}] is not recognized", (Object)IndexLifecycleRunner.getCurrentStepKey(lifecycleState), (Object)index, (Object)policy);
            return;
        }
        logger.trace("[{}] maybe running async action step ({}) with current step {}", (Object)index, (Object)currentStep.getClass().getSimpleName(), (Object)currentStep.getKey());
        if (!currentStep.getKey().equals((Object)expectedStepKey)) {
            throw new IllegalStateException("expected index [" + indexMetaData.getIndex().getName() + "] with policy [" + policy + "] to have current step consistent with provided step key (" + expectedStepKey + ") but it was " + currentStep.getKey());
        }
        if (currentStep instanceof AsyncActionStep) {
            logger.debug("[{}] running policy with async action step [{}]", (Object)index, (Object)currentStep.getKey());
            ((AsyncActionStep)currentStep).performAction(indexMetaData, currentState, new AsyncActionStep.Listener(){

                public void onResponse(boolean complete) {
                    logger.trace("cs-change-async-action-callback, [{}], current-step: {}", (Object)index, (Object)currentStep.getKey());
                    if (complete && ((AsyncActionStep)currentStep).indexSurvives()) {
                        IndexLifecycleRunner.this.moveToStep(indexMetaData.getIndex(), policy, currentStep.getKey(), currentStep.getNextStepKey());
                    }
                }

                public void onFailure(Exception e) {
                    IndexLifecycleRunner.this.moveToErrorStep(indexMetaData.getIndex(), policy, currentStep.getKey(), e);
                }
            });
        } else {
            logger.trace("[{}] ignoring non async action step execution from step transition [{}]", (Object)index, (Object)currentStep.getKey());
        }
    }

    public void runPolicyAfterStateChange(String policy, IndexMetaData indexMetaData) {
        String index = indexMetaData.getIndex().getName();
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)indexMetaData);
        Step currentStep = IndexLifecycleRunner.getCurrentStep(this.stepRegistry, policy, indexMetaData, lifecycleState);
        if (currentStep == null) {
            if (!this.stepRegistry.policyExists(policy)) {
                this.markPolicyDoesNotExist(policy, indexMetaData.getIndex(), lifecycleState);
                return;
            }
            logger.error("current step [{}] for index [{}] with policy [{}] is not recognized", (Object)IndexLifecycleRunner.getCurrentStepKey(lifecycleState), (Object)index, (Object)policy);
            return;
        }
        if (currentStep instanceof TerminalPolicyStep) {
            logger.debug("policy [{}] for index [{}] complete, skipping execution", (Object)policy, (Object)index);
            return;
        }
        if (currentStep instanceof ErrorStep) {
            logger.debug("policy [{}] for index [{}] on an error step, skipping execution", (Object)policy, (Object)index);
            return;
        }
        logger.trace("[{}] maybe running step ({}) after state change: {}", (Object)index, (Object)currentStep.getClass().getSimpleName(), (Object)currentStep.getKey());
        if (currentStep instanceof PhaseCompleteStep) {
            if (this.isReadyToTransitionToThisPhase(policy, indexMetaData, currentStep.getNextStepKey().getPhase())) {
                this.moveToStep(indexMetaData.getIndex(), policy, currentStep.getKey(), currentStep.getNextStepKey());
            }
        } else if (currentStep instanceof ClusterStateActionStep || currentStep instanceof ClusterStateWaitStep) {
            logger.debug("[{}] running policy with current-step [{}]", (Object)indexMetaData.getIndex().getName(), (Object)currentStep.getKey());
            this.clusterService.submitStateUpdateTask("ilm-execute-cluster-state-steps", (ClusterStateTaskConfig)new ExecuteStepsUpdateTask(policy, indexMetaData.getIndex(), currentStep, this.stepRegistry, this, this.nowSupplier));
        } else {
            logger.trace("[{}] ignoring step execution from cluster state change event [{}]", (Object)index, (Object)currentStep.getKey());
        }
    }

    public static Step.StepKey getCurrentStepKey(LifecycleExecutionState lifecycleState) {
        String currentPhase = lifecycleState.getPhase();
        String currentAction = lifecycleState.getAction();
        String currentStep = lifecycleState.getStep();
        if (Strings.isNullOrEmpty((String)currentStep)) {
            assert (Strings.isNullOrEmpty((String)currentPhase)) : "Current phase is not empty: " + currentPhase;
            assert (Strings.isNullOrEmpty((String)currentAction)) : "Current action is not empty: " + currentAction;
            return null;
        }
        assert (!Strings.isNullOrEmpty((String)currentPhase));
        assert (!Strings.isNullOrEmpty((String)currentAction));
        return new Step.StepKey(currentPhase, currentAction, currentStep);
    }

    static Step getCurrentStep(PolicyStepsRegistry stepRegistry, String policy, IndexMetaData indexMetaData, LifecycleExecutionState lifecycleState) {
        Step.StepKey currentStepKey = IndexLifecycleRunner.getCurrentStepKey(lifecycleState);
        logger.trace("[{}] retrieved current step key: {}", (Object)indexMetaData.getIndex().getName(), (Object)currentStepKey);
        if (currentStepKey == null) {
            return stepRegistry.getFirstStep(policy);
        }
        return stepRegistry.getStep(indexMetaData, currentStepKey);
    }

    static ClusterState moveClusterStateToStep(String indexName, ClusterState currentState, Step.StepKey currentStepKey, Step.StepKey nextStepKey, LongSupplier nowSupplier, PolicyStepsRegistry stepRegistry, boolean forcePhaseDefinitionRefresh) {
        IndexMetaData idxMeta = currentState.getMetaData().index(indexName);
        Settings indexSettings = idxMeta.getSettings();
        String indexPolicySetting = (String)LifecycleSettings.LIFECYCLE_NAME_SETTING.get(indexSettings);
        if (Strings.isNullOrEmpty((String)indexPolicySetting)) {
            throw new IllegalArgumentException("index [" + indexName + "] is not associated with an Index Lifecycle Policy");
        }
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)idxMeta);
        if (!currentStepKey.equals((Object)IndexLifecycleRunner.getCurrentStepKey(lifecycleState))) {
            throw new IllegalArgumentException("index [" + indexName + "] is not on current step [" + currentStepKey + "]");
        }
        if (!stepRegistry.stepExists(indexPolicySetting, nextStepKey)) {
            throw new IllegalArgumentException("step [" + nextStepKey + "] for index [" + idxMeta.getIndex().getName() + "] with policy [" + indexPolicySetting + "] does not exist");
        }
        return IndexLifecycleRunner.moveClusterStateToNextStep(idxMeta.getIndex(), currentState, currentStepKey, nextStepKey, nowSupplier, forcePhaseDefinitionRefresh);
    }

    static ClusterState moveClusterStateToNextStep(Index index, ClusterState clusterState, Step.StepKey currentStep, Step.StepKey nextStep, LongSupplier nowSupplier, boolean forcePhaseDefinitionRefresh) {
        IndexMetaData idxMeta = clusterState.getMetaData().index(index);
        IndexLifecycleMetadata ilmMeta = (IndexLifecycleMetadata)clusterState.metaData().custom("index_lifecycle");
        LifecyclePolicyMetadata policyMetadata = (LifecyclePolicyMetadata)ilmMeta.getPolicyMetadatas().get(LifecycleSettings.LIFECYCLE_NAME_SETTING.get(idxMeta.getSettings()));
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)idxMeta);
        LifecycleExecutionState newLifecycleState = IndexLifecycleRunner.moveExecutionStateToNextStep(policyMetadata, lifecycleState, currentStep, nextStep, nowSupplier, forcePhaseDefinitionRefresh);
        ClusterState.Builder newClusterStateBuilder = IndexLifecycleRunner.newClusterStateWithLifecycleState(index, clusterState, newLifecycleState);
        return newClusterStateBuilder.build();
    }

    static ClusterState moveClusterStateToErrorStep(Index index, ClusterState clusterState, Step.StepKey currentStep, Exception cause, LongSupplier nowSupplier) throws IOException {
        IndexMetaData idxMeta = clusterState.getMetaData().index(index);
        IndexLifecycleMetadata ilmMeta = (IndexLifecycleMetadata)clusterState.metaData().custom("index_lifecycle");
        LifecyclePolicyMetadata policyMetadata = (LifecyclePolicyMetadata)ilmMeta.getPolicyMetadatas().get(LifecycleSettings.LIFECYCLE_NAME_SETTING.get(idxMeta.getSettings()));
        XContentBuilder causeXContentBuilder = JsonXContent.contentBuilder();
        causeXContentBuilder.startObject();
        ElasticsearchException.generateThrowableXContent((XContentBuilder)causeXContentBuilder, (ToXContent.Params)STACKTRACE_PARAMS, (Throwable)cause);
        causeXContentBuilder.endObject();
        LifecycleExecutionState nextStepState = IndexLifecycleRunner.moveExecutionStateToNextStep(policyMetadata, LifecycleExecutionState.fromIndexMetadata((IndexMetaData)idxMeta), currentStep, new Step.StepKey(currentStep.getPhase(), currentStep.getAction(), "ERROR"), nowSupplier, false);
        LifecycleExecutionState.Builder failedState = LifecycleExecutionState.builder((LifecycleExecutionState)nextStepState);
        failedState.setFailedStep(currentStep.getName());
        failedState.setStepInfo(BytesReference.bytes((XContentBuilder)causeXContentBuilder).utf8ToString());
        ClusterState.Builder newClusterStateBuilder = IndexLifecycleRunner.newClusterStateWithLifecycleState(index, clusterState, failedState.build());
        return newClusterStateBuilder.build();
    }

    ClusterState moveClusterStateToFailedStep(ClusterState currentState, String[] indices) {
        ClusterState newState = currentState;
        for (String index : indices) {
            IndexMetaData indexMetaData = currentState.metaData().index(index);
            if (indexMetaData == null) {
                throw new IllegalArgumentException("index [" + index + "] does not exist");
            }
            LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)indexMetaData);
            Step.StepKey currentStepKey = IndexLifecycleRunner.getCurrentStepKey(lifecycleState);
            String failedStep = lifecycleState.getFailedStep();
            if (currentStepKey == null || !"ERROR".equals(currentStepKey.getName()) || Strings.isNullOrEmpty((String)failedStep)) {
                throw new IllegalArgumentException("cannot retry an action for an index [" + index + "] that has not encountered an error when running a Lifecycle Policy");
            }
            Step.StepKey nextStepKey = new Step.StepKey(currentStepKey.getPhase(), currentStepKey.getAction(), failedStep);
            newState = IndexLifecycleRunner.moveClusterStateToStep(index, currentState, currentStepKey, nextStepKey, this.nowSupplier, this.stepRegistry, true);
        }
        return newState;
    }

    private static LifecycleExecutionState moveExecutionStateToNextStep(LifecyclePolicyMetadata policyMetadata, LifecycleExecutionState existingState, Step.StepKey currentStep, Step.StepKey nextStep, LongSupplier nowSupplier, boolean forcePhaseDefinitionRefresh) {
        long nowAsMillis = nowSupplier.getAsLong();
        LifecycleExecutionState.Builder updatedState = LifecycleExecutionState.builder((LifecycleExecutionState)existingState);
        updatedState.setPhase(nextStep.getPhase());
        updatedState.setAction(nextStep.getAction());
        updatedState.setStep(nextStep.getName());
        updatedState.setStepTime(Long.valueOf(nowAsMillis));
        updatedState.setFailedStep(null);
        updatedState.setStepInfo(null);
        if (!currentStep.getPhase().equals(nextStep.getPhase()) || forcePhaseDefinitionRefresh) {
            Phase nextPhase = "new".equals(nextStep.getPhase()) || TerminalPolicyStep.KEY.equals((Object)nextStep) ? null : (Phase)policyMetadata.getPolicy().getPhases().get(nextStep.getPhase());
            PhaseExecutionInfo phaseExecutionInfo = new PhaseExecutionInfo(policyMetadata.getName(), nextPhase, policyMetadata.getVersion(), policyMetadata.getModifiedDate());
            String newPhaseDefinition = Strings.toString((ToXContent)phaseExecutionInfo, (boolean)false, (boolean)false);
            updatedState.setPhaseDefinition(newPhaseDefinition);
            updatedState.setPhaseTime(Long.valueOf(nowAsMillis));
        } else if (currentStep.getPhase().equals("new")) {
            updatedState.setPhaseTime(Long.valueOf(nowAsMillis));
        }
        if (!currentStep.getAction().equals(nextStep.getAction())) {
            updatedState.setActionTime(Long.valueOf(nowAsMillis));
        }
        return updatedState.build();
    }

    static ClusterState.Builder newClusterStateWithLifecycleState(Index index, ClusterState clusterState, LifecycleExecutionState lifecycleState) {
        ClusterState.Builder newClusterStateBuilder = ClusterState.builder((ClusterState)clusterState);
        newClusterStateBuilder.metaData(MetaData.builder((MetaData)clusterState.getMetaData()).put(IndexMetaData.builder((IndexMetaData)clusterState.getMetaData().index(index)).putCustom("ilm", lifecycleState.asMap())));
        return newClusterStateBuilder;
    }

    static ClusterState addStepInfoToClusterState(Index index, ClusterState clusterState, ToXContentObject stepInfo) throws IOException {
        String stepInfoString;
        IndexMetaData indexMetaData = clusterState.getMetaData().index(index);
        if (indexMetaData == null) {
            return clusterState;
        }
        LifecycleExecutionState lifecycleState = LifecycleExecutionState.fromIndexMetadata((IndexMetaData)indexMetaData);
        try (XContentBuilder infoXContentBuilder = JsonXContent.contentBuilder();){
            stepInfo.toXContent(infoXContentBuilder, ToXContent.EMPTY_PARAMS);
            stepInfoString = BytesReference.bytes((XContentBuilder)infoXContentBuilder).utf8ToString();
        }
        if (stepInfoString.equals(lifecycleState.getStepInfo())) {
            return clusterState;
        }
        LifecycleExecutionState.Builder newState = LifecycleExecutionState.builder((LifecycleExecutionState)lifecycleState);
        newState.setStepInfo(stepInfoString);
        ClusterState.Builder newClusterStateBuilder = IndexLifecycleRunner.newClusterStateWithLifecycleState(index, clusterState, newState.build());
        return newClusterStateBuilder.build();
    }

    private void moveToStep(Index index, String policy, Step.StepKey currentStepKey, Step.StepKey nextStepKey) {
        logger.debug("[{}] moving to step [{}] {} -> {}", (Object)index.getName(), (Object)policy, (Object)currentStepKey, (Object)nextStepKey);
        this.clusterService.submitStateUpdateTask("ilm-move-to-step", (ClusterStateTaskConfig)new MoveToNextStepUpdateTask(index, policy, currentStepKey, nextStepKey, this.nowSupplier, clusterState -> {
            IndexMetaData indexMetaData = clusterState.metaData().index(index);
            if (nextStepKey != null && nextStepKey != TerminalPolicyStep.KEY && indexMetaData != null) {
                this.maybeRunAsyncAction((ClusterState)clusterState, indexMetaData, policy, nextStepKey);
            }
        }));
    }

    private void moveToErrorStep(Index index, String policy, Step.StepKey currentStepKey, Exception e) {
        logger.error((Message)new ParameterizedMessage("policy [{}] for index [{}] failed on step [{}]. Moving to ERROR step", new Object[]{policy, index.getName(), currentStepKey}), (Throwable)e);
        this.clusterService.submitStateUpdateTask("ilm-move-to-error-step", (ClusterStateTaskConfig)new MoveToErrorStepUpdateTask(index, policy, currentStepKey, e, this.nowSupplier));
    }

    private void setStepInfo(Index index, String policy, Step.StepKey currentStepKey, ToXContentObject stepInfo) {
        this.clusterService.submitStateUpdateTask("ilm-set-step-info", (ClusterStateTaskConfig)new SetStepInfoUpdateTask(index, policy, currentStepKey, stepInfo));
    }

    public static ClusterState removePolicyForIndexes(Index[] indices, ClusterState currentState, List<String> failedIndexes) {
        MetaData.Builder newMetadata = MetaData.builder((MetaData)currentState.getMetaData());
        boolean clusterStateChanged = false;
        for (Index index : indices) {
            IndexMetaData indexMetadata = currentState.getMetaData().index(index);
            if (indexMetadata == null) {
                failedIndexes.add(index.getName());
                continue;
            }
            IndexMetaData.Builder newIdxMetadata = IndexLifecycleRunner.removePolicyForIndex(indexMetadata);
            if (newIdxMetadata == null) continue;
            newMetadata.put(newIdxMetadata);
            clusterStateChanged = true;
        }
        if (clusterStateChanged) {
            ClusterState.Builder newClusterState = ClusterState.builder((ClusterState)currentState);
            newClusterState.metaData(newMetadata);
            return newClusterState.build();
        }
        return currentState;
    }

    private static IndexMetaData.Builder removePolicyForIndex(IndexMetaData indexMetadata) {
        Settings idxSettings = indexMetadata.getSettings();
        Settings.Builder newSettings = Settings.builder().put(idxSettings);
        boolean notChanged = true;
        notChanged &= Strings.isNullOrEmpty((String)newSettings.remove(LifecycleSettings.LIFECYCLE_NAME_SETTING.getKey()));
        long newSettingsVersion = (notChanged &= Strings.isNullOrEmpty((String)newSettings.remove(RolloverAction.LIFECYCLE_ROLLOVER_ALIAS_SETTING.getKey()))) ? indexMetadata.getSettingsVersion() : 1L + indexMetadata.getSettingsVersion();
        IndexMetaData.Builder builder = IndexMetaData.builder((IndexMetaData)indexMetadata);
        builder.removeCustom("ilm");
        return builder.settings(newSettings).settingsVersion(newSettingsVersion);
    }

    private void markPolicyDoesNotExist(String policyName, Index index, LifecycleExecutionState executionState) {
        logger.debug("policy [{}] for index [{}] does not exist, recording this in step_info for this index", (Object)policyName, (Object)index.getName());
        this.setStepInfo(index, policyName, IndexLifecycleRunner.getCurrentStepKey(executionState), new SetStepInfoUpdateTask.ExceptionWrapper(new IllegalArgumentException("policy [" + policyName + "] does not exist")));
    }
}

