/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.storedscripts.DeleteStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.GetStoredScriptRequest;
import org.elasticsearch.action.admin.cluster.storedscripts.PutStoredScriptRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.breaker.CircuitBreaker;
import org.elasticsearch.common.breaker.CircuitBreakingException;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.script.GeneralScriptException;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptContext;
import org.elasticsearch.script.ScriptEngine;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.ScriptMetaData;
import org.elasticsearch.script.ScriptMetrics;
import org.elasticsearch.script.ScriptStats;
import org.elasticsearch.script.ScriptType;
import org.elasticsearch.script.StoredScriptSource;

public class ScriptService
implements Closeable,
ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(ScriptService.class);
    static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    static final Function<String, Tuple<Integer, TimeValue>> MAX_COMPILATION_RATE_FUNCTION = value -> {
        if (!value.contains("/") || value.startsWith("/") || value.endsWith("/")) {
            throw new IllegalArgumentException("parameter must contain a positive integer and a timevalue, i.e. 10/1m, but was [" + value + "]");
        }
        int idx = value.indexOf("/");
        String count = value.substring(0, idx);
        String time = value.substring(idx + 1);
        try {
            int rate = Integer.parseInt(count);
            if (rate < 0) {
                throw new IllegalArgumentException("rate [" + rate + "] must be positive");
            }
            TimeValue timeValue = TimeValue.parseTimeValue((String)time, (String)"script.max_compilations_rate");
            if (timeValue.nanos() <= 0L) {
                throw new IllegalArgumentException("time value [" + time + "] must be positive");
            }
            if (timeValue.seconds() < 60L) {
                throw new IllegalArgumentException("time value [" + time + "] must be at least on a one minute resolution");
            }
            return Tuple.tuple((Object)rate, (Object)timeValue);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("could not parse [" + count + "] as integer in value [" + value + "]", e);
        }
    };
    public static final Setting<Integer> SCRIPT_CACHE_SIZE_SETTING = Setting.intSetting("script.cache.max_size", 100, 0, Setting.Property.NodeScope);
    public static final Setting<TimeValue> SCRIPT_CACHE_EXPIRE_SETTING = Setting.positiveTimeSetting("script.cache.expire", TimeValue.timeValueMillis((long)0L), Setting.Property.NodeScope);
    public static final Setting<Integer> SCRIPT_MAX_SIZE_IN_BYTES = Setting.intSetting("script.max_size_in_bytes", 65535, 0, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Tuple<Integer, TimeValue>> SCRIPT_MAX_COMPILATIONS_RATE = new Setting<Tuple<Integer, TimeValue>>("script.max_compilations_rate", "75/5m", MAX_COMPILATION_RATE_FUNCTION, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final String ALLOW_NONE = "none";
    public static final Setting<List<String>> TYPES_ALLOWED_SETTING = Setting.listSetting("script.allowed_types", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);
    public static final Setting<List<String>> CONTEXTS_ALLOWED_SETTING = Setting.listSetting("script.allowed_contexts", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);
    private final Settings settings;
    private final Set<String> typesAllowed;
    private final Set<String> contextsAllowed;
    private final Map<String, ScriptEngine> engines;
    private final Map<String, ScriptContext<?>> contexts;
    private final Cache<CacheKey, Object> cache;
    private final ScriptMetrics scriptMetrics = new ScriptMetrics();
    private ClusterState clusterState;
    private int maxSizeInBytes;
    private Tuple<Integer, TimeValue> rate;
    private long lastInlineCompileTime;
    private double scriptsPerTimeWindow;
    private double compilesAllowedPerNano;

    public ScriptService(Settings settings, Map<String, ScriptEngine> engines, Map<String, ScriptContext<?>> contexts) {
        TimeValue cacheExpire;
        this.settings = Objects.requireNonNull(settings);
        this.engines = Objects.requireNonNull(engines);
        this.contexts = Objects.requireNonNull(contexts);
        if (Strings.hasLength(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING))) {
            throw new IllegalArgumentException("script.disable_dynamic is not a supported setting, replace with fine-grained script settings. \n Dynamic scripts can be enabled for all languages and all operations not using `script.disable_dynamic: false` in elasticsearch.yml");
        }
        HashSet hashSet = this.typesAllowed = TYPES_ALLOWED_SETTING.exists(settings) ? new HashSet() : null;
        if (this.typesAllowed != null) {
            List<String> typesAllowedList = TYPES_ALLOWED_SETTING.get(settings);
            if (typesAllowedList.isEmpty()) {
                throw new IllegalArgumentException("must specify at least one script type or none for setting [" + TYPES_ALLOWED_SETTING.getKey() + "].");
            }
            for (String settingType : typesAllowedList) {
                if (ALLOW_NONE.equals(settingType)) {
                    if (typesAllowedList.size() == 1) break;
                    throw new IllegalArgumentException("cannot specify both [none] and other script types for setting [" + TYPES_ALLOWED_SETTING.getKey() + "].");
                }
                boolean found = false;
                for (ScriptType scriptType : ScriptType.values()) {
                    if (!scriptType.getName().equals(settingType)) continue;
                    found = true;
                    this.typesAllowed.add(settingType);
                    break;
                }
                if (found) continue;
                throw new IllegalArgumentException("unknown script type [" + settingType + "] found in setting [" + TYPES_ALLOWED_SETTING.getKey() + "].");
            }
        }
        HashSet hashSet2 = this.contextsAllowed = CONTEXTS_ALLOWED_SETTING.exists(settings) ? new HashSet() : null;
        if (this.contextsAllowed != null) {
            List<String> contextsAllowedList = CONTEXTS_ALLOWED_SETTING.get(settings);
            if (contextsAllowedList.isEmpty()) {
                throw new IllegalArgumentException("must specify at least one script context or none for setting [" + CONTEXTS_ALLOWED_SETTING.getKey() + "].");
            }
            for (String settingContext : contextsAllowedList) {
                if (ALLOW_NONE.equals(settingContext)) {
                    if (contextsAllowedList.size() == 1) break;
                    throw new IllegalArgumentException("cannot specify both [none] and other script contexts for setting [" + CONTEXTS_ALLOWED_SETTING.getKey() + "].");
                }
                if (contexts.containsKey(settingContext)) {
                    this.contextsAllowed.add(settingContext);
                    continue;
                }
                throw new IllegalArgumentException("unknown script context [" + settingContext + "] found in setting [" + CONTEXTS_ALLOWED_SETTING.getKey() + "].");
            }
        }
        int cacheMaxSize = SCRIPT_CACHE_SIZE_SETTING.get(settings);
        CacheBuilder<CacheKey, Object> cacheBuilder = CacheBuilder.builder();
        if (cacheMaxSize >= 0) {
            cacheBuilder.setMaximumWeight(cacheMaxSize);
        }
        if ((cacheExpire = SCRIPT_CACHE_EXPIRE_SETTING.get(settings)).getNanos() != 0L) {
            cacheBuilder.setExpireAfterAccess(cacheExpire);
        }
        logger.debug("using script cache with max_size [{}], expire [{}]", (Object)cacheMaxSize, (Object)cacheExpire);
        this.cache = cacheBuilder.removalListener(new ScriptCacheRemovalListener()).build();
        this.lastInlineCompileTime = System.nanoTime();
        this.setMaxSizeInBytes(SCRIPT_MAX_SIZE_IN_BYTES.get(settings));
        this.setMaxCompilationRate(SCRIPT_MAX_COMPILATIONS_RATE.get(settings));
    }

    void registerClusterSettingsListeners(ClusterSettings clusterSettings) {
        clusterSettings.addSettingsUpdateConsumer(SCRIPT_MAX_SIZE_IN_BYTES, this::setMaxSizeInBytes);
        clusterSettings.addSettingsUpdateConsumer(SCRIPT_MAX_COMPILATIONS_RATE, this::setMaxCompilationRate);
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.engines.values());
    }

    private ScriptEngine getEngine(String lang) {
        ScriptEngine scriptEngine = this.engines.get(lang);
        if (scriptEngine == null) {
            throw new IllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        return scriptEngine;
    }

    void setMaxSizeInBytes(int newMaxSizeInBytes) {
        for (Map.Entry<String, StoredScriptSource> source : this.getScriptsFromClusterState().entrySet()) {
            if (source.getValue().getSource().getBytes(StandardCharsets.UTF_8).length <= newMaxSizeInBytes) continue;
            throw new IllegalArgumentException("script.max_size_in_bytes cannot be set to [" + newMaxSizeInBytes + "], stored script [" + source.getKey() + "] exceeds the new value with a size of [" + source.getValue().getSource().getBytes(StandardCharsets.UTF_8).length + "]");
        }
        this.maxSizeInBytes = newMaxSizeInBytes;
    }

    void setMaxCompilationRate(Tuple<Integer, TimeValue> newRate) {
        this.rate = newRate;
        this.scriptsPerTimeWindow = ((Integer)this.rate.v1()).intValue();
        this.compilesAllowedPerNano = (double)((Integer)this.rate.v1()).intValue() / (double)((TimeValue)newRate.v2()).nanos();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <FactoryType> FactoryType compile(Script script, ScriptContext<FactoryType> context) {
        CacheKey cacheKey;
        Object compiledScript;
        Objects.requireNonNull(script);
        Objects.requireNonNull(context);
        ScriptType type = script.getType();
        String lang = script.getLang();
        String idOrCode = script.getIdOrCode();
        Map<String, String> options = script.getOptions();
        String id = idOrCode;
        if (type == ScriptType.STORED) {
            StoredScriptSource source = this.getScriptFromClusterState(id);
            lang = source.getLang();
            idOrCode = source.getSource();
            options = source.getOptions();
        }
        ScriptEngine scriptEngine = this.getEngine(lang);
        if (!this.isTypeEnabled(type)) {
            throw new IllegalArgumentException("cannot execute [" + type + "] scripts");
        }
        if (!this.contexts.containsKey(context.name)) {
            throw new IllegalArgumentException("script context [" + context.name + "] not supported");
        }
        if (!this.isContextEnabled(context)) {
            throw new IllegalArgumentException("cannot execute scripts using [" + context.name + "] context");
        }
        if (type == ScriptType.INLINE && idOrCode.getBytes(StandardCharsets.UTF_8).length > this.maxSizeInBytes) {
            throw new IllegalArgumentException("exceeded max allowed inline script size in bytes [" + this.maxSizeInBytes + "] with size [" + idOrCode.getBytes(StandardCharsets.UTF_8).length + "] for script [" + idOrCode + "]");
        }
        if (logger.isTraceEnabled()) {
            logger.trace("compiling lang: [{}] type: [{}] script: {}", (Object)lang, (Object)type, (Object)idOrCode);
        }
        if ((compiledScript = this.cache.get(cacheKey = new CacheKey(lang, idOrCode, context.name, options))) != null) {
            return context.factoryClazz.cast(compiledScript);
        }
        ScriptService scriptService = this;
        synchronized (scriptService) {
            compiledScript = this.cache.get(cacheKey);
            if (compiledScript == null) {
                try {
                    if (logger.isTraceEnabled()) {
                        logger.trace("compiling script, type: [{}], lang: [{}], options: [{}]", (Object)type, (Object)lang, options);
                    }
                    this.checkCompilationLimit();
                    compiledScript = scriptEngine.compile(id, idOrCode, context, options);
                }
                catch (ScriptException good) {
                    throw good;
                }
                catch (Exception exception) {
                    throw new GeneralScriptException("Failed to compile " + type + " script [" + id + "] using lang [" + lang + "]", exception);
                }
                this.scriptMetrics.onCompilation();
                this.cache.put(cacheKey, compiledScript);
            }
            return context.factoryClazz.cast(compiledScript);
        }
    }

    void checkCompilationLimit() {
        long now = System.nanoTime();
        long timePassed = now - this.lastInlineCompileTime;
        this.lastInlineCompileTime = now;
        this.scriptsPerTimeWindow += (double)timePassed * this.compilesAllowedPerNano;
        if (this.scriptsPerTimeWindow > (double)((Integer)this.rate.v1()).intValue()) {
            this.scriptsPerTimeWindow = ((Integer)this.rate.v1()).intValue();
        }
        if (this.scriptsPerTimeWindow >= 1.0) {
            this.scriptsPerTimeWindow -= 1.0;
        } else {
            this.scriptMetrics.onCompilationLimit();
            throw new CircuitBreakingException("[script] Too many dynamic script compilations within, max: [" + this.rate.v1() + "/" + this.rate.v2() + "]; please use indexed, or scripts with parameters instead; this limit can be changed by the [" + SCRIPT_MAX_COMPILATIONS_RATE.getKey() + "] setting", CircuitBreaker.Durability.TRANSIENT);
        }
    }

    public boolean isLangSupported(String lang) {
        Objects.requireNonNull(lang);
        return this.engines.containsKey(lang);
    }

    public boolean isTypeEnabled(ScriptType scriptType) {
        return this.typesAllowed == null || this.typesAllowed.contains(scriptType.getName());
    }

    public boolean isContextEnabled(ScriptContext scriptContext) {
        return this.contextsAllowed == null || this.contextsAllowed.contains(scriptContext.name);
    }

    public boolean isAnyContextEnabled() {
        return this.contextsAllowed == null || !this.contextsAllowed.isEmpty();
    }

    Map<String, StoredScriptSource> getScriptsFromClusterState() {
        if (this.clusterState == null) {
            return Collections.emptyMap();
        }
        ScriptMetaData scriptMetadata = (ScriptMetaData)this.clusterState.metaData().custom("stored_scripts");
        if (scriptMetadata == null) {
            return Collections.emptyMap();
        }
        return scriptMetadata.getStoredScripts();
    }

    StoredScriptSource getScriptFromClusterState(String id) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)this.clusterState.metaData().custom("stored_scripts");
        if (scriptMetadata == null) {
            throw new ResourceNotFoundException("unable to find script [" + id + "] in cluster state", new Object[0]);
        }
        StoredScriptSource source = scriptMetadata.getStoredScript(id);
        if (source == null) {
            throw new ResourceNotFoundException("unable to find script [" + id + "] in cluster state", new Object[0]);
        }
        return source;
    }

    public void putStoredScript(ClusterService clusterService, final PutStoredScriptRequest request, ActionListener<AcknowledgedResponse> listener) {
        if (request.content().length() > this.maxSizeInBytes) {
            throw new IllegalArgumentException("exceeded max allowed stored script size in bytes [" + this.maxSizeInBytes + "] with size [" + request.content().length() + "] for script [" + request.id() + "]");
        }
        final StoredScriptSource source = request.source();
        if (!this.isLangSupported(source.getLang())) {
            throw new IllegalArgumentException("unable to put stored script with unsupported lang [" + source.getLang() + "]");
        }
        try {
            ScriptEngine scriptEngine = this.getEngine(source.getLang());
            if (!this.isTypeEnabled(ScriptType.STORED)) {
                throw new IllegalArgumentException("cannot put [" + ScriptType.STORED + "] script, [" + ScriptType.STORED + "] scripts are not enabled");
            }
            if (!this.isAnyContextEnabled()) {
                throw new IllegalArgumentException("cannot put [" + ScriptType.STORED + "] script, no script contexts are enabled");
            }
            if (request.context() != null) {
                ScriptContext<?> context = this.contexts.get(request.context());
                if (context == null) {
                    throw new IllegalArgumentException("Unknown context [" + request.context() + "]");
                }
                scriptEngine.compile(request.id(), source.getSource(), context, Collections.emptyMap());
            }
        }
        catch (ScriptException good) {
            throw good;
        }
        catch (Exception exception) {
            throw new IllegalArgumentException("failed to parse/compile stored script [" + request.id() + "]", exception);
        }
        clusterService.submitStateUpdateTask("put-script-" + request.id(), new AckedClusterStateUpdateTask<AcknowledgedResponse>((AckedRequest)request, listener){

            @Override
            protected AcknowledgedResponse newResponse(boolean acknowledged) {
                return new AcknowledgedResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ScriptMetaData smd = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
                smd = ScriptMetaData.putStoredScript(smd, request.id(), source);
                MetaData.Builder mdb = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", smd);
                return ClusterState.builder(currentState).metaData(mdb).build();
            }
        });
    }

    public void deleteStoredScript(ClusterService clusterService, final DeleteStoredScriptRequest request, ActionListener<AcknowledgedResponse> listener) {
        clusterService.submitStateUpdateTask("delete-script-" + request.id(), new AckedClusterStateUpdateTask<AcknowledgedResponse>((AckedRequest)request, listener){

            @Override
            protected AcknowledgedResponse newResponse(boolean acknowledged) {
                return new AcknowledgedResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ScriptMetaData smd = (ScriptMetaData)currentState.metaData().custom("stored_scripts");
                smd = ScriptMetaData.deleteStoredScript(smd, request.id());
                MetaData.Builder mdb = MetaData.builder(currentState.getMetaData()).putCustom("stored_scripts", smd);
                return ClusterState.builder(currentState).metaData(mdb).build();
            }
        });
    }

    public StoredScriptSource getStoredScript(ClusterState state, GetStoredScriptRequest request) {
        ScriptMetaData scriptMetadata = (ScriptMetaData)state.metaData().custom("stored_scripts");
        if (scriptMetadata != null) {
            return scriptMetadata.getStoredScript(request.id());
        }
        return null;
    }

    public ScriptStats stats() {
        return this.scriptMetrics.stats();
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        this.clusterState = event.state();
    }

    private static final class CacheKey {
        final String lang;
        final String idOrCode;
        final String context;
        final Map<String, String> options;

        private CacheKey(String lang, String idOrCode, String context, Map<String, String> options) {
            this.lang = lang;
            this.idOrCode = idOrCode;
            this.context = context;
            this.options = options;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            return Objects.equals(this.lang, cacheKey.lang) && Objects.equals(this.idOrCode, cacheKey.idOrCode) && Objects.equals(this.context, cacheKey.context) && Objects.equals(this.options, cacheKey.options);
        }

        public int hashCode() {
            return Objects.hash(this.lang, this.idOrCode, this.context, this.options);
        }
    }

    private class ScriptCacheRemovalListener
    implements RemovalListener<CacheKey, Object> {
        private ScriptCacheRemovalListener() {
        }

        @Override
        public void onRemoval(RemovalNotification<CacheKey, Object> notification) {
            if (logger.isDebugEnabled()) {
                logger.debug("removed {} from cache, reason: {}", notification.getValue(), (Object)notification.getRemovalReason());
            }
            ScriptService.this.scriptMetrics.onCacheEviction();
        }
    }
}

