/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.sql.expression.literal;

import java.time.Duration;
import java.time.Period;
import java.time.temporal.TemporalAmount;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.io.stream.NamedWriteableRegistry;
import org.elasticsearch.xpack.sql.SqlIllegalArgumentException;
import org.elasticsearch.xpack.sql.expression.Foldables;
import org.elasticsearch.xpack.sql.expression.Literal;
import org.elasticsearch.xpack.sql.expression.literal.Interval;
import org.elasticsearch.xpack.sql.expression.literal.IntervalDayTime;
import org.elasticsearch.xpack.sql.expression.literal.IntervalYearMonth;
import org.elasticsearch.xpack.sql.parser.ParsingException;
import org.elasticsearch.xpack.sql.tree.Location;
import org.elasticsearch.xpack.sql.type.DataType;
import org.elasticsearch.xpack.sql.util.Check;
import org.elasticsearch.xpack.sql.util.StringUtils;

public final class Intervals {
    static final Map<DataType, Parser> PARSERS = new LinkedHashMap<DataType, Parser>();

    private Intervals() {
    }

    public static long inMillis(Literal literal) {
        Object fold = Foldables.valueOf(literal);
        Check.isTrue(fold instanceof Interval, "Expected interval, received [{}]", fold);
        Object interval = ((Interval)fold).interval();
        long millis = 0L;
        if (interval instanceof Period) {
            Period p = (Period)interval;
            millis = p.toTotalMonths() * 30L * 24L * 60L * 60L * 1000L;
        } else {
            Duration d = (Duration)interval;
            millis = d.toMillis();
        }
        return millis;
    }

    public static TemporalAmount of(Location source, long duration, TimeUnit unit) {
        try {
            switch (unit) {
                case YEAR: {
                    return Period.ZERO.plusYears(duration);
                }
                case MONTH: {
                    return Period.ZERO.plusMonths(duration);
                }
                case DAY: {
                    return Duration.ZERO.plusDays(duration);
                }
                case HOUR: {
                    return Duration.ZERO.plusHours(duration);
                }
                case MINUTE: {
                    return Duration.ZERO.plusMinutes(duration);
                }
                case SECOND: {
                    return Duration.ZERO.plusSeconds(duration);
                }
                case MILLISECOND: {
                    return Duration.ZERO.plusMillis(duration);
                }
            }
            throw new ParsingException(source, "Cannot parse duration [{}]", new Object[]{unit});
        }
        catch (ArithmeticException ae) {
            throw new ParsingException(source, "Value [{}] cannot be used as it is too large to convert into [{}]s", new Object[]{duration, unit});
        }
    }

    public static DataType intervalType(Location source, TimeUnit leading, TimeUnit trailing) {
        if (trailing == null) {
            switch (leading) {
                case YEAR: {
                    return DataType.INTERVAL_YEAR;
                }
                case MONTH: {
                    return DataType.INTERVAL_MONTH;
                }
                case DAY: {
                    return DataType.INTERVAL_DAY;
                }
                case HOUR: {
                    return DataType.INTERVAL_HOUR;
                }
                case MINUTE: {
                    return DataType.INTERVAL_MINUTE;
                }
                case SECOND: {
                    return DataType.INTERVAL_SECOND;
                }
            }
            throw new ParsingException(source, "Cannot determine datatype for [{}]", new Object[]{leading});
        }
        if (leading == TimeUnit.YEAR && trailing == TimeUnit.MONTH) {
            return DataType.INTERVAL_YEAR_TO_MONTH;
        }
        if (leading == TimeUnit.DAY && trailing == TimeUnit.HOUR) {
            return DataType.INTERVAL_DAY_TO_HOUR;
        }
        if (leading == TimeUnit.DAY && trailing == TimeUnit.MINUTE) {
            return DataType.INTERVAL_DAY_TO_MINUTE;
        }
        if (leading == TimeUnit.DAY && trailing == TimeUnit.SECOND) {
            return DataType.INTERVAL_DAY_TO_SECOND;
        }
        if (leading == TimeUnit.HOUR && trailing == TimeUnit.MINUTE) {
            return DataType.INTERVAL_HOUR_TO_MINUTE;
        }
        if (leading == TimeUnit.HOUR && trailing == TimeUnit.SECOND) {
            return DataType.INTERVAL_HOUR_TO_SECOND;
        }
        if (leading == TimeUnit.MINUTE && trailing == TimeUnit.SECOND) {
            return DataType.INTERVAL_MINUTE_TO_SECOND;
        }
        throw new ParsingException(source, "Cannot determine datatype for combination [{}] [{}]", new Object[]{leading, trailing});
    }

    public static TemporalAmount negate(TemporalAmount interval) {
        return interval instanceof Period ? ((Period)interval).negated() : ((Duration)interval).negated();
    }

    public static TemporalAmount parseInterval(Location source, String value, DataType intervalType) {
        return PARSERS.get((Object)intervalType).parse(source, value);
    }

    public static Collection<? extends NamedWriteableRegistry.Entry> getNamedWriteables() {
        ArrayList<NamedWriteableRegistry.Entry> entries = new ArrayList<NamedWriteableRegistry.Entry>();
        entries.add(new NamedWriteableRegistry.Entry(IntervalDayTime.class, "idt", IntervalDayTime::new));
        entries.add(new NamedWriteableRegistry.Entry(IntervalYearMonth.class, "iym", IntervalYearMonth::new));
        return entries;
    }

    static {
        int MAX_MONTH = 11;
        int MAX_HOUR = 23;
        int MAX_MINUTE = 59;
        int MAX_SECOND = 59;
        int MAX_MILLI = 999999999;
        char DOT = '.';
        char SPACE = ' ';
        char MINUS = '-';
        char COLON = ':';
        PARSERS.put(DataType.INTERVAL_YEAR, new ParserBuilder(DataType.INTERVAL_YEAR).unit(TimeUnit.YEAR).build());
        PARSERS.put(DataType.INTERVAL_MONTH, new ParserBuilder(DataType.INTERVAL_MONTH).unit(TimeUnit.MONTH).build());
        PARSERS.put(DataType.INTERVAL_DAY, new ParserBuilder(DataType.INTERVAL_DAY).unit(TimeUnit.DAY).build());
        PARSERS.put(DataType.INTERVAL_HOUR, new ParserBuilder(DataType.INTERVAL_HOUR).unit(TimeUnit.HOUR).build());
        PARSERS.put(DataType.INTERVAL_MINUTE, new ParserBuilder(DataType.INTERVAL_MINUTE).unit(TimeUnit.MINUTE).build());
        PARSERS.put(DataType.INTERVAL_SECOND, new ParserBuilder(DataType.INTERVAL_SECOND).unit(TimeUnit.SECOND).optional().separator(DOT).unit(TimeUnit.MILLISECOND, MAX_MILLI).build());
        PARSERS.put(DataType.INTERVAL_YEAR_TO_MONTH, new ParserBuilder(DataType.INTERVAL_YEAR_TO_MONTH).unit(TimeUnit.YEAR).separator(MINUS).unit(TimeUnit.MONTH, MAX_MONTH).build());
        PARSERS.put(DataType.INTERVAL_DAY_TO_HOUR, new ParserBuilder(DataType.INTERVAL_DAY_TO_HOUR).unit(TimeUnit.DAY).separator(SPACE).unit(TimeUnit.HOUR, MAX_HOUR).build());
        PARSERS.put(DataType.INTERVAL_DAY_TO_MINUTE, new ParserBuilder(DataType.INTERVAL_DAY_TO_MINUTE).unit(TimeUnit.DAY).separator(SPACE).unit(TimeUnit.HOUR, MAX_HOUR).separator(COLON).unit(TimeUnit.MINUTE, MAX_MINUTE).build());
        PARSERS.put(DataType.INTERVAL_DAY_TO_SECOND, new ParserBuilder(DataType.INTERVAL_DAY_TO_SECOND).unit(TimeUnit.DAY).separator(SPACE).unit(TimeUnit.HOUR, MAX_HOUR).separator(COLON).unit(TimeUnit.MINUTE, MAX_MINUTE).separator(COLON).unit(TimeUnit.SECOND, MAX_SECOND).optional().separator(DOT).unit(TimeUnit.MILLISECOND, MAX_MILLI).build());
        PARSERS.put(DataType.INTERVAL_HOUR_TO_MINUTE, new ParserBuilder(DataType.INTERVAL_HOUR_TO_MINUTE).unit(TimeUnit.HOUR).separator(COLON).unit(TimeUnit.MINUTE, MAX_MINUTE).build());
        PARSERS.put(DataType.INTERVAL_HOUR_TO_SECOND, new ParserBuilder(DataType.INTERVAL_HOUR_TO_SECOND).unit(TimeUnit.HOUR).separator(COLON).unit(TimeUnit.MINUTE, MAX_MINUTE).separator(COLON).unit(TimeUnit.SECOND, MAX_SECOND).optional().separator(DOT).unit(TimeUnit.MILLISECOND, MAX_MILLI).build());
        PARSERS.put(DataType.INTERVAL_MINUTE_TO_SECOND, new ParserBuilder(DataType.INTERVAL_MINUTE_TO_SECOND).unit(TimeUnit.MINUTE).separator(COLON).unit(TimeUnit.SECOND, MAX_SECOND).optional().separator(DOT).unit(TimeUnit.MILLISECOND, MAX_MILLI).build());
    }

    private static class Parser {
        private static final char PLUS = '+';
        private static final char MINUS = '-';
        private final List<TimeUnit> units;
        private final List<Token> tokens;
        private final String name;

        Parser(List<TimeUnit> units, List<Token> tokens, String name) {
            this.units = units;
            this.tokens = tokens;
            this.name = name;
        }

        TemporalAmount parse(Location source, String string) {
            int unitIndex = 0;
            int startToken = 0;
            int endToken = 0;
            long[] values = new long[this.units.size()];
            boolean negate = false;
            char maybeSign = string.charAt(0);
            if ('+' == maybeSign) {
                startToken = 1;
            } else if ('-' == maybeSign) {
                startToken = 1;
                negate = true;
            }
            for (Token token : this.tokens) {
                if (startToken >= string.length()) {
                    if (token.optional) break;
                    throw new ParsingException(source, this.invalidIntervalMessage(string) + ": incorrect format, expecting {}", Strings.collectionToDelimitedString(this.tokens, (String)""));
                }
                if (token.ch != '\u0000') {
                    char found = string.charAt(startToken);
                    if (found != token.ch) {
                        throw new ParsingException(source, this.invalidIntervalMessage(string) + ": expected [{}] (at [{}]) but found [{}]", Character.valueOf(token.ch), startToken, Character.valueOf(found));
                    }
                    ++startToken;
                    continue;
                }
                for (endToken = startToken; endToken < string.length() && Character.isDigit(string.charAt(endToken)); ++endToken) {
                }
                if (endToken == startToken) {
                    throw new ParsingException(source, this.invalidIntervalMessage(string) + ": expected digit (at [{}]) but found [{}]", endToken, Character.valueOf(string.charAt(endToken)));
                }
                String number = string.substring(startToken, endToken);
                try {
                    long v = StringUtils.parseLong(number);
                    if (token.maxValue > 0 && v > (long)token.maxValue) {
                        throw new ParsingException(source, this.invalidIntervalMessage(string) + ": [{}] unit has illegal value [{}], expected a positive number up to [{}]", this.units.get(unitIndex).name(), v, token.maxValue);
                    }
                    if (v < 0L) {
                        throw new ParsingException(source, this.invalidIntervalMessage(string) + ": negative value [{}] not allowed (negate the entire interval instead)", v);
                    }
                    values[unitIndex++] = v;
                }
                catch (SqlIllegalArgumentException siae) {
                    throw new ParsingException(source, this.invalidIntervalMessage(string), siae.getMessage());
                }
                startToken = endToken;
            }
            if (endToken <= string.length() - 1) {
                throw new ParsingException(source, this.invalidIntervalMessage(string) + ": unexpected trailing characters found [{}]", string.substring(endToken));
            }
            TemporalAmount interval = this.units.get(0) == TimeUnit.YEAR || this.units.get(0) == TimeUnit.MONTH ? Period.ZERO : Duration.ZERO;
            for (int i = 0; i < values.length; ++i) {
                TemporalAmount ta = Intervals.of(source, values[i], this.units.get(i));
                interval = ta instanceof Period ? ((Period)ta).plus(interval) : ((Duration)ta).plus((Duration)interval);
            }
            if (negate) {
                interval = Intervals.negate(interval);
            }
            return interval;
        }

        private String invalidIntervalMessage(String interval) {
            return "Invalid [" + this.name + "] value [" + interval + "]";
        }

        public String toString() {
            return this.name;
        }
    }

    private static class Token {
        private final char ch;
        private final int maxValue;
        private final boolean optional;

        Token(char ch, int maxValue, boolean optional) {
            this.ch = ch;
            this.maxValue = maxValue;
            this.optional = optional;
        }

        public String toString() {
            return this.ch > '\u0000' ? String.valueOf(this.ch) : "[numeric]";
        }
    }

    private static class ParserBuilder {
        private final List<TimeUnit> units = new ArrayList<TimeUnit>(10);
        private final List<Token> tokens = new ArrayList<Token>(6);
        private final String name;
        private boolean optional = false;

        ParserBuilder(DataType dataType) {
            this.name = dataType.name().replace('_', ' ');
        }

        ParserBuilder unit(TimeUnit unit) {
            this.unit(unit, 0);
            return this;
        }

        ParserBuilder unit(TimeUnit unit, int maxValue) {
            this.units.add(unit);
            this.tokens.add(new Token('\u0000', maxValue, this.optional));
            return this;
        }

        ParserBuilder separator(char ch) {
            this.tokens.add(new Token(ch, 0, this.optional));
            return this;
        }

        ParserBuilder optional() {
            this.optional = true;
            return this;
        }

        Parser build() {
            return new Parser(this.units, this.tokens, this.name);
        }
    }

    public static enum TimeUnit {
        YEAR,
        MONTH,
        DAY,
        HOUR,
        MINUTE,
        SECOND,
        MILLISECOND;

    }
}

