/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.admin.cluster.bootstrap.BootstrapClusterRequest;
import org.elasticsearch.action.admin.cluster.bootstrap.BootstrapClusterResponse;
import org.elasticsearch.action.admin.cluster.bootstrap.BootstrapConfiguration;
import org.elasticsearch.action.admin.cluster.bootstrap.GetDiscoveredNodesRequest;
import org.elasticsearch.action.admin.cluster.bootstrap.GetDiscoveredNodesResponse;
import org.elasticsearch.cluster.coordination.ClusterAlreadyBootstrappedException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.discovery.zen.SettingsBasedHostsProvider;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class ClusterBootstrapService {
    private static final Logger logger = LogManager.getLogger(ClusterBootstrapService.class);
    public static final Setting<Integer> INITIAL_MASTER_NODE_COUNT_SETTING = Setting.intSetting("cluster.unsafe_initial_master_node_count", 0, 0, Setting.Property.NodeScope);
    public static final Setting<List<String>> INITIAL_MASTER_NODES_SETTING = Setting.listSetting("cluster.initial_master_nodes", Collections.emptyList(), Function.identity(), Setting.Property.NodeScope);
    public static final Setting<TimeValue> UNCONFIGURED_BOOTSTRAP_TIMEOUT_SETTING = Setting.timeSetting("discovery.unconfigured_bootstrap_timeout", TimeValue.timeValueSeconds(3L), TimeValue.timeValueMillis(1L), Setting.Property.NodeScope);
    private final int initialMasterNodeCount;
    private final List<String> initialMasterNodes;
    @Nullable
    private final TimeValue unconfiguredBootstrapTimeout;
    private final TransportService transportService;
    private volatile boolean running;

    public ClusterBootstrapService(Settings settings, TransportService transportService) {
        this.initialMasterNodeCount = INITIAL_MASTER_NODE_COUNT_SETTING.get(settings);
        this.initialMasterNodes = INITIAL_MASTER_NODES_SETTING.get(settings);
        this.unconfiguredBootstrapTimeout = ClusterBootstrapService.discoveryIsConfigured(settings) ? null : UNCONFIGURED_BOOTSTRAP_TIMEOUT_SETTING.get(settings);
        this.transportService = transportService;
    }

    public static boolean discoveryIsConfigured(Settings settings) {
        return Stream.of(DiscoveryModule.DISCOVERY_HOSTS_PROVIDER_SETTING, SettingsBasedHostsProvider.DISCOVERY_ZEN_PING_UNICAST_HOSTS_SETTING, INITIAL_MASTER_NODE_COUNT_SETTING, INITIAL_MASTER_NODES_SETTING).anyMatch(s -> s.exists(settings));
    }

    public void start() {
        assert (!this.running);
        this.running = true;
        if (!this.transportService.getLocalNode().isMasterNode()) {
            return;
        }
        if (this.unconfiguredBootstrapTimeout != null) {
            logger.info("no discovery configuration found, will perform best-effort cluster bootstrapping after [{}] unless existing master is discovered", (Object)this.unconfiguredBootstrapTimeout);
            final ThreadContext threadContext = this.transportService.getThreadPool().getThreadContext();
            try (ThreadContext.StoredContext ignore = threadContext.stashContext();){
                threadContext.markAsSystemContext();
                this.transportService.getThreadPool().scheduleUnlessShuttingDown(this.unconfiguredBootstrapTimeout, "same", new Runnable(){

                    @Override
                    public void run() {
                        threadContext.markAsSystemContext();
                        GetDiscoveredNodesRequest request = new GetDiscoveredNodesRequest();
                        logger.trace("sending {}", (Object)request);
                        ClusterBootstrapService.this.transportService.sendRequest(ClusterBootstrapService.this.transportService.getLocalNode(), "cluster:admin/bootstrap/discover_nodes", request, new TransportResponseHandler<GetDiscoveredNodesResponse>(){

                            @Override
                            public void handleResponse(GetDiscoveredNodesResponse response) {
                                logger.debug("discovered {}, starting to bootstrap", (Object)response.getNodes());
                                ClusterBootstrapService.this.awaitBootstrap(response.getBootstrapConfiguration());
                            }

                            @Override
                            public void handleException(TransportException exp) {
                                Throwable rootCause = exp.getRootCause();
                                if (rootCause instanceof ClusterAlreadyBootstrappedException) {
                                    logger.debug(rootCause.getMessage(), rootCause);
                                } else {
                                    logger.warn("discovery attempt failed", (Throwable)exp);
                                }
                            }

                            @Override
                            public String executor() {
                                return "same";
                            }

                            @Override
                            public GetDiscoveredNodesResponse read(StreamInput in) throws IOException {
                                return new GetDiscoveredNodesResponse(in);
                            }
                        });
                    }

                    public String toString() {
                        return "unconfigured-discovery delayed bootstrap";
                    }
                });
            }
        }
        if (this.initialMasterNodeCount > 0 || !this.initialMasterNodes.isEmpty()) {
            logger.debug("unsafely waiting for discovery of [{}] master-eligible nodes", (Object)this.initialMasterNodeCount);
            ThreadContext threadContext = this.transportService.getThreadPool().getThreadContext();
            try (ThreadContext.StoredContext ignore = threadContext.stashContext();){
                threadContext.markAsSystemContext();
                GetDiscoveredNodesRequest request = new GetDiscoveredNodesRequest();
                if (this.initialMasterNodeCount > 0) {
                    request.setWaitForNodes(this.initialMasterNodeCount);
                }
                request.setRequiredNodes(this.initialMasterNodes);
                request.setTimeout(null);
                logger.trace("sending {}", (Object)request);
                this.transportService.sendRequest(this.transportService.getLocalNode(), "cluster:admin/bootstrap/discover_nodes", request, new TransportResponseHandler<GetDiscoveredNodesResponse>(){

                    @Override
                    public void handleResponse(GetDiscoveredNodesResponse response) {
                        assert (response.getNodes().size() >= ClusterBootstrapService.this.initialMasterNodeCount);
                        assert (response.getNodes().stream().allMatch(DiscoveryNode::isMasterNode));
                        logger.debug("discovered {}, starting to bootstrap", (Object)response.getNodes());
                        ClusterBootstrapService.this.awaitBootstrap(response.getBootstrapConfiguration());
                    }

                    @Override
                    public void handleException(TransportException exp) {
                        logger.warn("discovery attempt failed", (Throwable)exp);
                    }

                    @Override
                    public String executor() {
                        return "same";
                    }

                    @Override
                    public GetDiscoveredNodesResponse read(StreamInput in) throws IOException {
                        return new GetDiscoveredNodesResponse(in);
                    }
                });
            }
        }
    }

    public void stop() {
        this.running = false;
    }

    private void awaitBootstrap(final BootstrapConfiguration bootstrapConfiguration) {
        if (!this.running) {
            logger.debug("awaitBootstrap: not running");
            return;
        }
        BootstrapClusterRequest request = new BootstrapClusterRequest(bootstrapConfiguration);
        logger.trace("sending {}", (Object)request);
        this.transportService.sendRequest(this.transportService.getLocalNode(), "cluster:admin/bootstrap/set_voting_config", request, new TransportResponseHandler<BootstrapClusterResponse>(){

            @Override
            public void handleResponse(BootstrapClusterResponse response) {
                logger.debug("automatic cluster bootstrapping successful: received {}", (Object)response);
            }

            @Override
            public void handleException(TransportException exp) {
                logger.warn(new ParameterizedMessage("automatic cluster bootstrapping failed, retrying [{}]", (Object)bootstrapConfiguration.getNodeDescriptions()), (Throwable)exp);
                ClusterBootstrapService.this.transportService.getThreadPool().scheduleUnlessShuttingDown(TimeValue.timeValueSeconds(10L), "same", new Runnable(){

                    @Override
                    public void run() {
                        ClusterBootstrapService.this.transportService.getThreadPool().getThreadContext().markAsSystemContext();
                        ClusterBootstrapService.this.awaitBootstrap(bootstrapConfiguration);
                    }

                    public String toString() {
                        return "retry bootstrapping with " + bootstrapConfiguration.getNodeDescriptions();
                    }
                });
            }

            @Override
            public String executor() {
                return "same";
            }

            @Override
            public BootstrapClusterResponse read(StreamInput in) throws IOException {
                return new BootstrapClusterResponse(in);
            }
        });
    }
}

