/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobStore;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.SocketAccess;

class S3BlobContainer
extends AbstractBlobContainer {
    private static final int MAX_BULK_DELETES = 1000;
    private final S3BlobStore blobStore;
    private final String keyPath;

    S3BlobContainer(BlobPath path, S3BlobStore blobStore) {
        super(path);
        this.blobStore = blobStore;
        this.keyPath = path.buildAsString();
    }

    public boolean blobExists(String blobName) {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            boolean bl = SocketAccess.doPrivileged(() -> clientReference.client().doesObjectExist(this.blobStore.bucket(), this.buildKey(blobName)));
            if (clientReference != null) {
                clientReference.close();
            }
            return bl;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new BlobStoreException("Failed to check if blob [" + blobName + "] exists", (Throwable)e);
            }
        }
    }

    public InputStream readBlob(String blobName) throws IOException {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            S3Object s3Object = SocketAccess.doPrivileged(() -> clientReference.client().getObject(this.blobStore.bucket(), this.buildKey(blobName)));
            S3ObjectInputStream s3ObjectInputStream = s3Object.getObjectContent();
            if (clientReference != null) {
                clientReference.close();
            }
            return s3ObjectInputStream;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (AmazonClientException e) {
                if (e instanceof AmazonS3Exception && 404 == ((AmazonS3Exception)e).getStatusCode()) {
                    throw new NoSuchFileException("Blob object [" + blobName + "] not found: " + e.getMessage());
                }
                throw e;
            }
        }
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        SocketAccess.doPrivilegedIOException(() -> {
            if (blobSize <= this.blobStore.bufferSizeInBytes()) {
                this.executeSingleUpload(this.blobStore, this.buildKey(blobName), inputStream, blobSize);
            } else {
                this.executeMultipartUpload(this.blobStore, this.buildKey(blobName), inputStream, blobSize);
            }
            return null;
        });
    }

    public void writeBlobAtomic(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        this.writeBlob(blobName, inputStream, blobSize, failIfAlreadyExists);
    }

    public void deleteBlob(String blobName) throws IOException {
        this.deleteBlobIgnoringIfNotExists(blobName);
    }

    public void deleteBlobsIgnoringIfNotExists(List<String> blobNames) throws IOException {
        if (blobNames.isEmpty()) {
            return;
        }
        Set outstanding = blobNames.stream().map(this::buildKey).collect(Collectors.toSet());
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            ArrayList<DeleteObjectsRequest> deleteRequests = new ArrayList<DeleteObjectsRequest>();
            ArrayList<String> partition = new ArrayList<String>();
            for (String key : outstanding) {
                partition.add(key);
                if (partition.size() != 1000) continue;
                deleteRequests.add(S3BlobContainer.bulkDelete(this.blobStore.bucket(), partition));
                partition.clear();
            }
            if (!partition.isEmpty()) {
                deleteRequests.add(S3BlobContainer.bulkDelete(this.blobStore.bucket(), partition));
            }
            SocketAccess.doPrivilegedVoid(() -> {
                AmazonClientException aex = null;
                for (DeleteObjectsRequest deleteRequest : deleteRequests) {
                    List keysInRequest = deleteRequest.getKeys().stream().map(DeleteObjectsRequest.KeyVersion::getKey).collect(Collectors.toList());
                    try {
                        clientReference.client().deleteObjects(deleteRequest);
                        outstanding.removeAll(keysInRequest);
                    }
                    catch (MultiObjectDeleteException e) {
                        outstanding.removeAll(keysInRequest);
                        outstanding.addAll(e.getErrors().stream().map(MultiObjectDeleteException.DeleteError::getKey).collect(Collectors.toSet()));
                        aex = (AmazonClientException)ExceptionsHelper.useOrSuppress((Throwable)aex, (Throwable)e);
                    }
                    catch (AmazonClientException e) {
                        aex = (AmazonClientException)ExceptionsHelper.useOrSuppress(aex, (Throwable)e);
                    }
                }
                if (aex != null) {
                    throw aex;
                }
            });
        }
        catch (Exception e) {
            throw new IOException("Failed to delete blobs [" + outstanding + "]", e);
        }
        assert (outstanding.isEmpty());
    }

    private static DeleteObjectsRequest bulkDelete(String bucket, List<String> blobs) {
        return new DeleteObjectsRequest(bucket).withKeys(blobs.toArray(Strings.EMPTY_ARRAY)).withQuiet(true);
    }

    public void deleteBlobIgnoringIfNotExists(String blobName) throws IOException {
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().deleteObject(this.blobStore.bucket(), this.buildKey(blobName)));
        }
        catch (AmazonClientException e) {
            throw new IOException("Exception when deleting blob [" + blobName + "]", e);
        }
    }

    public Map<String, BlobMetaData> listBlobsByPrefix(@Nullable String blobNamePrefix) throws IOException {
        MapBuilder blobsBuilder = MapBuilder.newMapBuilder();
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            ObjectListing prevListing = null;
            while (true) {
                ObjectListing list;
                if (prevListing != null) {
                    ObjectListing finalPrevListing = prevListing;
                    list = SocketAccess.doPrivileged(() -> clientReference.client().listNextBatchOfObjects(finalPrevListing));
                } else {
                    ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
                    listObjectsRequest.setBucketName(this.blobStore.bucket());
                    listObjectsRequest.setDelimiter("/");
                    if (blobNamePrefix != null) {
                        listObjectsRequest.setPrefix(this.buildKey(blobNamePrefix));
                    } else {
                        listObjectsRequest.setPrefix(this.keyPath);
                    }
                    list = SocketAccess.doPrivileged(() -> clientReference.client().listObjects(listObjectsRequest));
                }
                for (S3ObjectSummary summary : list.getObjectSummaries()) {
                    String name = summary.getKey().substring(this.keyPath.length());
                    blobsBuilder.put((Object)name, (Object)new PlainBlobMetaData(name, summary.getSize()));
                }
                if (!list.isTruncated()) break;
                prevListing = list;
            }
            Map map = blobsBuilder.immutableMap();
            if (clientReference != null) {
                clientReference.close();
            }
            return map;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (AmazonClientException e) {
                throw new IOException("Exception when listing blobs by prefix [" + blobNamePrefix + "]", e);
            }
        }
    }

    public Map<String, BlobMetaData> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    public Map<String, BlobContainer> children() throws IOException {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            ObjectListing prevListing = null;
            HashMap<String, BlobContainer> entries = new HashMap<String, BlobContainer>();
            while (true) {
                ObjectListing list;
                if (prevListing != null) {
                    ObjectListing finalPrevListing = prevListing;
                    list = SocketAccess.doPrivileged(() -> clientReference.client().listNextBatchOfObjects(finalPrevListing));
                } else {
                    ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
                    listObjectsRequest.setBucketName(this.blobStore.bucket());
                    listObjectsRequest.setPrefix(this.keyPath);
                    listObjectsRequest.setDelimiter("/");
                    list = SocketAccess.doPrivileged(() -> clientReference.client().listObjects(listObjectsRequest));
                }
                for (String summary : list.getCommonPrefixes()) {
                    String name = summary.substring(this.keyPath.length());
                    if (name.isEmpty()) continue;
                    String last = name.substring(0, name.length() - 1);
                    BlobPath path = this.path().add(last);
                    entries.put(last, this.blobStore.blobContainer(path));
                }
                assert (list.getObjectSummaries().stream().noneMatch(s -> {
                    for (String commonPrefix : list.getCommonPrefixes()) {
                        if (!s.getKey().substring(this.keyPath.length()).startsWith(commonPrefix)) continue;
                        return true;
                    }
                    return false;
                })) : "Response contained children for listed common prefixes.";
                if (!list.isTruncated()) break;
                prevListing = list;
            }
            Map<String, BlobContainer> map = Collections.unmodifiableMap(entries);
            if (clientReference != null) {
                clientReference.close();
            }
            return map;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (AmazonClientException e) {
                throw new IOException("Exception when listing children of [" + this.path().buildAsString() + ']', e);
            }
        }
    }

    private String buildKey(String blobName) {
        return this.keyPath + blobName;
    }

    void executeSingleUpload(S3BlobStore blobStore, String blobName, InputStream input, long blobSize) throws IOException {
        if (blobSize > S3Repository.MAX_FILE_SIZE.getBytes()) {
            throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than " + S3Repository.MAX_FILE_SIZE);
        }
        if (blobSize > blobStore.bufferSizeInBytes()) {
            throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than buffer size");
        }
        ObjectMetadata md = new ObjectMetadata();
        md.setContentLength(blobSize);
        if (blobStore.serverSideEncryption()) {
            md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
        }
        PutObjectRequest putRequest = new PutObjectRequest(blobStore.bucket(), blobName, input, md);
        putRequest.setStorageClass(blobStore.getStorageClass());
        putRequest.setCannedAcl(blobStore.getCannedACL());
        try (AmazonS3Reference clientReference = blobStore.clientReference();){
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().putObject(putRequest));
        }
        catch (AmazonClientException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using a single upload", e);
        }
    }

    void executeMultipartUpload(S3BlobStore blobStore, String blobName, InputStream input, long blobSize) throws IOException {
        if (blobSize > S3Repository.MAX_FILE_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be larger than " + S3Repository.MAX_FILE_SIZE_USING_MULTIPART);
        }
        if (blobSize < S3Repository.MIN_PART_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be smaller than " + S3Repository.MIN_PART_SIZE_USING_MULTIPART);
        }
        long partSize = blobStore.bufferSizeInBytes();
        Tuple<Long, Long> multiparts = S3BlobContainer.numberOfMultiparts(blobSize, partSize);
        if ((Long)multiparts.v1() > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Too many multipart upload requests, maybe try a larger buffer size?");
        }
        int nbParts = ((Long)multiparts.v1()).intValue();
        long lastPartSize = (Long)multiparts.v2();
        assert (blobSize == (long)(nbParts - 1) * partSize + lastPartSize) : "blobSize does not match multipart sizes";
        SetOnce uploadId = new SetOnce();
        String bucketName = blobStore.bucket();
        boolean success = false;
        InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(bucketName, blobName);
        initRequest.setStorageClass(blobStore.getStorageClass());
        initRequest.setCannedACL(blobStore.getCannedACL());
        if (blobStore.serverSideEncryption()) {
            ObjectMetadata md = new ObjectMetadata();
            md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
            initRequest.setObjectMetadata(md);
        }
        try (AmazonS3Reference clientReference = blobStore.clientReference();){
            uploadId.set((Object)SocketAccess.doPrivileged(() -> clientReference.client().initiateMultipartUpload(initRequest).getUploadId()));
            if (Strings.isEmpty((CharSequence)((CharSequence)uploadId.get()))) {
                throw new IOException("Failed to initialize multipart upload " + blobName);
            }
            ArrayList<PartETag> parts = new ArrayList<PartETag>();
            long bytesCount = 0L;
            for (int i = 1; i <= nbParts; ++i) {
                UploadPartRequest uploadRequest = new UploadPartRequest();
                uploadRequest.setBucketName(bucketName);
                uploadRequest.setKey(blobName);
                uploadRequest.setUploadId((String)uploadId.get());
                uploadRequest.setPartNumber(i);
                uploadRequest.setInputStream(input);
                if (i < nbParts) {
                    uploadRequest.setPartSize(partSize);
                    uploadRequest.setLastPart(false);
                } else {
                    uploadRequest.setPartSize(lastPartSize);
                    uploadRequest.setLastPart(true);
                }
                bytesCount += uploadRequest.getPartSize();
                UploadPartResult uploadResponse = SocketAccess.doPrivileged(() -> clientReference.client().uploadPart(uploadRequest));
                parts.add(uploadResponse.getPartETag());
            }
            if (bytesCount != blobSize) {
                throw new IOException("Failed to execute multipart upload for [" + blobName + "], expected " + blobSize + "bytes sent but got " + bytesCount);
            }
            CompleteMultipartUploadRequest complRequest = new CompleteMultipartUploadRequest(bucketName, blobName, (String)uploadId.get(), parts);
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().completeMultipartUpload(complRequest));
            success = true;
        }
        catch (AmazonClientException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using multipart upload", e);
        }
        finally {
            if (!success && Strings.hasLength((String)((String)uploadId.get()))) {
                AbortMultipartUploadRequest abortRequest = new AbortMultipartUploadRequest(bucketName, blobName, (String)uploadId.get());
                try (AmazonS3Reference clientReference2 = blobStore.clientReference();){
                    SocketAccess.doPrivilegedVoid(() -> clientReference2.client().abortMultipartUpload(abortRequest));
                }
            }
        }
    }

    static Tuple<Long, Long> numberOfMultiparts(long totalSize, long partSize) {
        if (partSize <= 0L) {
            throw new IllegalArgumentException("Part size must be greater than zero");
        }
        if (totalSize == 0L || totalSize <= partSize) {
            return Tuple.tuple((Object)1L, (Object)totalSize);
        }
        long parts = totalSize / partSize;
        long remaining = totalSize % partSize;
        if (remaining == 0L) {
            return Tuple.tuple((Object)parts, (Object)partSize);
        }
        return Tuple.tuple((Object)(parts + 1L), (Object)remaining);
    }
}

