/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetadata;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.PlainBlobMetadata;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.repositories.blobstore.ChunkedBlobOutputStream;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobStore;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.S3RetryingInputStream;
import org.elasticsearch.repositories.s3.SocketAccess;

class S3BlobContainer
extends AbstractBlobContainer {
    private static final Logger logger = LogManager.getLogger(S3BlobContainer.class);
    private static final int MAX_BULK_DELETES = 1000;
    private final S3BlobStore blobStore;
    private final String keyPath;

    S3BlobContainer(BlobPath path, S3BlobStore blobStore) {
        super(path);
        this.blobStore = blobStore;
        this.keyPath = path.buildAsString();
    }

    public boolean blobExists(String blobName) {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            boolean bl = SocketAccess.doPrivileged(() -> clientReference.client().doesObjectExist(this.blobStore.bucket(), this.buildKey(blobName)));
            if (clientReference != null) {
                clientReference.close();
            }
            return bl;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new BlobStoreException("Failed to check if blob [" + blobName + "] exists", (Throwable)e);
            }
        }
    }

    public InputStream readBlob(String blobName) throws IOException {
        return new S3RetryingInputStream(this.blobStore, this.buildKey(blobName));
    }

    public InputStream readBlob(String blobName, long position, long length) throws IOException {
        if (position < 0L) {
            throw new IllegalArgumentException("position must be non-negative");
        }
        if (length < 0L) {
            throw new IllegalArgumentException("length must be non-negative");
        }
        if (length == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        return new S3RetryingInputStream(this.blobStore, this.buildKey(blobName), position, Math.addExact(position, length - 1L));
    }

    public long readBlobPreferredLength() {
        return new ByteSizeValue(32L, ByteSizeUnit.MB).getBytes();
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (inputStream.markSupported()) : "No mark support on inputStream breaks the S3 SDK's ability to retry requests";
        SocketAccess.doPrivilegedIOException(() -> {
            if (blobSize <= this.getLargeBlobThresholdInBytes()) {
                this.executeSingleUpload(this.blobStore, this.buildKey(blobName), inputStream, blobSize);
            } else {
                this.executeMultipartUpload(this.blobStore, this.buildKey(blobName), inputStream, blobSize);
            }
            return null;
        });
    }

    public void writeBlob(final String blobName, final boolean failIfAlreadyExists, boolean atomic, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
        final String absoluteBlobKey = this.buildKey(blobName);
        try (final AmazonS3Reference clientReference = this.blobStore.clientReference();
             ChunkedBlobOutputStream<PartETag> out = new ChunkedBlobOutputStream<PartETag>(this.blobStore.bigArrays(), this.blobStore.bufferSizeInBytes()){
            private final SetOnce<String> uploadId;
            {
                super(arg0, arg1);
                this.uploadId = new SetOnce();
            }

            protected void flushBuffer() throws IOException {
                this.flushBuffer(false);
            }

            private void flushBuffer(boolean lastPart) throws IOException {
                if (this.buffer.size() == 0) {
                    return;
                }
                if (this.flushedBytes == 0L) {
                    assert (!lastPart) : "use single part upload if there's only a single part";
                    this.uploadId.set((Object)SocketAccess.doPrivileged(() -> clientReference.client().initiateMultipartUpload(S3BlobContainer.this.initiateMultiPartUpload(absoluteBlobKey)).getUploadId()));
                    if (Strings.isEmpty((CharSequence)((CharSequence)this.uploadId.get()))) {
                        throw new IOException("Failed to initialize multipart upload " + absoluteBlobKey);
                    }
                }
                assert (!lastPart || this.successful) : "must only write last part if successful";
                UploadPartRequest uploadRequest = S3BlobContainer.this.createPartUploadRequest((InputStream)this.buffer.bytes().streamInput(), (String)this.uploadId.get(), this.parts.size() + 1, absoluteBlobKey, this.buffer.size(), lastPart);
                UploadPartResult uploadResponse = SocketAccess.doPrivileged(() -> clientReference.client().uploadPart(uploadRequest));
                this.finishPart(uploadResponse.getPartETag());
            }

            protected void onCompletion() throws IOException {
                if (this.flushedBytes == 0L) {
                    S3BlobContainer.this.writeBlob(blobName, this.buffer.bytes(), failIfAlreadyExists);
                } else {
                    this.flushBuffer(true);
                    CompleteMultipartUploadRequest complRequest = new CompleteMultipartUploadRequest(S3BlobContainer.this.blobStore.bucket(), absoluteBlobKey, (String)this.uploadId.get(), this.parts);
                    complRequest.setRequestMetricCollector(((S3BlobContainer)S3BlobContainer.this).blobStore.multiPartUploadMetricCollector);
                    SocketAccess.doPrivilegedVoid(() -> clientReference.client().completeMultipartUpload(complRequest));
                }
            }

            protected void onFailure() {
                if (Strings.hasText((String)((String)this.uploadId.get()))) {
                    S3BlobContainer.this.abortMultiPartUpload((String)this.uploadId.get(), absoluteBlobKey);
                }
            }
        };){
            writer.accept((Object)out);
            out.markSuccess();
        }
    }

    private UploadPartRequest createPartUploadRequest(InputStream stream, String uploadId, int number, String blobName, long size, boolean lastPart) {
        UploadPartRequest uploadRequest = new UploadPartRequest();
        uploadRequest.setBucketName(this.blobStore.bucket());
        uploadRequest.setKey(blobName);
        uploadRequest.setUploadId(uploadId);
        uploadRequest.setPartNumber(number);
        uploadRequest.setInputStream(stream);
        uploadRequest.setRequestMetricCollector(this.blobStore.multiPartUploadMetricCollector);
        uploadRequest.setPartSize(size);
        uploadRequest.setLastPart(lastPart);
        return uploadRequest;
    }

    private void abortMultiPartUpload(String uploadId, String blobName) {
        AbortMultipartUploadRequest abortRequest = new AbortMultipartUploadRequest(this.blobStore.bucket(), blobName, uploadId);
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().abortMultipartUpload(abortRequest));
        }
    }

    private InitiateMultipartUploadRequest initiateMultiPartUpload(String blobName) {
        InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(this.blobStore.bucket(), blobName);
        initRequest.setStorageClass(this.blobStore.getStorageClass());
        initRequest.setCannedACL(this.blobStore.getCannedACL());
        initRequest.setRequestMetricCollector(this.blobStore.multiPartUploadMetricCollector);
        if (this.blobStore.serverSideEncryption()) {
            ObjectMetadata md = new ObjectMetadata();
            md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
            initRequest.setObjectMetadata(md);
        }
        return initRequest;
    }

    long getLargeBlobThresholdInBytes() {
        return this.blobStore.bufferSizeInBytes();
    }

    public void writeBlobAtomic(String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
        this.writeBlob(blobName, bytes, failIfAlreadyExists);
    }

    public DeleteResult delete() throws IOException {
        final AtomicLong deletedBlobs = new AtomicLong();
        final AtomicLong deletedBytes = new AtomicLong();
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            Iterator<String> blobNameIterator;
            ObjectListing prevListing = null;
            while (true) {
                ObjectListing list;
                if (prevListing != null) {
                    ObjectListing finalPrevListing = prevListing;
                    list = SocketAccess.doPrivileged(() -> clientReference.client().listNextBatchOfObjects(finalPrevListing));
                } else {
                    ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
                    listObjectsRequest.setBucketName(this.blobStore.bucket());
                    listObjectsRequest.setPrefix(this.keyPath);
                    listObjectsRequest.setRequestMetricCollector(this.blobStore.listMetricCollector);
                    list = SocketAccess.doPrivileged(() -> clientReference.client().listObjects(listObjectsRequest));
                }
                final Iterator objectSummaryIterator = list.getObjectSummaries().iterator();
                blobNameIterator = new Iterator<String>(){

                    @Override
                    public boolean hasNext() {
                        return objectSummaryIterator.hasNext();
                    }

                    @Override
                    public String next() {
                        S3ObjectSummary summary = (S3ObjectSummary)objectSummaryIterator.next();
                        deletedBlobs.incrementAndGet();
                        deletedBytes.addAndGet(summary.getSize());
                        return summary.getKey();
                    }
                };
                if (!list.isTruncated()) break;
                this.doDeleteBlobs(blobNameIterator, false);
                prevListing = list;
            }
            this.doDeleteBlobs(Iterators.concat((Iterator[])new Iterator[]{blobNameIterator, Collections.singletonList(this.keyPath).iterator()}), false);
        }
        catch (AmazonClientException e) {
            throw new IOException("Exception when deleting blob container [" + this.keyPath + "]", e);
        }
        return new DeleteResult(deletedBlobs.get(), deletedBytes.get());
    }

    public void deleteBlobsIgnoringIfNotExists(Iterator<String> blobNames) throws IOException {
        this.doDeleteBlobs(blobNames, true);
    }

    private void doDeleteBlobs(final Iterator<String> blobNames, boolean relative) throws IOException {
        if (!blobNames.hasNext()) {
            return;
        }
        Iterator<String> outstanding = relative ? new Iterator<String>(){

            @Override
            public boolean hasNext() {
                return blobNames.hasNext();
            }

            @Override
            public String next() {
                return S3BlobContainer.this.buildKey((String)blobNames.next());
            }
        } : blobNames;
        ArrayList partition = new ArrayList();
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            AtomicReference aex = new AtomicReference();
            SocketAccess.doPrivilegedVoid(() -> {
                outstanding.forEachRemaining(key -> {
                    partition.add(key);
                    if (partition.size() == 1000) {
                        this.deletePartition(clientReference, partition, aex);
                        partition.clear();
                    }
                });
                if (!partition.isEmpty()) {
                    this.deletePartition(clientReference, partition, aex);
                }
            });
            if (aex.get() != null) {
                throw (Exception)aex.get();
            }
        }
        catch (Exception e) {
            throw new IOException("Failed to delete blobs " + partition.stream().limit(10L).collect(Collectors.toList()), e);
        }
    }

    private void deletePartition(AmazonS3Reference clientReference, List<String> partition, AtomicReference<Exception> aex) {
        try {
            clientReference.client().deleteObjects(S3BlobContainer.bulkDelete(this.blobStore.bucket(), partition));
        }
        catch (MultiObjectDeleteException e) {
            logger.warn(() -> new ParameterizedMessage("Failed to delete some blobs {}", e.getErrors().stream().map(err -> "[" + err.getKey() + "][" + err.getCode() + "][" + err.getMessage() + "]").collect(Collectors.toList())), (Throwable)e);
            aex.set((Exception)ExceptionsHelper.useOrSuppress((Throwable)aex.get(), (Throwable)e));
        }
        catch (AmazonClientException e) {
            aex.set((Exception)ExceptionsHelper.useOrSuppress((Throwable)aex.get(), (Throwable)e));
        }
    }

    private static DeleteObjectsRequest bulkDelete(String bucket, List<String> blobs) {
        return new DeleteObjectsRequest(bucket).withKeys(blobs.toArray(Strings.EMPTY_ARRAY)).withQuiet(true);
    }

    public Map<String, BlobMetadata> listBlobsByPrefix(@Nullable String blobNamePrefix) throws IOException {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            Map<String, BlobMetadata> map = S3BlobContainer.executeListing(clientReference, this.listObjectsRequest(blobNamePrefix == null ? this.keyPath : this.buildKey(blobNamePrefix))).stream().flatMap(listing -> listing.getObjectSummaries().stream()).map(summary -> new PlainBlobMetadata(summary.getKey().substring(this.keyPath.length()), summary.getSize())).collect(Collectors.toMap(PlainBlobMetadata::name, Function.identity()));
            if (clientReference != null) {
                clientReference.close();
            }
            return map;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (AmazonClientException e) {
                throw new IOException("Exception when listing blobs by prefix [" + blobNamePrefix + "]", e);
            }
        }
    }

    public Map<String, BlobMetadata> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    public Map<String, BlobContainer> children() throws IOException {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            Map<String, BlobContainer> map = S3BlobContainer.executeListing(clientReference, this.listObjectsRequest(this.keyPath)).stream().flatMap(listing -> {
                assert (listing.getObjectSummaries().stream().noneMatch(s -> {
                    for (String commonPrefix : listing.getCommonPrefixes()) {
                        if (!s.getKey().substring(this.keyPath.length()).startsWith(commonPrefix)) continue;
                        return true;
                    }
                    return false;
                })) : "Response contained children for listed common prefixes.";
                return listing.getCommonPrefixes().stream();
            }).map(prefix -> prefix.substring(this.keyPath.length())).filter(name -> !name.isEmpty()).map(name -> name.substring(0, name.length() - 1)).collect(Collectors.toMap(Function.identity(), name -> this.blobStore.blobContainer(this.path().add(name))));
            if (clientReference != null) {
                clientReference.close();
            }
            return map;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (AmazonClientException e) {
                throw new IOException("Exception when listing children of [" + this.path().buildAsString() + ']', e);
            }
        }
    }

    private static List<ObjectListing> executeListing(AmazonS3Reference clientReference, ListObjectsRequest listObjectsRequest) {
        ArrayList<ObjectListing> results = new ArrayList<ObjectListing>();
        ObjectListing prevListing = null;
        while (true) {
            ObjectListing list;
            if (prevListing != null) {
                ObjectListing finalPrevListing = prevListing;
                list = SocketAccess.doPrivileged(() -> clientReference.client().listNextBatchOfObjects(finalPrevListing));
            } else {
                list = SocketAccess.doPrivileged(() -> clientReference.client().listObjects(listObjectsRequest));
            }
            results.add(list);
            if (!list.isTruncated()) break;
            prevListing = list;
        }
        return results;
    }

    private ListObjectsRequest listObjectsRequest(String keyPath) {
        return (ListObjectsRequest)new ListObjectsRequest().withBucketName(this.blobStore.bucket()).withPrefix(keyPath).withDelimiter("/").withRequestMetricCollector(this.blobStore.listMetricCollector);
    }

    private String buildKey(String blobName) {
        return this.keyPath + blobName;
    }

    void executeSingleUpload(S3BlobStore blobStore, String blobName, InputStream input, long blobSize) throws IOException {
        if (blobSize > S3Repository.MAX_FILE_SIZE.getBytes()) {
            throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than " + S3Repository.MAX_FILE_SIZE);
        }
        if (blobSize > blobStore.bufferSizeInBytes()) {
            throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than buffer size");
        }
        ObjectMetadata md = new ObjectMetadata();
        md.setContentLength(blobSize);
        if (blobStore.serverSideEncryption()) {
            md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
        }
        PutObjectRequest putRequest = new PutObjectRequest(blobStore.bucket(), blobName, input, md);
        putRequest.setStorageClass(blobStore.getStorageClass());
        putRequest.setCannedAcl(blobStore.getCannedACL());
        putRequest.setRequestMetricCollector(blobStore.putMetricCollector);
        try (AmazonS3Reference clientReference = blobStore.clientReference();){
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().putObject(putRequest));
        }
        catch (AmazonClientException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using a single upload", e);
        }
    }

    void executeMultipartUpload(S3BlobStore blobStore, String blobName, InputStream input, long blobSize) throws IOException {
        this.ensureMultiPartUploadSize(blobSize);
        long partSize = blobStore.bufferSizeInBytes();
        Tuple<Long, Long> multiparts = S3BlobContainer.numberOfMultiparts(blobSize, partSize);
        if ((Long)multiparts.v1() > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Too many multipart upload requests, maybe try a larger buffer size?");
        }
        int nbParts = ((Long)multiparts.v1()).intValue();
        long lastPartSize = (Long)multiparts.v2();
        assert (blobSize == (long)(nbParts - 1) * partSize + lastPartSize) : "blobSize does not match multipart sizes";
        SetOnce uploadId = new SetOnce();
        String bucketName = blobStore.bucket();
        boolean success = false;
        try (AmazonS3Reference clientReference = blobStore.clientReference();){
            uploadId.set((Object)SocketAccess.doPrivileged(() -> clientReference.client().initiateMultipartUpload(this.initiateMultiPartUpload(blobName)).getUploadId()));
            if (Strings.isEmpty((CharSequence)((CharSequence)uploadId.get()))) {
                throw new IOException("Failed to initialize multipart upload " + blobName);
            }
            ArrayList<PartETag> parts = new ArrayList<PartETag>();
            long bytesCount = 0L;
            for (int i = 1; i <= nbParts; ++i) {
                boolean lastPart = i == nbParts;
                UploadPartRequest uploadRequest = this.createPartUploadRequest(input, (String)uploadId.get(), i, blobName, lastPart ? lastPartSize : partSize, lastPart);
                bytesCount += uploadRequest.getPartSize();
                UploadPartResult uploadResponse = SocketAccess.doPrivileged(() -> clientReference.client().uploadPart(uploadRequest));
                parts.add(uploadResponse.getPartETag());
            }
            if (bytesCount != blobSize) {
                throw new IOException("Failed to execute multipart upload for [" + blobName + "], expected " + blobSize + "bytes sent but got " + bytesCount);
            }
            CompleteMultipartUploadRequest complRequest = new CompleteMultipartUploadRequest(bucketName, blobName, (String)uploadId.get(), parts);
            complRequest.setRequestMetricCollector(blobStore.multiPartUploadMetricCollector);
            SocketAccess.doPrivilegedVoid(() -> clientReference.client().completeMultipartUpload(complRequest));
            success = true;
        }
        catch (AmazonClientException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using multipart upload", e);
        }
        finally {
            if (!success && Strings.hasLength((String)((String)uploadId.get()))) {
                this.abortMultiPartUpload((String)uploadId.get(), blobName);
            }
        }
    }

    void ensureMultiPartUploadSize(long blobSize) {
        if (blobSize > S3Repository.MAX_FILE_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be larger than " + S3Repository.MAX_FILE_SIZE_USING_MULTIPART);
        }
        if (blobSize < S3Repository.MIN_PART_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be smaller than " + S3Repository.MIN_PART_SIZE_USING_MULTIPART);
        }
    }

    static Tuple<Long, Long> numberOfMultiparts(long totalSize, long partSize) {
        if (partSize <= 0L) {
            throw new IllegalArgumentException("Part size must be greater than zero");
        }
        if (totalSize == 0L || totalSize <= partSize) {
            return Tuple.tuple((Object)1L, (Object)totalSize);
        }
        long parts = totalSize / partSize;
        long remaining = totalSize % partSize;
        if (remaining == 0L) {
            return Tuple.tuple((Object)parts, (Object)partSize);
        }
        return Tuple.tuple((Object)(parts + 1L), (Object)remaining);
    }
}

