/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.security.AccessController;
import java.security.Permission;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Map;
import org.elasticsearch.SpecialPermission;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.repositories.s3.DefaultS3OutputStream;
import org.elasticsearch.repositories.s3.S3BlobStore;

class S3BlobContainer
extends AbstractBlobContainer {
    protected final S3BlobStore blobStore;
    protected final String keyPath;

    S3BlobContainer(BlobPath path, S3BlobStore blobStore) {
        super(path);
        this.blobStore = blobStore;
        this.keyPath = path.buildAsString();
    }

    public boolean blobExists(String blobName) {
        try {
            return this.doPrivileged(() -> {
                try {
                    this.blobStore.client().getObjectMetadata(this.blobStore.bucket(), this.buildKey(blobName));
                    return true;
                }
                catch (AmazonS3Exception e) {
                    return false;
                }
            });
        }
        catch (AmazonS3Exception e) {
            return false;
        }
        catch (Exception e) {
            throw new BlobStoreException("failed to check if blob exists", (Throwable)e);
        }
    }

    public InputStream readBlob(String blobName) throws IOException {
        try {
            S3Object s3Object = this.blobStore.client().getObject(this.blobStore.bucket(), this.buildKey(blobName));
            return s3Object.getObjectContent();
        }
        catch (AmazonClientException e) {
            if (e instanceof AmazonS3Exception && 404 == ((AmazonS3Exception)((Object)e)).getStatusCode()) {
                throw new NoSuchFileException("Blob object [" + blobName + "] not found: " + e.getMessage());
            }
            throw e;
        }
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize) throws IOException {
        if (this.blobExists(blobName)) {
            throw new FileAlreadyExistsException("blob [" + blobName + "] already exists, cannot overwrite");
        }
        try (OutputStream stream = this.createOutput(blobName);){
            Streams.copy((InputStream)inputStream, (OutputStream)stream);
        }
    }

    public void deleteBlob(String blobName) throws IOException {
        if (!this.blobExists(blobName)) {
            throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
        }
        try {
            this.blobStore.client().deleteObject(this.blobStore.bucket(), this.buildKey(blobName));
        }
        catch (AmazonClientException e) {
            throw new IOException("Exception when deleting blob [" + blobName + "]", e);
        }
    }

    private OutputStream createOutput(String blobName) throws IOException {
        return new DefaultS3OutputStream(this.blobStore, this.blobStore.bucket(), this.buildKey(blobName), this.blobStore.bufferSizeInBytes(), this.blobStore.serverSideEncryption());
    }

    public Map<String, BlobMetaData> listBlobsByPrefix(@Nullable String blobNamePrefix) throws IOException {
        MapBuilder blobsBuilder = MapBuilder.newMapBuilder();
        ObjectListing prevListing = null;
        while (true) {
            ObjectListing list = prevListing != null ? this.blobStore.client().listNextBatchOfObjects(prevListing) : (blobNamePrefix != null ? this.blobStore.client().listObjects(this.blobStore.bucket(), this.buildKey(blobNamePrefix)) : this.blobStore.client().listObjects(this.blobStore.bucket(), this.keyPath));
            for (S3ObjectSummary summary : list.getObjectSummaries()) {
                String name = summary.getKey().substring(this.keyPath.length());
                blobsBuilder.put((Object)name, (Object)new PlainBlobMetaData(name, summary.getSize()));
            }
            if (!list.isTruncated()) break;
            prevListing = list;
        }
        return blobsBuilder.immutableMap();
    }

    public void move(String sourceBlobName, String targetBlobName) throws IOException {
        try {
            CopyObjectRequest request = new CopyObjectRequest(this.blobStore.bucket(), this.buildKey(sourceBlobName), this.blobStore.bucket(), this.buildKey(targetBlobName));
            if (this.blobStore.serverSideEncryption()) {
                ObjectMetadata objectMetadata = new ObjectMetadata();
                objectMetadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
                request.setNewObjectMetadata(objectMetadata);
            }
            this.blobStore.client().copyObject(request);
            this.blobStore.client().deleteObject(this.blobStore.bucket(), this.buildKey(sourceBlobName));
        }
        catch (AmazonS3Exception e) {
            throw new IOException(e);
        }
    }

    public Map<String, BlobMetaData> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    protected String buildKey(String blobName) {
        return this.keyPath + blobName;
    }

    <T> T doPrivileged(PrivilegedExceptionAction<T> operation) throws IOException {
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            sm.checkPermission((Permission)new SpecialPermission());
        }
        try {
            return AccessController.doPrivileged(operation);
        }
        catch (PrivilegedActionException e) {
            throw (IOException)e.getException();
        }
    }
}

