/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cloud.aws.blobstore;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CreateBucketRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.StorageClass;
import java.util.ArrayList;
import java.util.Locale;
import org.elasticsearch.cloud.aws.blobstore.S3BlobContainer;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;

public class S3BlobStore
extends AbstractComponent
implements BlobStore {
    private final AmazonS3 client;
    private final String bucket;
    private final String region;
    private final ByteSizeValue bufferSize;
    private final boolean serverSideEncryption;
    private final int numberOfRetries;
    private final CannedAccessControlList cannedACL;
    private final StorageClass storageClass;

    public S3BlobStore(Settings settings, AmazonS3 client, String bucket, @Nullable String region, boolean serverSideEncryption, ByteSizeValue bufferSize, int maxRetries, String cannedACL, String storageClass) {
        super(settings);
        this.client = client;
        this.bucket = bucket;
        this.region = region;
        this.serverSideEncryption = serverSideEncryption;
        this.bufferSize = bufferSize;
        this.cannedACL = S3BlobStore.initCannedACL(cannedACL);
        this.numberOfRetries = maxRetries;
        this.storageClass = S3BlobStore.initStorageClass(storageClass);
        for (int retry = 0; retry <= maxRetries; ++retry) {
            try {
                if (client.doesBucketExist(bucket)) break;
                this.deprecationLogger.deprecated("Auto creation of the bucket for an s3 backed repository is deprecated and will be removed in 6.0.", new Object[0]);
                CreateBucketRequest request = null;
                request = region != null ? new CreateBucketRequest(bucket, region) : new CreateBucketRequest(bucket);
                request.setCannedAcl(this.cannedACL);
                client.createBucket(request);
                break;
            }
            catch (AmazonClientException e) {
                if (this.shouldRetry(e) && retry < maxRetries) {
                    continue;
                }
                this.logger.debug("S3 client create bucket failed");
                throw e;
            }
        }
    }

    public String toString() {
        return (this.region == null ? "" : this.region + "/") + this.bucket;
    }

    public AmazonS3 client() {
        return this.client;
    }

    public String bucket() {
        return this.bucket;
    }

    public boolean serverSideEncryption() {
        return this.serverSideEncryption;
    }

    public int bufferSizeInBytes() {
        return this.bufferSize.bytesAsInt();
    }

    public int numberOfRetries() {
        return this.numberOfRetries;
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new S3BlobContainer(path, this);
    }

    public void delete(BlobPath path) {
        ObjectListing prevListing = null;
        DeleteObjectsRequest multiObjectDeleteRequest = null;
        ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        while (true) {
            ObjectListing list;
            if (prevListing != null) {
                list = this.client.listNextBatchOfObjects(prevListing);
            } else {
                list = this.client.listObjects(this.bucket, path.buildAsString());
                multiObjectDeleteRequest = new DeleteObjectsRequest(list.getBucketName());
            }
            for (S3ObjectSummary summary : list.getObjectSummaries()) {
                keys.add(new DeleteObjectsRequest.KeyVersion(summary.getKey()));
                if (keys.size() <= 500) continue;
                multiObjectDeleteRequest.setKeys(keys);
                this.client.deleteObjects(multiObjectDeleteRequest);
                multiObjectDeleteRequest = new DeleteObjectsRequest(list.getBucketName());
                keys.clear();
            }
            if (!list.isTruncated()) break;
            prevListing = list;
        }
        if (!keys.isEmpty()) {
            multiObjectDeleteRequest.setKeys(keys);
            this.client.deleteObjects(multiObjectDeleteRequest);
        }
    }

    protected boolean shouldRetry(AmazonClientException e) {
        AmazonS3Exception s3e;
        if (e instanceof AmazonS3Exception && (s3e = (AmazonS3Exception)e).getStatusCode() == 400 && "RequestTimeout".equals(s3e.getErrorCode())) {
            return true;
        }
        return e.isRetryable();
    }

    public void close() {
    }

    public CannedAccessControlList getCannedACL() {
        return this.cannedACL;
    }

    public StorageClass getStorageClass() {
        return this.storageClass;
    }

    public static StorageClass initStorageClass(String storageClass) {
        if (storageClass == null || storageClass.equals("")) {
            return StorageClass.Standard;
        }
        try {
            StorageClass _storageClass = StorageClass.fromValue((String)storageClass.toUpperCase(Locale.ENGLISH));
            if (_storageClass.equals((Object)StorageClass.Glacier)) {
                throw new BlobStoreException("Glacier storage class is not supported");
            }
            return _storageClass;
        }
        catch (IllegalArgumentException illegalArgumentException) {
            throw new BlobStoreException("`" + storageClass + "` is not a valid S3 Storage Class.");
        }
    }

    public static CannedAccessControlList initCannedACL(String cannedACL) {
        if (cannedACL == null || cannedACL.equals("")) {
            return CannedAccessControlList.Private;
        }
        for (CannedAccessControlList cur : CannedAccessControlList.values()) {
            if (!cur.toString().equalsIgnoreCase(cannedACL)) continue;
            return cur;
        }
        throw new BlobStoreException("cannedACL is not valid: [" + cannedACL + "]");
    }
}

