/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.hdfs;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.security.AccessController;
import java.util.Locale;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.AbstractFileSystem;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.io.retry.FailoverProxyProvider;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.SpecialPermission;
import org.elasticsearch.cluster.metadata.RepositoryMetaData;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.env.Environment;
import org.elasticsearch.repositories.blobstore.BlobStoreRepository;
import org.elasticsearch.repositories.hdfs.HdfsBlobStore;
import org.elasticsearch.repositories.hdfs.HdfsSecurityContext;
import org.elasticsearch.threadpool.ThreadPool;

public final class HdfsRepository
extends BlobStoreRepository {
    private static final Logger logger = LogManager.getLogger(HdfsRepository.class);
    private static final String CONF_SECURITY_PRINCIPAL = "security.principal";
    private final Environment environment;
    private final ByteSizeValue chunkSize;
    private final BlobPath basePath = BlobPath.cleanPath();
    private final URI uri;
    private final String pathSetting;
    private static final ByteSizeValue DEFAULT_BUFFER_SIZE = new ByteSizeValue(100L, ByteSizeUnit.KB);

    public HdfsRepository(RepositoryMetaData metadata, Environment environment, NamedXContentRegistry namedXContentRegistry, ThreadPool threadPool) {
        super(metadata, environment.settings(), metadata.settings().getAsBoolean("compress", Boolean.valueOf(false)).booleanValue(), namedXContentRegistry, threadPool);
        this.environment = environment;
        this.chunkSize = metadata.settings().getAsBytesSize("chunk_size", null);
        String uriSetting = this.getMetadata().settings().get("uri");
        if (!Strings.hasText((String)uriSetting)) {
            throw new IllegalArgumentException("No 'uri' defined for hdfs snapshot/restore");
        }
        this.uri = URI.create(uriSetting);
        if (!"hdfs".equalsIgnoreCase(this.uri.getScheme())) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Invalid scheme [%s] specified in uri [%s]; only 'hdfs' uri allowed for hdfs snapshot/restore", this.uri.getScheme(), uriSetting));
        }
        if (Strings.hasLength((String)this.uri.getPath()) && !this.uri.getPath().equals("/")) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Use 'path' option to specify a path [%s], not the uri [%s] for hdfs snapshot/restore", this.uri.getPath(), uriSetting));
        }
        this.pathSetting = this.getMetadata().settings().get("path");
        if (this.pathSetting == null) {
            throw new IllegalArgumentException("No 'path' defined for hdfs snapshot/restore");
        }
    }

    private HdfsBlobStore createBlobstore(URI uri, String path, Settings repositorySettings) {
        Configuration hadoopConfiguration = new Configuration(repositorySettings.getAsBoolean("load_defaults", Boolean.valueOf(true)).booleanValue());
        hadoopConfiguration.setClassLoader(HdfsRepository.class.getClassLoader());
        hadoopConfiguration.reloadConfiguration();
        Settings confSettings = repositorySettings.getByPrefix("conf.");
        for (String key : confSettings.keySet()) {
            logger.debug("Adding configuration to HDFS Client Configuration : {} = {}", (Object)key, (Object)confSettings.get(key));
            hadoopConfiguration.set(key, confSettings.get(key));
        }
        hadoopConfiguration.setBoolean("fs.hdfs.impl.disable.cache", true);
        UserGroupInformation ugi = this.login(hadoopConfiguration, repositorySettings);
        String host = uri.getHost();
        String configKey = "dfs.client.failover.proxy.provider." + host;
        Class ret = hadoopConfiguration.getClass(configKey, null, FailoverProxyProvider.class);
        boolean haEnabled = ret != null;
        int bufferSize = repositorySettings.getAsBytesSize("buffer_size", DEFAULT_BUFFER_SIZE).bytesAsInt();
        FileContext fileContext = (FileContext)ugi.doAs(() -> {
            try {
                AbstractFileSystem fs = AbstractFileSystem.get((URI)uri, (Configuration)hadoopConfiguration);
                return FileContext.getFileContext((AbstractFileSystem)fs, (Configuration)hadoopConfiguration);
            }
            catch (UnsupportedFileSystemException e) {
                throw new UncheckedIOException((IOException)((Object)e));
            }
        });
        logger.debug("Using file-system [{}] for URI [{}], path [{}]", (Object)fileContext.getDefaultFileSystem(), (Object)fileContext.getDefaultFileSystem().getUri(), (Object)path);
        try {
            return new HdfsBlobStore(fileContext, path, bufferSize, this.isReadOnly(), haEnabled);
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format(Locale.ROOT, "Cannot create HDFS repository for uri [%s]", uri), e);
        }
    }

    private UserGroupInformation login(Configuration hadoopConfiguration, Settings repositorySettings) {
        UserGroupInformation.AuthenticationMethod authMethod = SecurityUtil.getAuthenticationMethod((Configuration)hadoopConfiguration);
        if (!authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE) && !authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS)) {
            throw new RuntimeException("Unsupported authorization mode [" + authMethod + "]");
        }
        String kerberosPrincipal = repositorySettings.get(CONF_SECURITY_PRINCIPAL);
        if (kerberosPrincipal != null && authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.SIMPLE)) {
            logger.warn("Hadoop authentication method is set to [SIMPLE], but a Kerberos principal is specified. Continuing with [KERBEROS] authentication.");
            SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)hadoopConfiguration);
        } else if (kerberosPrincipal == null && authMethod.equals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS)) {
            throw new RuntimeException("HDFS Repository does not support [KERBEROS] authentication without a valid Kerberos principal and keytab. Please specify a principal in the repository settings with [security.principal].");
        }
        UserGroupInformation.setConfiguration((Configuration)hadoopConfiguration);
        logger.debug("Hadoop security enabled: [{}]", (Object)UserGroupInformation.isSecurityEnabled());
        logger.debug("Using Hadoop authentication method: [{}]", (Object)SecurityUtil.getAuthenticationMethod((Configuration)hadoopConfiguration));
        try {
            if (UserGroupInformation.isSecurityEnabled()) {
                String principal = HdfsRepository.preparePrincipal(kerberosPrincipal);
                String keytab = HdfsSecurityContext.locateKeytabFile(this.environment).toString();
                logger.debug("Using kerberos principal [{}] and keytab located at [{}]", (Object)principal, (Object)keytab);
                return UserGroupInformation.loginUserFromKeytabAndReturnUGI((String)principal, (String)keytab);
            }
            return UserGroupInformation.getCurrentUser();
        }
        catch (IOException e) {
            throw new UncheckedIOException("Could not retrieve the current user information", e);
        }
    }

    private static String preparePrincipal(String originalPrincipal) {
        String finalPrincipal = originalPrincipal;
        if (originalPrincipal.contains("_HOST")) {
            try {
                finalPrincipal = SecurityUtil.getServerPrincipal((String)originalPrincipal, (String)HdfsRepository.getHostName());
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            if (!originalPrincipal.equals(finalPrincipal)) {
                logger.debug("Found service principal. Converted original principal name [{}] to server principal [{}]", (Object)originalPrincipal, (Object)finalPrincipal);
            }
        }
        return finalPrincipal;
    }

    @SuppressForbidden(reason="InetAddress.getLocalHost(); Needed for filling in hostname for a kerberos principal name pattern.")
    private static String getHostName() {
        try {
            return InetAddress.getLocalHost().getCanonicalHostName();
        }
        catch (UnknownHostException e) {
            throw new RuntimeException("Could not locate host information", e);
        }
    }

    protected HdfsBlobStore createBlobStore() {
        SpecialPermission.check();
        HdfsBlobStore blobStore = AccessController.doPrivileged(() -> this.createBlobstore(this.uri, this.pathSetting, this.getMetadata().settings()));
        return blobStore;
    }

    protected BlobPath basePath() {
        return this.basePath;
    }

    protected ByteSizeValue chunkSize() {
        return this.chunkSize;
    }
}

