/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.hdfs;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.Map;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.repositories.hdfs.HdfsBlobStore;
import org.elasticsearch.repositories.hdfs.HdfsSecurityContext;

final class HdfsBlobContainer
extends AbstractBlobContainer {
    private final HdfsBlobStore store;
    private final HdfsSecurityContext securityContext;
    private final Path path;
    private final int bufferSize;

    HdfsBlobContainer(BlobPath blobPath, HdfsBlobStore store, Path path, int bufferSize, HdfsSecurityContext hdfsSecurityContext) {
        super(blobPath);
        this.store = store;
        this.securityContext = hdfsSecurityContext;
        this.path = path;
        this.bufferSize = bufferSize;
    }

    public boolean blobExists(String blobName) {
        try {
            return this.store.execute(fileContext -> fileContext.util().exists(new Path(this.path, blobName)));
        }
        catch (Exception e) {
            return false;
        }
    }

    public void deleteBlob(String blobName) throws IOException {
        if (!this.blobExists(blobName)) {
            throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
        }
        this.store.execute(fileContext -> fileContext.delete(new Path(this.path, blobName), true));
    }

    public void move(String sourceBlobName, String targetBlobName) throws IOException {
        this.store.execute(fileContext -> {
            fileContext.rename(new Path(this.path, sourceBlobName), new Path(this.path, targetBlobName), new Options.Rename[0]);
            return null;
        });
    }

    public InputStream readBlob(String blobName) throws IOException {
        if (!this.blobExists(blobName)) {
            throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
        }
        return this.store.execute(fileContext -> new HDFSPrivilegedInputSteam((InputStream)fileContext.open(new Path(this.path, blobName), this.bufferSize), this.securityContext));
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize) throws IOException {
        if (this.blobExists(blobName)) {
            throw new FileAlreadyExistsException("blob [" + blobName + "] already exists, cannot overwrite");
        }
        this.store.execute(fileContext -> {
            Path blob = new Path(this.path, blobName);
            EnumSet<CreateFlag> flags = EnumSet.of(CreateFlag.CREATE, CreateFlag.SYNC_BLOCK);
            Options.CreateOpts[] opts = new Options.CreateOpts[]{Options.CreateOpts.bufferSize((int)this.bufferSize)};
            try (FSDataOutputStream stream = fileContext.create(blob, flags, opts);){
                int bytesRead;
                byte[] buffer = new byte[this.bufferSize];
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    stream.write(buffer, 0, bytesRead);
                    stream.hsync();
                }
            }
            return null;
        });
    }

    public Map<String, BlobMetaData> listBlobsByPrefix(@Nullable String prefix) throws IOException {
        FileStatus[] files = this.store.execute(fileContext -> fileContext.util().listStatus(this.path, path -> prefix == null || path.getName().startsWith(prefix)));
        LinkedHashMap<String, PlainBlobMetaData> map = new LinkedHashMap<String, PlainBlobMetaData>();
        for (FileStatus file : files) {
            map.put(file.getPath().getName(), new PlainBlobMetaData(file.getPath().getName(), file.getLen()));
        }
        return Collections.unmodifiableMap(map);
    }

    public Map<String, BlobMetaData> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    private static class HDFSPrivilegedInputSteam
    extends FilterInputStream {
        private final HdfsSecurityContext securityContext;

        HDFSPrivilegedInputSteam(InputStream in, HdfsSecurityContext hdfsSecurityContext) {
            super(in);
            this.securityContext = hdfsSecurityContext;
        }

        @Override
        public int read() throws IOException {
            return this.securityContext.doPrivilegedOrThrow(this.in::read);
        }

        @Override
        public int read(byte[] b) throws IOException {
            return this.securityContext.doPrivilegedOrThrow(() -> this.in.read(b));
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return this.securityContext.doPrivilegedOrThrow(() -> this.in.read(b, off, len));
        }

        @Override
        public long skip(long n) throws IOException {
            return this.securityContext.doPrivilegedOrThrow(() -> this.in.skip(n));
        }

        @Override
        public int available() throws IOException {
            return this.securityContext.doPrivilegedOrThrow(() -> this.in.available());
        }

        @Override
        public synchronized void reset() throws IOException {
            this.securityContext.doPrivilegedOrThrow(() -> {
                this.in.reset();
                return null;
            });
        }
    }
}

