/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.gcs;

import com.google.cloud.ReadChannel;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.common.CheckedRunnable;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.repositories.gcs.SocketAccess;

class GoogleCloudStorageRetryingInputStream
extends InputStream {
    private static final Logger logger = LogManager.getLogger(GoogleCloudStorageRetryingInputStream.class);
    static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final Storage client;
    private final BlobId blobId;
    private final int maxRetries;
    private InputStream currentStream;
    private int attempt = 1;
    private List<StorageException> failures = new ArrayList<StorageException>(10);
    private long currentOffset;
    private boolean closed;

    GoogleCloudStorageRetryingInputStream(Storage client, BlobId blobId) throws IOException {
        this.client = client;
        this.blobId = blobId;
        this.maxRetries = ((StorageOptions)client.getOptions()).getRetrySettings().getMaxAttempts() + 1;
        this.currentStream = this.openStream();
    }

    private InputStream openStream() throws IOException {
        try {
            final ReadChannel readChannel = SocketAccess.doPrivilegedIOException(() -> this.client.reader(this.blobId, new Storage.BlobSourceOption[0]));
            if (this.currentOffset > 0L) {
                readChannel.seek(this.currentOffset);
            }
            return Channels.newInputStream(new ReadableByteChannel(){

                @Override
                @SuppressForbidden(reason="Channel is based of a socket not a file")
                public int read(ByteBuffer dst) throws IOException {
                    try {
                        return SocketAccess.doPrivilegedIOException(() -> readChannel.read(dst));
                    }
                    catch (StorageException e) {
                        if (e.getCode() == 404) {
                            throw new NoSuchFileException("Blob [" + GoogleCloudStorageRetryingInputStream.this.blobId.getName() + "] does not exist");
                        }
                        throw e;
                    }
                }

                @Override
                public boolean isOpen() {
                    return readChannel.isOpen();
                }

                @Override
                public void close() throws IOException {
                    SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> ((ReadChannel)readChannel).close()));
                }
            });
        }
        catch (StorageException e) {
            throw this.addSuppressedExceptions(e);
        }
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int result = this.currentStream.read();
                ++this.currentOffset;
                return result;
            }
            catch (StorageException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead == -1) {
                    return -1;
                }
                this.currentOffset += (long)bytesRead;
                return bytesRead;
            }
            catch (StorageException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using GoogleCloudStorageRetryingInputStream after close";
            throw new IllegalStateException("using GoogleCloudStorageRetryingInputStream after close");
        }
    }

    private void reopenStreamOrFail(StorageException e) throws IOException {
        if (this.attempt >= this.maxRetries) {
            throw this.addSuppressedExceptions(e);
        }
        logger.debug((Message)new ParameterizedMessage("failed reading [{}] at offset [{}], attempt [{}] of [{}], retrying", new Object[]{this.blobId, this.currentOffset, this.attempt, 10}), (Throwable)e);
        ++this.attempt;
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.currentStream});
        this.currentStream = this.openStream();
    }

    @Override
    public void close() throws IOException {
        this.currentStream.close();
        this.closed = true;
    }

    @Override
    public long skip(long n) {
        throw new UnsupportedOperationException("GoogleCloudStorageRetryingInputStream does not support seeking");
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("GoogleCloudStorageRetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (StorageException failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }
}

