/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.gcs;

import com.google.api.gax.paging.Page;
import com.google.cloud.BatchResult;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageBatch;
import com.google.cloud.storage.StorageException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.CheckedRunnable;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetadata;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.support.PlainBlobMetadata;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.internal.io.Streams;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageBlobContainer;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageOperationsStats;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageRetryingInputStream;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageService;
import org.elasticsearch.repositories.gcs.SocketAccess;

class GoogleCloudStorageBlobStore
implements BlobStore {
    private static final Logger logger = LogManager.getLogger(GoogleCloudStorageBlobStore.class);
    public static final int LARGE_BLOB_THRESHOLD_BYTE_SIZE;
    private final String bucketName;
    private final String clientName;
    private final String repositoryName;
    private final GoogleCloudStorageService storageService;
    private final GoogleCloudStorageOperationsStats stats;
    private final int bufferSize;

    GoogleCloudStorageBlobStore(String bucketName, String clientName, String repositoryName, GoogleCloudStorageService storageService, int bufferSize) {
        this.bucketName = bucketName;
        this.clientName = clientName;
        this.repositoryName = repositoryName;
        this.storageService = storageService;
        this.stats = new GoogleCloudStorageOperationsStats(bucketName);
        this.bufferSize = bufferSize;
    }

    private Storage client() throws IOException {
        return this.storageService.client(this.clientName, this.repositoryName, this.stats);
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new GoogleCloudStorageBlobContainer(path, this);
    }

    public void close() {
        this.storageService.closeRepositoryClient(this.repositoryName);
    }

    Map<String, BlobMetadata> listBlobs(String path) throws IOException {
        return this.listBlobsByPrefix(path, "");
    }

    Map<String, BlobMetadata> listBlobsByPrefix(String path, String prefix) throws IOException {
        String pathPrefix = GoogleCloudStorageBlobStore.buildKey(path, prefix);
        MapBuilder mapBuilder = MapBuilder.newMapBuilder();
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)pathPrefix)}).iterateAll().forEach(blob -> {
            assert (blob.getName().startsWith(path));
            if (!blob.isDirectory()) {
                String suffixName = blob.getName().substring(path.length());
                mapBuilder.put((Object)suffixName, (Object)new PlainBlobMetadata(suffixName, blob.getSize().longValue()));
            }
        })));
        return mapBuilder.immutableMap();
    }

    Map<String, BlobContainer> listChildren(BlobPath path) throws IOException {
        String pathStr = path.buildAsString();
        MapBuilder mapBuilder = MapBuilder.newMapBuilder();
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)pathStr)}).iterateAll().forEach(blob -> {
            if (blob.isDirectory()) {
                assert (blob.getName().startsWith(pathStr));
                assert (blob.getName().endsWith("/"));
                String suffixName = blob.getName().substring(pathStr.length(), blob.getName().length() - 1);
                if (!suffixName.isEmpty()) {
                    mapBuilder.put((Object)suffixName, (Object)new GoogleCloudStorageBlobContainer(path.add(suffixName), this));
                }
            }
        })));
        return mapBuilder.immutableMap();
    }

    boolean blobExists(String blobName) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        Blob blob = SocketAccess.doPrivilegedIOException(() -> this.client().get(blobId));
        return blob != null;
    }

    InputStream readBlob(String blobName) throws IOException {
        return new GoogleCloudStorageRetryingInputStream(this.client(), BlobId.of((String)this.bucketName, (String)blobName));
    }

    InputStream readBlob(String blobName, long position, long length) throws IOException {
        if (position < 0L) {
            throw new IllegalArgumentException("position must be non-negative");
        }
        if (length < 0L) {
            throw new IllegalArgumentException("length must be non-negative");
        }
        if (length == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        return new GoogleCloudStorageRetryingInputStream(this.client(), BlobId.of((String)this.bucketName, (String)blobName), position, Math.addExact(position, length - 1L));
    }

    void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        BlobInfo blobInfo = BlobInfo.newBuilder((String)this.bucketName, (String)blobName).build();
        if (blobSize > this.getLargeBlobThresholdInBytes()) {
            this.writeBlobResumable(blobInfo, inputStream, blobSize, failIfAlreadyExists);
        } else {
            this.writeBlobMultipart(blobInfo, inputStream, blobSize, failIfAlreadyExists);
        }
    }

    long getLargeBlobThresholdInBytes() {
        return LARGE_BLOB_THRESHOLD_BYTE_SIZE;
    }

    private void writeBlobResumable(BlobInfo blobInfo, InputStream inputStream, long size, boolean failIfAlreadyExists) throws IOException {
        Storage.BlobWriteOption[] blobWriteOptionArray;
        assert (inputStream.markSupported());
        inputStream.mark(Integer.MAX_VALUE);
        byte[] buffer = new byte[size < (long)this.bufferSize ? Math.toIntExact(size) : this.bufferSize];
        StorageException storageException = null;
        if (failIfAlreadyExists) {
            Storage.BlobWriteOption[] blobWriteOptionArray2 = new Storage.BlobWriteOption[1];
            blobWriteOptionArray = blobWriteOptionArray2;
            blobWriteOptionArray2[0] = Storage.BlobWriteOption.doesNotExist();
        } else {
            blobWriteOptionArray = new Storage.BlobWriteOption[]{};
        }
        Storage.BlobWriteOption[] writeOptions = blobWriteOptionArray;
        for (int retry = 0; retry < 3; ++retry) {
            try {
                final WriteChannel writeChannel = SocketAccess.doPrivilegedIOException(() -> this.client().writer(blobInfo, writeOptions));
                Streams.copy((InputStream)inputStream, (OutputStream)Channels.newOutputStream(new WritableByteChannel(){

                    @Override
                    @SuppressForbidden(reason="channel is based on a socket")
                    public int write(ByteBuffer src) throws IOException {
                        return SocketAccess.doPrivilegedIOException(() -> writeChannel.write(src));
                    }

                    @Override
                    public boolean isOpen() {
                        return writeChannel.isOpen();
                    }

                    @Override
                    public void close() throws IOException {
                        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> writeChannel.close()));
                    }
                }), (byte[])buffer);
                this.stats.trackPutOperation();
                return;
            }
            catch (StorageException se) {
                int errorCode = se.getCode();
                if (errorCode != 410) {
                    if (failIfAlreadyExists && errorCode == 412) {
                        throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
                    }
                    if (storageException != null) {
                        se.addSuppressed(storageException);
                    }
                    throw se;
                }
                logger.warn(() -> new ParameterizedMessage("Retrying broken resumable upload session for blob {}", (Object)blobInfo), (Throwable)se);
                storageException = (StorageException)ExceptionsHelper.useOrSuppress(storageException, (Throwable)se);
                inputStream.reset();
                continue;
            }
        }
        assert (storageException != null);
        throw storageException;
    }

    private void writeBlobMultipart(BlobInfo blobInfo, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (blobSize <= this.getLargeBlobThresholdInBytes()) : "large blob uploads should use the resumable upload method";
        byte[] buffer = new byte[Math.toIntExact(blobSize)];
        org.elasticsearch.common.io.Streams.readFully((InputStream)inputStream, (byte[])buffer);
        try {
            Storage.BlobTargetOption[] blobTargetOptionArray;
            if (failIfAlreadyExists) {
                Storage.BlobTargetOption[] blobTargetOptionArray2 = new Storage.BlobTargetOption[1];
                blobTargetOptionArray = blobTargetOptionArray2;
                blobTargetOptionArray2[0] = Storage.BlobTargetOption.doesNotExist();
            } else {
                blobTargetOptionArray = new Storage.BlobTargetOption[]{};
            }
            Storage.BlobTargetOption[] targetOptions = blobTargetOptionArray;
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().create(blobInfo, buffer, targetOptions)));
            this.stats.trackPostOperation();
        }
        catch (StorageException se) {
            if (failIfAlreadyExists && se.getCode() == 412) {
                throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
            }
            throw se;
        }
    }

    DeleteResult deleteDirectory(String pathStr) throws IOException {
        return SocketAccess.doPrivilegedIOException(() -> {
            DeleteResult deleteResult = DeleteResult.ZERO;
            Page page = this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)pathStr)});
            do {
                ArrayList<String> blobsToDelete = new ArrayList<String>();
                AtomicLong blobsDeleted = new AtomicLong(0L);
                AtomicLong bytesDeleted = new AtomicLong(0L);
                page.getValues().forEach(b -> {
                    blobsToDelete.add(b.getName());
                    blobsDeleted.incrementAndGet();
                    bytesDeleted.addAndGet(b.getSize());
                });
                this.deleteBlobsIgnoringIfNotExists(blobsToDelete);
                deleteResult = deleteResult.add(blobsDeleted.get(), bytesDeleted.get());
            } while ((page = page.getNextPage()) != null);
            return deleteResult;
        });
    }

    void deleteBlobsIgnoringIfNotExists(Collection<String> blobNames) throws IOException {
        if (blobNames.isEmpty()) {
            return;
        }
        List blobIdsToDelete = blobNames.stream().map(blob -> BlobId.of((String)this.bucketName, (String)blob)).collect(Collectors.toList());
        final List failedBlobs = Collections.synchronizedList(new ArrayList());
        try {
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> {
                final AtomicReference ioe = new AtomicReference();
                StorageBatch batch = this.client().batch();
                for (final BlobId blob : blobIdsToDelete) {
                    batch.delete(blob, new Storage.BlobSourceOption[0]).notify((BatchResult.Callback)new BatchResult.Callback<Boolean, StorageException>(){

                        public void success(Boolean result) {
                        }

                        public void error(StorageException exception) {
                            if (exception.getCode() != 404) {
                                failedBlobs.add(blob);
                                if (!ioe.compareAndSet(null, exception)) {
                                    ((StorageException)((Object)ioe.get())).addSuppressed((Throwable)exception);
                                }
                            }
                        }
                    });
                }
                batch.submit();
                StorageException exception = (StorageException)((Object)((Object)ioe.get()));
                if (exception != null) {
                    throw exception;
                }
            }));
        }
        catch (Exception e) {
            throw new IOException("Exception when deleting blobs [" + failedBlobs + "]", e);
        }
        assert (failedBlobs.isEmpty());
    }

    private static String buildKey(String keyPath, String s) {
        assert (s != null);
        return keyPath + s;
    }

    public Map<String, Long> stats() {
        return this.stats.toMap();
    }

    static {
        String key = "es.repository_gcs.large_blob_threshold_byte_size";
        String largeBlobThresholdByteSizeProperty = System.getProperty("es.repository_gcs.large_blob_threshold_byte_size");
        if (largeBlobThresholdByteSizeProperty == null) {
            LARGE_BLOB_THRESHOLD_BYTE_SIZE = Math.toIntExact(new ByteSizeValue(5L, ByteSizeUnit.MB).getBytes());
        } else {
            int largeBlobThresholdByteSize;
            try {
                largeBlobThresholdByteSize = Integer.parseInt(largeBlobThresholdByteSizeProperty);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("failed to parse es.repository_gcs.large_blob_threshold_byte_size having value [" + largeBlobThresholdByteSizeProperty + "]");
            }
            if (largeBlobThresholdByteSize <= 0) {
                throw new IllegalArgumentException("es.repository_gcs.large_blob_threshold_byte_size must be positive but was [" + largeBlobThresholdByteSizeProperty + "]");
            }
            LARGE_BLOB_THRESHOLD_BYTE_SIZE = largeBlobThresholdByteSize;
        }
    }
}

