/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.gcs;

import com.google.cloud.ReadChannel;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.elasticsearch.common.CheckedRunnable;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.core.internal.io.Streams;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageBlobContainer;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageService;
import org.elasticsearch.repositories.gcs.SocketAccess;

class GoogleCloudStorageBlobStore
extends AbstractComponent
implements BlobStore {
    private static final int LARGE_BLOB_THRESHOLD_BYTE_SIZE = 0x500000;
    private final String bucketName;
    private final String clientName;
    private final GoogleCloudStorageService storageService;

    GoogleCloudStorageBlobStore(String bucketName, String clientName, GoogleCloudStorageService storageService) {
        this.bucketName = bucketName;
        this.clientName = clientName;
        this.storageService = storageService;
        if (!this.doesBucketExist(bucketName)) {
            throw new BlobStoreException("Bucket [" + bucketName + "] does not exist");
        }
    }

    private Storage client() throws IOException {
        return this.storageService.client(this.clientName);
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new GoogleCloudStorageBlobContainer(path, this);
    }

    public void delete(BlobPath path) throws IOException {
        this.deleteBlobsByPrefix(path.buildAsString());
    }

    public void close() {
    }

    boolean doesBucketExist(String bucketName) {
        try {
            Bucket bucket = SocketAccess.doPrivilegedIOException(() -> this.client().get(bucketName, new Storage.BucketGetOption[0]));
            return bucket != null;
        }
        catch (Exception e) {
            throw new BlobStoreException("Unable to check if bucket [" + bucketName + "] exists", (Throwable)e);
        }
    }

    Map<String, BlobMetaData> listBlobs(String path) throws IOException {
        return this.listBlobsByPrefix(path, "");
    }

    Map<String, BlobMetaData> listBlobsByPrefix(String path, String prefix) throws IOException {
        String pathPrefix = GoogleCloudStorageBlobStore.buildKey(path, prefix);
        MapBuilder mapBuilder = MapBuilder.newMapBuilder();
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().get(this.bucketName, new Storage.BucketGetOption[0]).list(new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)pathPrefix)}).iterateAll().forEach(blob -> {
            assert (blob.getName().startsWith(path));
            String suffixName = blob.getName().substring(path.length());
            mapBuilder.put((Object)suffixName, (Object)new PlainBlobMetaData(suffixName, blob.getSize().longValue()));
        })));
        return mapBuilder.immutableMap();
    }

    boolean blobExists(String blobName) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        Blob blob = SocketAccess.doPrivilegedIOException(() -> this.client().get(blobId));
        return blob != null;
    }

    InputStream readBlob(final String blobName) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        final ReadChannel readChannel = SocketAccess.doPrivilegedIOException(() -> this.client().reader(blobId, new Storage.BlobSourceOption[0]));
        return Channels.newInputStream(new ReadableByteChannel(){

            @Override
            @SuppressForbidden(reason="Channel is based of a socket not a file")
            public int read(ByteBuffer dst) throws IOException {
                try {
                    return SocketAccess.doPrivilegedIOException(() -> readChannel.read(dst));
                }
                catch (StorageException e) {
                    if (e.getCode() == 404) {
                        throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
                    }
                    throw e;
                }
            }

            @Override
            public boolean isOpen() {
                return readChannel.isOpen();
            }

            @Override
            public void close() throws IOException {
                SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> ((ReadChannel)readChannel).close()));
            }
        });
    }

    void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        BlobInfo blobInfo = BlobInfo.newBuilder((String)this.bucketName, (String)blobName).build();
        if (blobSize > 0x500000L) {
            this.writeBlobResumable(blobInfo, inputStream, failIfAlreadyExists);
        } else {
            this.writeBlobMultipart(blobInfo, inputStream, blobSize, failIfAlreadyExists);
        }
    }

    private void writeBlobResumable(BlobInfo blobInfo, InputStream inputStream, boolean failIfAlreadyExists) throws IOException {
        try {
            Storage.BlobWriteOption[] blobWriteOptionArray;
            if (failIfAlreadyExists) {
                Storage.BlobWriteOption[] blobWriteOptionArray2 = new Storage.BlobWriteOption[1];
                blobWriteOptionArray = blobWriteOptionArray2;
                blobWriteOptionArray2[0] = Storage.BlobWriteOption.doesNotExist();
            } else {
                blobWriteOptionArray = new Storage.BlobWriteOption[]{};
            }
            Storage.BlobWriteOption[] writeOptions = blobWriteOptionArray;
            final WriteChannel writeChannel = SocketAccess.doPrivilegedIOException(() -> this.client().writer(blobInfo, writeOptions));
            Streams.copy((InputStream)inputStream, (OutputStream)Channels.newOutputStream(new WritableByteChannel(){

                @Override
                public boolean isOpen() {
                    return writeChannel.isOpen();
                }

                @Override
                public void close() throws IOException {
                    SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> writeChannel.close()));
                }

                @Override
                @SuppressForbidden(reason="Channel is based of a socket not a file")
                public int write(ByteBuffer src) throws IOException {
                    return SocketAccess.doPrivilegedIOException(() -> writeChannel.write(src));
                }
            }));
        }
        catch (StorageException se) {
            if (failIfAlreadyExists && se.getCode() == 412) {
                throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
            }
            throw se;
        }
    }

    private void writeBlobMultipart(BlobInfo blobInfo, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (blobSize <= 0x500000L) : "large blob uploads should use the resumable upload method";
        ByteArrayOutputStream baos = new ByteArrayOutputStream(Math.toIntExact(blobSize));
        Streams.copy((InputStream)inputStream, (OutputStream)baos);
        try {
            Storage.BlobTargetOption[] blobTargetOptionArray;
            if (failIfAlreadyExists) {
                Storage.BlobTargetOption[] blobTargetOptionArray2 = new Storage.BlobTargetOption[1];
                blobTargetOptionArray = blobTargetOptionArray2;
                blobTargetOptionArray2[0] = Storage.BlobTargetOption.doesNotExist();
            } else {
                blobTargetOptionArray = new Storage.BlobTargetOption[]{};
            }
            Storage.BlobTargetOption[] targetOptions = blobTargetOptionArray;
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().create(blobInfo, baos.toByteArray(), targetOptions)));
        }
        catch (StorageException se) {
            if (failIfAlreadyExists && se.getCode() == 412) {
                throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
            }
            throw se;
        }
    }

    void deleteBlob(String blobName) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        boolean deleted = SocketAccess.doPrivilegedIOException(() -> this.client().delete(blobId));
        if (!deleted) {
            throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
        }
    }

    void deleteBlobsByPrefix(String prefix) throws IOException {
        this.deleteBlobs(this.listBlobsByPrefix("", prefix).keySet());
    }

    void deleteBlobs(Collection<String> blobNames) throws IOException {
        if (blobNames.isEmpty()) {
            return;
        }
        if (blobNames.size() == 1) {
            this.deleteBlob(blobNames.iterator().next());
            return;
        }
        List blobIdsToDelete = blobNames.stream().map(blob -> BlobId.of((String)this.bucketName, (String)blob)).collect(Collectors.toList());
        List deletedStatuses = SocketAccess.doPrivilegedIOException(() -> this.client().delete((Iterable)blobIdsToDelete));
        assert (blobIdsToDelete.size() == deletedStatuses.size());
        boolean failed = false;
        for (int i = 0; i < blobIdsToDelete.size(); ++i) {
            if (((Boolean)deletedStatuses.get(i)).booleanValue()) continue;
            this.logger.error("Failed to delete blob [{}] in bucket [{}]", (Object)((BlobId)blobIdsToDelete.get(i)).getName(), (Object)this.bucketName);
            failed = true;
        }
        if (failed) {
            throw new IOException("Failed to delete all [" + blobIdsToDelete.size() + "] blobs");
        }
    }

    private static String buildKey(String keyPath, String s) {
        assert (s != null);
        return keyPath + s;
    }
}

