/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.gcs;

import com.google.api.client.googleapis.batch.BatchRequest;
import com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.http.AbstractInputStreamContent;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.InputStreamContent;
import com.google.api.services.storage.Storage;
import com.google.api.services.storage.model.Bucket;
import com.google.api.services.storage.model.Objects;
import com.google.api.services.storage.model.StorageObject;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.elasticsearch.common.CheckedRunnable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageBlobContainer;
import org.elasticsearch.repositories.gcs.SocketAccess;

class GoogleCloudStorageBlobStore
extends AbstractComponent
implements BlobStore {
    private static final int MAX_BATCHING_REQUESTS = 999;
    private final Storage client;
    private final String bucket;

    GoogleCloudStorageBlobStore(Settings settings, String bucket, Storage storageClient) {
        super(settings);
        this.bucket = bucket;
        this.client = storageClient;
        if (!this.doesBucketExist(bucket)) {
            throw new BlobStoreException("Bucket [" + bucket + "] does not exist");
        }
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new GoogleCloudStorageBlobContainer(path, this);
    }

    public void delete(BlobPath path) throws IOException {
        this.deleteBlobsByPrefix(path.buildAsString());
    }

    public void close() {
    }

    boolean doesBucketExist(String bucketName) {
        try {
            return SocketAccess.doPrivilegedIOException(() -> {
                try {
                    Bucket bucket = (Bucket)this.client.buckets().get(bucketName).execute();
                    if (bucket != null) {
                        return Strings.hasText((String)bucket.getId());
                    }
                }
                catch (GoogleJsonResponseException e) {
                    GoogleJsonError error = e.getDetails();
                    if (e.getStatusCode() == 404 || error != null && error.getCode() == 404) {
                        return false;
                    }
                    throw e;
                }
                return false;
            });
        }
        catch (IOException e) {
            throw new BlobStoreException("Unable to check if bucket [" + bucketName + "] exists", (Throwable)e);
        }
    }

    Map<String, BlobMetaData> listBlobs(String path) throws IOException {
        return SocketAccess.doPrivilegedIOException(() -> this.listBlobsByPath(this.bucket, path, path));
    }

    Map<String, BlobMetaData> listBlobsByPrefix(String path, String prefix) throws IOException {
        return SocketAccess.doPrivilegedIOException(() -> this.listBlobsByPath(this.bucket, this.buildKey(path, prefix), path));
    }

    private Map<String, BlobMetaData> listBlobsByPath(String bucketName, String path, String pathToRemove) throws IOException {
        return GoogleCloudStorageBlobStore.blobsStream(this.client, bucketName, path, 999L).map(new BlobMetaDataConverter(pathToRemove)).collect(Collectors.toMap(PlainBlobMetaData::name, Function.identity()));
    }

    boolean blobExists(String blobName) throws IOException {
        try {
            StorageObject blob = SocketAccess.doPrivilegedIOException(() -> (StorageObject)this.client.objects().get(this.bucket, blobName).execute());
            if (blob != null) {
                return Strings.hasText((String)blob.getId());
            }
        }
        catch (GoogleJsonResponseException e) {
            GoogleJsonError error = e.getDetails();
            if (e.getStatusCode() == 404 || error != null && error.getCode() == 404) {
                return false;
            }
            throw e;
        }
        return false;
    }

    InputStream readBlob(String blobName) throws IOException {
        try {
            return SocketAccess.doPrivilegedIOException(() -> {
                Storage.Objects.Get object = this.client.objects().get(this.bucket, blobName);
                return object.executeMediaAsInputStream();
            });
        }
        catch (GoogleJsonResponseException e) {
            GoogleJsonError error = e.getDetails();
            if (e.getStatusCode() == 404 || error != null && error.getCode() == 404) {
                throw new NoSuchFileException(e.getMessage());
            }
            throw e;
        }
    }

    void writeBlob(String blobName, InputStream inputStream, long blobSize) throws IOException {
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> {
            InputStreamContent stream = new InputStreamContent(null, inputStream);
            stream.setLength(blobSize);
            Storage.Objects.Insert insert = this.client.objects().insert(this.bucket, null, (AbstractInputStreamContent)stream);
            insert.setName(blobName);
            insert.execute();
        }));
    }

    void deleteBlob(String blobName) throws IOException {
        if (!this.blobExists(blobName)) {
            throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
        }
        SocketAccess.doPrivilegedIOException(() -> (Void)this.client.objects().delete(this.bucket, blobName).execute());
    }

    void deleteBlobsByPrefix(String prefix) throws IOException {
        this.deleteBlobs(this.listBlobsByPath(this.bucket, prefix, null).keySet());
    }

    void deleteBlobs(Collection<String> blobNames) throws IOException {
        if (blobNames == null || blobNames.isEmpty()) {
            return;
        }
        if (blobNames.size() == 1) {
            this.deleteBlob(blobNames.iterator().next());
            return;
        }
        ArrayList deletions = new ArrayList(Math.min(999, blobNames.size()));
        Iterator<String> blobs = blobNames.iterator();
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> {
            while (blobs.hasNext()) {
                deletions.add(this.client.objects().delete(this.bucket, (String)blobs.next()));
                if (blobs.hasNext() && deletions.size() != 999) continue;
                try {
                    BatchRequest batch = this.client.batch();
                    final CountDown countDown = new CountDown(deletions.size());
                    for (final Storage.Objects.Delete delete : deletions) {
                        delete.queue(batch, (JsonBatchCallback)new JsonBatchCallback<Void>(){

                            public void onFailure(GoogleJsonError e, HttpHeaders responseHeaders) throws IOException {
                                GoogleCloudStorageBlobStore.this.logger.error("failed to delete blob [{}] in bucket [{}]: {}", (Object)delete.getObject(), (Object)delete.getBucket(), (Object)e.getMessage());
                            }

                            public void onSuccess(Void aVoid, HttpHeaders responseHeaders) throws IOException {
                                countDown.countDown();
                            }
                        });
                    }
                    batch.execute();
                    if (countDown.isCountedDown()) continue;
                    throw new IOException("Failed to delete all [" + deletions.size() + "] blobs");
                }
                finally {
                    deletions.clear();
                }
            }
        }));
    }

    void moveBlob(String sourceBlob, String targetBlob) throws IOException {
        SocketAccess.doPrivilegedIOException(() -> {
            this.client.objects().copy(this.bucket, sourceBlob, this.bucket, targetBlob, null).execute();
            this.client.objects().delete(this.bucket, sourceBlob).execute();
            return null;
        });
    }

    private String buildKey(String keyPath, String s) {
        assert (s != null);
        return keyPath + s;
    }

    static Stream<StorageObject> blobsStream(Storage client, String bucketName, String prefix, long pageSize) throws IOException {
        return StreamSupport.stream(new StorageObjectsSpliterator(client, bucketName, prefix, pageSize), false);
    }

    static class StorageObjectsSpliterator
    implements Spliterator<StorageObject> {
        private final Storage.Objects.List list = SocketAccess.doPrivilegedIOException(() -> client.objects().list(bucketName));

        StorageObjectsSpliterator(Storage client, String bucketName, String prefix, long pageSize) throws IOException {
            this.list.setMaxResults(Long.valueOf(pageSize));
            if (prefix != null) {
                this.list.setPrefix(prefix);
            }
        }

        @Override
        public boolean tryAdvance(Consumer<? super StorageObject> action) {
            try {
                Objects objects = SocketAccess.doPrivilegedIOException(() -> ((Storage.Objects.List)this.list).execute());
                if (objects == null || objects.getItems() == null || objects.getItems().isEmpty()) {
                    return false;
                }
                objects.getItems().forEach(action::accept);
                String next = objects.getNextPageToken();
                if (next != null) {
                    this.list.setPageToken(next);
                    return true;
                }
                return false;
            }
            catch (Exception e) {
                throw new BlobStoreException("Exception while listing objects", (Throwable)e);
            }
        }

        @Override
        public Spliterator<StorageObject> trySplit() {
            return null;
        }

        @Override
        public long estimateSize() {
            return Long.MAX_VALUE;
        }

        @Override
        public int characteristics() {
            return 0;
        }
    }

    class BlobMetaDataConverter
    implements Function<StorageObject, PlainBlobMetaData> {
        private final String pathToRemove;

        BlobMetaDataConverter(String pathToRemove) {
            this.pathToRemove = pathToRemove;
        }

        @Override
        public PlainBlobMetaData apply(StorageObject storageObject) {
            String blobName = storageObject.getName();
            if (Strings.hasLength((String)this.pathToRemove)) {
                blobName = blobName.substring(this.pathToRemove.length());
            }
            return new PlainBlobMetaData(blobName, storageObject.getSize().longValue());
        }
    }
}

