/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cloud.azure.storage;

import com.microsoft.azure.storage.CloudStorageAccount;
import com.microsoft.azure.storage.LocationMode;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.BlobProperties;
import com.microsoft.azure.storage.blob.CloudBlobClient;
import com.microsoft.azure.storage.blob.CloudBlobContainer;
import com.microsoft.azure.storage.blob.CloudBlockBlob;
import com.microsoft.azure.storage.blob.ListBlobItem;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.cloud.azure.storage.AzureStorageService;
import org.elasticsearch.cloud.azure.storage.AzureStorageSettings;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.repositories.RepositoryException;

public class AzureStorageServiceImpl
extends AbstractComponent
implements AzureStorageService {
    final AzureStorageSettings primaryStorageSettings;
    final Map<String, AzureStorageSettings> secondariesStorageSettings;
    final Map<String, CloudBlobClient> clients;

    public AzureStorageServiceImpl(Settings settings) {
        super(settings);
        Tuple<AzureStorageSettings, Map<String, AzureStorageSettings>> storageSettings = AzureStorageSettings.parse(settings);
        this.primaryStorageSettings = (AzureStorageSettings)storageSettings.v1();
        this.secondariesStorageSettings = (Map)storageSettings.v2();
        this.clients = new HashMap<String, CloudBlobClient>();
        this.logger.debug("starting azure storage client instance");
        if (this.primaryStorageSettings != null) {
            this.logger.debug("registering primary client for account [{}]", (Object)this.primaryStorageSettings.getAccount());
            this.createClient(this.primaryStorageSettings);
        }
        for (Map.Entry<String, AzureStorageSettings> azureStorageSettingsEntry : this.secondariesStorageSettings.entrySet()) {
            this.logger.debug("registering secondary client for account [{}]", (Object)azureStorageSettingsEntry.getKey());
            this.createClient(azureStorageSettingsEntry.getValue());
        }
    }

    void createClient(AzureStorageSettings azureStorageSettings) {
        try {
            this.logger.trace("creating new Azure storage client using account [{}], key [{}]", (Object)azureStorageSettings.getAccount(), (Object)azureStorageSettings.getKey());
            String storageConnectionString = "DefaultEndpointsProtocol=https;AccountName=" + azureStorageSettings.getAccount() + ";AccountKey=" + azureStorageSettings.getKey();
            CloudStorageAccount storageAccount = CloudStorageAccount.parse((String)storageConnectionString);
            CloudBlobClient client = storageAccount.createCloudBlobClient();
            this.clients.put(azureStorageSettings.getAccount(), client);
        }
        catch (Exception e) {
            this.logger.error("can not create azure storage client: {}", (Object)e.getMessage());
        }
    }

    CloudBlobClient getSelectedClient(String account, LocationMode mode) {
        this.logger.trace("selecting a client for account [{}], mode [{}]", (Object)account, (Object)mode.name());
        AzureStorageSettings azureStorageSettings = null;
        if (this.primaryStorageSettings == null) {
            throw new IllegalArgumentException("No primary azure storage can be found. Check your elasticsearch.yml.");
        }
        if (Strings.hasLength((String)account)) {
            azureStorageSettings = this.secondariesStorageSettings.get(account);
        }
        if (azureStorageSettings == null && (!Strings.hasLength((String)account) || this.primaryStorageSettings.getName() == null || account.equals(this.primaryStorageSettings.getName()))) {
            azureStorageSettings = this.primaryStorageSettings;
        }
        if (azureStorageSettings == null) {
            throw new IllegalArgumentException("Can not find azure account [" + account + "]. Check your elasticsearch.yml.");
        }
        CloudBlobClient client = this.clients.get(azureStorageSettings.getAccount());
        if (client == null) {
            throw new IllegalArgumentException("Can not find an azure client for account [" + account + "]");
        }
        client.getDefaultRequestOptions().setLocationMode(mode);
        if (azureStorageSettings.getTimeout().getSeconds() > 0L) {
            try {
                int timeout = (int)azureStorageSettings.getTimeout().getMillis();
                client.getDefaultRequestOptions().setTimeoutIntervalInMs(Integer.valueOf(timeout));
            }
            catch (ClassCastException e) {
                throw new IllegalArgumentException("Can not convert [" + azureStorageSettings.getTimeout() + "]. It can not be longer than 2,147,483,647ms.");
            }
        }
        return client;
    }

    @Override
    public boolean doesContainerExist(String account, LocationMode mode, String container) {
        try {
            CloudBlobClient client = this.getSelectedClient(account, mode);
            CloudBlobContainer blobContainer = client.getContainerReference(container);
            return blobContainer.exists();
        }
        catch (Exception e) {
            this.logger.error("can not access container [{}]", (Object)container);
            return false;
        }
    }

    @Override
    public void removeContainer(String account, LocationMode mode, String container) throws URISyntaxException, StorageException {
        CloudBlobClient client = this.getSelectedClient(account, mode);
        CloudBlobContainer blobContainer = client.getContainerReference(container);
        this.logger.trace("removing container [{}]", (Object)container);
        blobContainer.deleteIfExists();
    }

    @Override
    public void createContainer(String account, LocationMode mode, String container) throws URISyntaxException, StorageException {
        try {
            CloudBlobClient client = this.getSelectedClient(account, mode);
            CloudBlobContainer blobContainer = client.getContainerReference(container);
            this.logger.trace("creating container [{}]", (Object)container);
            blobContainer.createIfNotExists();
        }
        catch (IllegalArgumentException e) {
            this.logger.trace(() -> new ParameterizedMessage("fails creating container [{}]", (Object)container), (Throwable)e);
            throw new RepositoryException(container, e.getMessage(), (Throwable)e);
        }
    }

    @Override
    public void deleteFiles(String account, LocationMode mode, String container, String path) throws URISyntaxException, StorageException {
        this.logger.trace("delete files container [{}], path [{}]", (Object)container, (Object)path);
        CloudBlobClient client = this.getSelectedClient(account, mode);
        CloudBlobContainer blobContainer = client.getContainerReference(container);
        if (blobContainer.exists()) {
            for (ListBlobItem blobItem : blobContainer.listBlobs(path, true)) {
                String blobName = AzureStorageServiceImpl.blobNameFromUri(blobItem.getUri());
                this.logger.trace("removing blob [{}] full URI was [{}]", (Object)blobName, (Object)blobItem.getUri());
                this.deleteBlob(account, mode, container, blobName);
            }
        }
    }

    public static String blobNameFromUri(URI uri) {
        String path = uri.getPath();
        String[] splits = path.split("/", 3);
        return splits[2];
    }

    @Override
    public boolean blobExists(String account, LocationMode mode, String container, String blob) throws URISyntaxException, StorageException {
        CloudBlobClient client = this.getSelectedClient(account, mode);
        CloudBlobContainer blobContainer = client.getContainerReference(container);
        if (blobContainer.exists()) {
            CloudBlockBlob azureBlob = blobContainer.getBlockBlobReference(blob);
            return azureBlob.exists();
        }
        return false;
    }

    @Override
    public void deleteBlob(String account, LocationMode mode, String container, String blob) throws URISyntaxException, StorageException {
        this.logger.trace("delete blob for container [{}], blob [{}]", (Object)container, (Object)blob);
        CloudBlobClient client = this.getSelectedClient(account, mode);
        CloudBlobContainer blobContainer = client.getContainerReference(container);
        if (blobContainer.exists()) {
            this.logger.trace("container [{}]: blob [{}] found. removing.", (Object)container, (Object)blob);
            CloudBlockBlob azureBlob = blobContainer.getBlockBlobReference(blob);
            azureBlob.delete();
        }
    }

    @Override
    public InputStream getInputStream(String account, LocationMode mode, String container, String blob) throws URISyntaxException, StorageException {
        this.logger.trace("reading container [{}], blob [{}]", (Object)container, (Object)blob);
        CloudBlobClient client = this.getSelectedClient(account, mode);
        return client.getContainerReference(container).getBlockBlobReference(blob).openInputStream();
    }

    @Override
    public OutputStream getOutputStream(String account, LocationMode mode, String container, String blob) throws URISyntaxException, StorageException {
        this.logger.trace("writing container [{}], blob [{}]", (Object)container, (Object)blob);
        CloudBlobClient client = this.getSelectedClient(account, mode);
        return client.getContainerReference(container).getBlockBlobReference(blob).openOutputStream();
    }

    @Override
    public Map<String, BlobMetaData> listBlobsByPrefix(String account, LocationMode mode, String container, String keyPath, String prefix) throws URISyntaxException, StorageException {
        this.logger.debug("listing container [{}], keyPath [{}], prefix [{}]", (Object)container, (Object)keyPath, (Object)prefix);
        MapBuilder blobsBuilder = MapBuilder.newMapBuilder();
        CloudBlobClient client = this.getSelectedClient(account, mode);
        CloudBlobContainer blobContainer = client.getContainerReference(container);
        if (blobContainer.exists()) {
            for (ListBlobItem blobItem : blobContainer.listBlobs(keyPath + (prefix == null ? "" : prefix))) {
                URI uri = blobItem.getUri();
                this.logger.trace("blob url [{}]", (Object)uri);
                String blobPath = uri.getPath().substring(1 + container.length() + 1);
                CloudBlockBlob blob = blobContainer.getBlockBlobReference(blobPath);
                blob.downloadAttributes();
                BlobProperties properties = blob.getProperties();
                String name = blobPath.substring(keyPath.length());
                this.logger.trace("blob url [{}], name [{}], size [{}]", (Object)uri, (Object)name, (Object)properties.getLength());
                blobsBuilder.put((Object)name, (Object)new PlainBlobMetaData(name, properties.getLength()));
            }
        }
        return blobsBuilder.immutableMap();
    }

    @Override
    public void moveBlob(String account, LocationMode mode, String container, String sourceBlob, String targetBlob) throws URISyntaxException, StorageException {
        this.logger.debug("moveBlob container [{}], sourceBlob [{}], targetBlob [{}]", (Object)container, (Object)sourceBlob, (Object)targetBlob);
        CloudBlobClient client = this.getSelectedClient(account, mode);
        CloudBlobContainer blobContainer = client.getContainerReference(container);
        CloudBlockBlob blobSource = blobContainer.getBlockBlobReference(sourceBlob);
        if (blobSource.exists()) {
            CloudBlockBlob blobTarget = blobContainer.getBlockBlobReference(targetBlob);
            blobTarget.startCopy(blobSource);
            blobSource.delete();
            this.logger.debug("moveBlob container [{}], sourceBlob [{}], targetBlob [{}] -> done", (Object)container, (Object)sourceBlob, (Object)targetBlob);
        }
    }
}

