/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cloud.gce;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.googleapis.GoogleUtils;
import com.google.api.client.googleapis.compute.ComputeCredential;
import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpHeaders;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpRequestFactory;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.jackson2.JacksonFactory;
import com.google.api.client.util.SecurityUtils;
import com.google.api.services.compute.Compute;
import com.google.api.services.compute.model.Instance;
import com.google.api.services.compute.model.InstanceList;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.elasticsearch.cloud.gce.GceInstancesService;
import org.elasticsearch.cloud.gce.GceMetadataService;
import org.elasticsearch.cloud.gce.util.Access;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.discovery.gce.RetryHttpInitializerWrapper;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;

public class GceInstancesServiceImpl
implements GceInstancesService {
    private static final Logger logger = LogManager.getLogger(GceInstancesServiceImpl.class);
    public static final Setting<Boolean> GCE_VALIDATE_CERTIFICATES = Setting.boolSetting((String)"cloud.gce.validate_certificates", (boolean)true, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<String> GCE_ROOT_URL = Setting.simpleString((String)"cloud.gce.root_url", (String)"https://www.googleapis.com", (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private final String project;
    private final List<String> zones;
    private final Settings settings;
    private Compute client;
    private TimeValue refreshInterval = null;
    private long lastRefresh;
    private HttpTransport gceHttpTransport;
    private JsonFactory gceJsonFactory;
    private final boolean validateCerts;

    @Override
    public Collection<Instance> instances() {
        logger.debug("get instances for project [{}], zones [{}]", new Object[]{this.project, this.zones});
        List instances = this.zones.stream().map(zoneId -> {
            try {
                return Access.doPrivilegedIOException(() -> {
                    String nextPageToken = null;
                    ArrayList zoneInstances = new ArrayList();
                    do {
                        Compute.Instances.List list = this.client().instances().list(this.project, zoneId).setPageToken(nextPageToken);
                        InstanceList instanceList = (InstanceList)list.execute();
                        nextPageToken = instanceList.getNextPageToken();
                        if (instanceList.isEmpty() || instanceList.getItems() == null) continue;
                        zoneInstances.addAll(instanceList.getItems());
                    } while (nextPageToken != null);
                    return zoneInstances;
                });
            }
            catch (IOException e) {
                logger.warn(() -> "Problem fetching instance list for zone " + zoneId, (Throwable)e);
                logger.debug("Full exception:", (Throwable)e);
                return Collections.emptyList();
            }
        }).reduce(new ArrayList(), (a, b) -> {
            a.addAll(b);
            return a;
        });
        if (instances.isEmpty()) {
            logger.warn("disabling GCE discovery. Can not get list of nodes");
        }
        return instances;
    }

    public GceInstancesServiceImpl(Settings settings) {
        this.settings = settings;
        this.validateCerts = (Boolean)GCE_VALIDATE_CERTIFICATES.get(settings);
        this.project = this.resolveProject();
        this.zones = this.resolveZones();
    }

    private String resolveProject() {
        if (PROJECT_SETTING.exists(this.settings)) {
            return (String)PROJECT_SETTING.get(this.settings);
        }
        try {
            return this.getAppEngineValueFromMetadataServer("/computeMetadata/v1/project/project-id");
        }
        catch (Exception e) {
            logger.warn("unable to resolve project from metadata server for GCE discovery service", (Throwable)e);
            return null;
        }
    }

    private List<String> resolveZones() {
        if (ZONE_SETTING.exists(this.settings)) {
            return (List)ZONE_SETTING.get(this.settings);
        }
        try {
            String defaultZone = this.getAppEngineValueFromMetadataServer("/computeMetadata/v1/project/attributes/google-compute-default-zone");
            return Collections.singletonList(defaultZone);
        }
        catch (Exception e) {
            logger.warn("unable to resolve default zone from metadata server for GCE discovery service", (Throwable)e);
            return null;
        }
    }

    String getAppEngineValueFromMetadataServer(String serviceURL) throws GeneralSecurityException, IOException {
        String metadata = (String)GceMetadataService.GCE_HOST.get(this.settings);
        GenericUrl url = Access.doPrivileged(() -> new GenericUrl(metadata + serviceURL));
        HttpTransport httpTransport = this.getGceHttpTransport();
        HttpRequestFactory requestFactory = httpTransport.createRequestFactory();
        HttpRequest request = requestFactory.buildGetRequest(url).setConnectTimeout(500).setReadTimeout(500).setHeaders(new HttpHeaders().set("Metadata-Flavor", (Object)"Google"));
        HttpResponse response = Access.doPrivilegedIOException(() -> request.execute());
        return GceInstancesServiceImpl.headerContainsMetadataFlavor(response) ? response.parseAsString() : null;
    }

    private static boolean headerContainsMetadataFlavor(HttpResponse response) {
        String metadataFlavorValue = response.getHeaders().getFirstHeaderStringValue("Metadata-Flavor");
        return "Google".equals(metadataFlavorValue);
    }

    protected synchronized HttpTransport getGceHttpTransport() throws GeneralSecurityException, IOException {
        if (this.gceHttpTransport == null) {
            if (this.validateCerts) {
                KeyStore certTrustStore = SecurityUtils.getJavaKeyStore();
                try (InputStream is = GoogleUtils.class.getResourceAsStream("google.jks");){
                    SecurityUtils.loadKeyStore((KeyStore)certTrustStore, (InputStream)is, (String)"notasecret");
                }
                this.gceHttpTransport = new NetHttpTransport.Builder().trustCertificates(certTrustStore).build();
            } else {
                this.gceHttpTransport = new NetHttpTransport.Builder().doNotValidateCertificate().build();
            }
        }
        return this.gceHttpTransport;
    }

    public synchronized Compute client() {
        if (this.refreshInterval != null && this.refreshInterval.millis() != 0L) {
            if (this.client != null && (this.refreshInterval.millis() < 0L || System.currentTimeMillis() - this.lastRefresh < this.refreshInterval.millis())) {
                if (logger.isTraceEnabled()) {
                    logger.trace("using cache to retrieve client");
                }
                return this.client;
            }
            this.lastRefresh = System.currentTimeMillis();
        }
        try {
            this.gceJsonFactory = new JacksonFactory();
            logger.info("starting GCE discovery service");
            String tokenServerEncodedUrl = (String)GceMetadataService.GCE_HOST.get(this.settings) + "/computeMetadata/v1/instance/service-accounts/default/token";
            ComputeCredential credential = new ComputeCredential.Builder(this.getGceHttpTransport(), this.gceJsonFactory).setTokenServerEncodedUrl(tokenServerEncodedUrl).build();
            Access.doPrivilegedIOException(() -> ((ComputeCredential)credential).refreshToken());
            logger.debug("token [{}] will expire in [{}] s", new Object[]{credential.getAccessToken(), credential.getExpiresInSeconds()});
            if (credential.getExpiresInSeconds() != null) {
                this.refreshInterval = TimeValue.timeValueSeconds((long)(credential.getExpiresInSeconds() - 1L));
            }
            Compute.Builder builder = new Compute.Builder(this.getGceHttpTransport(), this.gceJsonFactory, null).setApplicationName("Elasticsearch/GceCloud/1.0").setRootUrl((String)GCE_ROOT_URL.get(this.settings));
            if (RETRY_SETTING.exists(this.settings)) {
                TimeValue maxWait = (TimeValue)MAX_WAIT_SETTING.get(this.settings);
                RetryHttpInitializerWrapper retryHttpInitializerWrapper = maxWait.getMillis() > 0L ? new RetryHttpInitializerWrapper((Credential)credential, maxWait) : new RetryHttpInitializerWrapper((Credential)credential);
                builder.setHttpRequestInitializer((HttpRequestInitializer)retryHttpInitializerWrapper);
            } else {
                builder.setHttpRequestInitializer((HttpRequestInitializer)credential);
            }
            this.client = builder.build();
        }
        catch (Exception e) {
            logger.warn("unable to start GCE discovery service", (Throwable)e);
            throw new IllegalArgumentException("unable to start GCE discovery service", e);
        }
        return this.client;
    }

    @Override
    public String projectId() {
        return this.project;
    }

    @Override
    public List<String> zones() {
        return this.zones;
    }

    @Override
    public void close() throws IOException {
        if (this.gceHttpTransport != null) {
            this.gceHttpTransport.shutdown();
        }
    }
}

