/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.watcher.watch;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.watcher.support.Exceptions;
import org.elasticsearch.watcher.support.init.proxy.ClientProxy;
import org.elasticsearch.watcher.watch.Watch;

public class WatchStore
extends AbstractComponent {
    public static final String INDEX = ".watches";
    public static final String DOC_TYPE = "watch";
    private final ClientProxy client;
    private final Watch.Parser watchParser;
    private final ConcurrentMap<String, Watch> watches;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final int scrollSize;
    private final TimeValue scrollTimeout;

    @Inject
    public WatchStore(Settings settings, ClientProxy client, Watch.Parser watchParser) {
        super(settings);
        this.client = client;
        this.watchParser = watchParser;
        this.watches = ConcurrentCollections.newConcurrentMap();
        this.scrollTimeout = settings.getAsTime("watcher.watch.scroll.timeout", TimeValue.timeValueSeconds((long)30L));
        this.scrollSize = settings.getAsInt("watcher.watch.scroll.size", Integer.valueOf(100));
    }

    public void start(ClusterState state) throws Exception {
        if (this.started.get()) {
            this.logger.debug("watch store already started", new Object[0]);
            return;
        }
        IndexMetaData watchesIndexMetaData = state.getMetaData().index(INDEX);
        if (watchesIndexMetaData != null) {
            try {
                int count = this.loadWatches(watchesIndexMetaData.getNumberOfShards());
                this.logger.debug("loaded [{}] watches from the watches index [{}]", new Object[]{count, INDEX});
                this.started.set(true);
            }
            catch (Exception e) {
                this.logger.debug("failed to load watches for watch index [{}]", (Throwable)e, new Object[]{INDEX});
                this.watches.clear();
                throw e;
            }
        } else {
            this.started.set(true);
        }
    }

    public boolean validate(ClusterState state) {
        IndexMetaData watchesIndexMetaData = state.getMetaData().index(INDEX);
        if (watchesIndexMetaData == null) {
            this.logger.debug("index [{}] doesn't exist, so we can start", new Object[]{INDEX});
            return true;
        }
        if (state.routingTable().index(INDEX).allPrimaryShardsActive()) {
            this.logger.debug("index [{}] exists and all primary shards are started, so we can start", new Object[]{INDEX});
            return true;
        }
        this.logger.debug("not all primary shards active for index [{}], so we cannot start", new Object[]{INDEX});
        return false;
    }

    public boolean started() {
        return this.started.get();
    }

    public void stop() {
        if (this.started.compareAndSet(true, false)) {
            this.watches.clear();
            this.logger.info("stopped watch store", new Object[0]);
        }
    }

    public Watch get(String id) {
        this.ensureStarted();
        return (Watch)this.watches.get(id);
    }

    public WatchPut put(Watch watch) throws IOException {
        this.ensureStarted();
        IndexRequest indexRequest = this.createIndexRequest(watch.id(), watch.getAsBytes(), -3L);
        IndexResponse response = this.client.index(indexRequest, (TimeValue)null);
        watch.status().version(response.getVersion());
        watch.version(response.getVersion());
        Watch previous = this.watches.put(watch.id(), watch);
        return new WatchPut(previous, watch, response);
    }

    public void updateStatus(Watch watch) throws IOException {
        this.ensureStarted();
        if (!watch.status().dirty()) {
            return;
        }
        XContentBuilder source = JsonXContent.contentBuilder().startObject().field(Watch.Field.STATUS.getPreferredName(), (ToXContent)watch.status(), ToXContent.EMPTY_PARAMS).endObject();
        UpdateRequest updateRequest = new UpdateRequest(INDEX, DOC_TYPE, watch.id());
        updateRequest.doc(source);
        updateRequest.version(watch.version());
        UpdateResponse response = this.client.update(updateRequest);
        watch.status().version(response.getVersion());
        watch.version(response.getVersion());
        watch.status().resetDirty();
    }

    public WatchDelete delete(String id, boolean force) {
        this.ensureStarted();
        Watch watch = (Watch)this.watches.remove(id);
        DeleteRequest request = new DeleteRequest(INDEX, DOC_TYPE, id);
        if (watch != null && !force) {
            request.version(watch.version());
        }
        DeleteResponse response = this.client.delete(request);
        if (watch != null) {
            watch.version(response.getVersion());
        }
        return new WatchDelete(response);
    }

    public Collection<Watch> watches() {
        return this.watches.values();
    }

    public Collection<Watch> activeWatches() {
        HashSet<Watch> watches = new HashSet<Watch>();
        for (Watch watch : this.watches()) {
            if (!watch.status().state().isActive()) continue;
            watches.add(watch);
        }
        return watches;
    }

    public int watchCount() {
        return this.watches.size();
    }

    IndexRequest createIndexRequest(String id, BytesReference source, long version) {
        IndexRequest indexRequest = new IndexRequest(INDEX, DOC_TYPE, id);
        indexRequest.source(source.toBytes());
        indexRequest.version(version);
        return indexRequest;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int loadWatches(int numPrimaryShards) {
        assert (this.watches.isEmpty()) : "no watches should reside, but there are [" + this.watches.size() + "] watches.";
        RefreshResponse refreshResponse = this.client.refresh(new RefreshRequest(new String[]{INDEX}));
        if (refreshResponse.getSuccessfulShards() < numPrimaryShards) {
            throw Exceptions.illegalState("not all required shards have been refreshed", new Object[0]);
        }
        int count = 0;
        SearchRequest searchRequest = new SearchRequest(new String[]{INDEX}).types(new String[]{DOC_TYPE}).preference("_primary").searchType(SearchType.SCAN).scroll(this.scrollTimeout).source(new SearchSourceBuilder().size(this.scrollSize).version(Boolean.valueOf(true)));
        SearchResponse response = this.client.search(searchRequest, null);
        try {
            if (response.getTotalShards() != response.getSuccessfulShards()) {
                throw new ElasticsearchException("Partial response while loading watches", new Object[0]);
            }
            if (response.getHits().getTotalHits() > 0L) {
                response = this.client.searchScroll(response.getScrollId(), this.scrollTimeout);
                while (response.getHits().hits().length != 0) {
                    for (SearchHit hit : response.getHits()) {
                        String id = hit.getId();
                        try {
                            Watch watch = this.watchParser.parse(id, true, hit.getSourceRef());
                            watch.status().version(hit.version());
                            watch.version(hit.version());
                            this.watches.put(id, watch);
                            ++count;
                        }
                        catch (Exception e) {
                            this.logger.error("couldn't load watch [{}], ignoring it...", (Throwable)e, new Object[]{id});
                        }
                    }
                    response = this.client.searchScroll(response.getScrollId(), this.scrollTimeout);
                }
            }
        }
        finally {
            this.client.clearScroll(response.getScrollId());
        }
        return count;
    }

    private void ensureStarted() {
        if (!this.started.get()) {
            throw new IllegalStateException("watch store not started");
        }
    }

    public class WatchDelete {
        private final DeleteResponse response;

        public WatchDelete(DeleteResponse response) {
            this.response = response;
        }

        public DeleteResponse deleteResponse() {
            return this.response;
        }
    }

    public class WatchPut {
        private final Watch previous;
        private final Watch current;
        private final IndexResponse response;

        public WatchPut(Watch previous, Watch current, IndexResponse response) {
            this.current = current;
            this.previous = previous;
            this.response = response;
        }

        public Watch current() {
            return this.current;
        }

        public Watch previous() {
            return this.previous;
        }

        public IndexResponse indexResponse() {
            return this.response;
        }
    }
}

