/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.watcher;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicReference;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.watcher.WatcherState;
import org.elasticsearch.watcher.execution.ExecutionService;
import org.elasticsearch.watcher.support.Exceptions;
import org.elasticsearch.watcher.support.WatcherIndexTemplateRegistry;
import org.elasticsearch.watcher.support.clock.Clock;
import org.elasticsearch.watcher.trigger.TriggerService;
import org.elasticsearch.watcher.watch.Watch;
import org.elasticsearch.watcher.watch.WatchLockService;
import org.elasticsearch.watcher.watch.WatchStatus;
import org.elasticsearch.watcher.watch.WatchStore;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.PeriodType;

public class WatcherService
extends AbstractComponent {
    private final Clock clock;
    private final TriggerService triggerService;
    private final Watch.Parser watchParser;
    private final WatchStore watchStore;
    private final WatchLockService watchLockService;
    private final ExecutionService executionService;
    private final WatcherIndexTemplateRegistry watcherIndexTemplateRegistry;
    final AtomicReference<WatcherState> state = new AtomicReference<WatcherState>(WatcherState.STOPPED);

    @Inject
    public WatcherService(Settings settings, Clock clock, TriggerService triggerService, WatchStore watchStore, Watch.Parser watchParser, ExecutionService executionService, WatchLockService watchLockService, WatcherIndexTemplateRegistry watcherIndexTemplateRegistry) {
        super(settings);
        this.clock = clock;
        this.triggerService = triggerService;
        this.watchStore = watchStore;
        this.watchParser = watchParser;
        this.watchLockService = watchLockService;
        this.executionService = executionService;
        this.watcherIndexTemplateRegistry = watcherIndexTemplateRegistry;
    }

    public void start(ClusterState clusterState) throws Exception {
        if (this.state.compareAndSet(WatcherState.STOPPED, WatcherState.STARTING)) {
            try {
                this.logger.info("starting watch service...", new Object[0]);
                this.watcherIndexTemplateRegistry.addTemplatesIfMissing();
                this.watchLockService.start();
                this.watchStore.start(clusterState);
                this.executionService.start(clusterState);
                this.triggerService.start(this.watchStore.activeWatches());
                this.state.set(WatcherState.STARTED);
                this.logger.info("watch service has started", new Object[0]);
            }
            catch (Exception e) {
                this.state.set(WatcherState.STOPPED);
                throw e;
            }
        } else {
            this.logger.debug("not starting watcher, because its state is [{}] while [{}] is expected", new Object[]{this.state, WatcherState.STOPPED});
        }
    }

    public boolean validate(ClusterState state) {
        return this.watchStore.validate(state) && this.executionService.validate(state);
    }

    public void stop() {
        if (this.state.compareAndSet(WatcherState.STARTED, WatcherState.STOPPING)) {
            this.logger.info("stopping watch service...", new Object[0]);
            this.triggerService.stop();
            this.executionService.stop();
            try {
                this.watchLockService.stop();
            }
            catch (ElasticsearchTimeoutException te) {
                this.logger.warn("error stopping WatchLockService", (Throwable)te, new Object[0]);
            }
            this.watchStore.stop();
            this.state.set(WatcherState.STOPPED);
            this.logger.info("watch service has stopped", new Object[0]);
        } else {
            this.logger.debug("not stopping watcher, because its state is [{}] while [{}] is expected", new Object[]{this.state, WatcherState.STARTED});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public WatchStore.WatchDelete deleteWatch(String id, TimeValue timeout, boolean force) {
        this.ensureStarted();
        WatchLockService.Lock lock = null;
        if (!force && (lock = this.watchLockService.tryAcquire(id, timeout)) == null) {
            throw new ElasticsearchTimeoutException("could not delete watch [{}] within [{}]... wait and try again. If this error continues to occur there is a high chance that the watch execution is stuck (either due to unresponsive external system such as an email service, or due to a bad script", new Object[]{id, timeout.format(PeriodType.seconds())});
        }
        try {
            WatchStore.WatchDelete delete = this.watchStore.delete(id, force);
            if (delete.deleteResponse().isFound()) {
                this.triggerService.remove(id);
            }
            WatchStore.WatchDelete watchDelete = delete;
            return watchDelete;
        }
        finally {
            if (lock != null) {
                lock.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public IndexResponse putWatch(String id, BytesReference watchSource, TimeValue timeout, boolean active) throws IOException {
        this.ensureStarted();
        WatchLockService.Lock lock = this.watchLockService.tryAcquire(id, timeout);
        if (lock == null) {
            throw new ElasticsearchTimeoutException("could not put watch [{}] within [{}]... wait and try again. If this error continues to occur there is a high chance that the watch execution is stuck (either due to unresponsive external system such as an email service, or due to a bad script", new Object[]{id, timeout.format(PeriodType.seconds())});
        }
        try {
            DateTime now = this.clock.nowUTC();
            Watch watch = this.watchParser.parseWithSecrets(id, false, watchSource, now);
            watch.setState(active, now);
            WatchStore.WatchPut result = this.watchStore.put(watch);
            if (result.previous() == null) {
                if (result.current().status().state().isActive()) {
                    this.triggerService.add(result.current());
                }
            } else if (result.current().status().state().isActive()) {
                if (!result.previous().status().state().isActive()) {
                    this.triggerService.add(result.current());
                } else if (!result.previous().trigger().equals(result.current().trigger())) {
                    this.triggerService.add(result.current());
                }
            } else {
                this.triggerService.remove(result.current().id());
            }
            IndexResponse indexResponse = result.indexResponse();
            return indexResponse;
        }
        finally {
            lock.release();
        }
    }

    public Watch getWatch(String name) {
        return this.watchStore.get(name);
    }

    public WatcherState state() {
        return this.state.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public WatchStatus ackWatch(String id, String[] actionIds, TimeValue timeout) throws IOException {
        this.ensureStarted();
        WatchLockService.Lock lock = this.watchLockService.tryAcquire(id, timeout);
        if (lock == null) {
            throw new ElasticsearchTimeoutException("could not ack watch [{}] within [{}]... wait and try again. If this error continues to occur there is a high chance that the watch execution is stuck (either due to unresponsive external system such as an email service, or due to a bad script", new Object[]{id, timeout.format(PeriodType.seconds())});
        }
        if (actionIds == null || actionIds.length == 0) {
            actionIds = new String[]{"_all"};
        }
        try {
            Watch watch = this.watchStore.get(id);
            if (watch == null) {
                throw Exceptions.illegalArgument("watch [{}] does not exist", id);
            }
            if (watch.ack(this.clock.now(DateTimeZone.UTC), actionIds)) {
                try {
                    this.watchStore.updateStatus(watch);
                }
                catch (IOException ioe) {
                    throw Exceptions.ioException("failed to update the watch [{}] on ack", ioe, watch.id());
                }
                catch (VersionConflictEngineException vcee) {
                    throw Exceptions.illegalState("failed to update the watch [{}] on ack, perhaps it was force deleted", vcee, watch.id());
                }
            }
            WatchStatus watchStatus = new WatchStatus(watch.status());
            return watchStatus;
        }
        finally {
            lock.release();
        }
    }

    public WatchStatus activateWatch(String id, TimeValue timeout) throws IOException {
        return this.setWatchState(id, true, timeout);
    }

    public WatchStatus deactivateWatch(String id, TimeValue timeout) throws IOException {
        return this.setWatchState(id, false, timeout);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    WatchStatus setWatchState(String id, boolean active, TimeValue timeout) throws IOException {
        this.ensureStarted();
        WatchLockService.Lock lock = this.watchLockService.tryAcquire(id, timeout);
        if (lock == null) {
            throw new ElasticsearchTimeoutException("could not ack watch [{}] within [{}]... wait and try again. If this error continues to occur there is a high chance that the watch execution is stuck (either due to unresponsive external system such as an email service, or due to a bad script", new Object[]{id, timeout.format(PeriodType.seconds())});
        }
        try {
            Watch watch = this.watchStore.get(id);
            if (watch == null) {
                throw Exceptions.illegalArgument("watch [{}] does not exist", id);
            }
            if (watch.setState(active, this.clock.nowUTC())) {
                try {
                    this.watchStore.updateStatus(watch);
                    if (active) {
                        this.triggerService.add(watch);
                    } else {
                        this.triggerService.remove(watch.id());
                    }
                }
                catch (IOException ioe) {
                    throw Exceptions.ioException("failed to update the watch [{}] on ack", ioe, watch.id());
                }
                catch (VersionConflictEngineException vcee) {
                    throw Exceptions.illegalState("failed to update the watch [{}] on ack, perhaps it was force deleted", vcee, watch.id());
                }
            }
            WatchStatus watchStatus = new WatchStatus(watch.status());
            return watchStatus;
        }
        finally {
            lock.release();
        }
    }

    public long watchesCount() {
        return this.watchStore.watches().size();
    }

    private void ensureStarted() {
        if (this.state.get() != WatcherState.STARTED) {
            throw new IllegalStateException("not started");
        }
    }
}

