/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.watcher.execution;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.watcher.execution.TriggeredWatch;
import org.elasticsearch.watcher.execution.Wid;
import org.elasticsearch.watcher.support.Exceptions;
import org.elasticsearch.watcher.support.init.proxy.ClientProxy;

public class TriggeredWatchStore
extends AbstractComponent {
    public static final String INDEX_NAME = ".triggered_watches";
    public static final String DOC_TYPE = "triggered_watch";
    private final int scrollSize;
    private final ClientProxy client;
    private final TimeValue scrollTimeout;
    private final TriggeredWatch.Parser triggeredWatchParser;
    private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    private final Lock accessLock = this.readWriteLock.readLock();
    private final Lock stopLock = this.readWriteLock.writeLock();
    private final AtomicBoolean started = new AtomicBoolean(false);

    @Inject
    public TriggeredWatchStore(Settings settings, ClientProxy client, TriggeredWatch.Parser triggeredWatchParser) {
        super(settings);
        this.scrollSize = settings.getAsInt("watcher.execution.scroll.size", Integer.valueOf(100));
        this.client = client;
        this.scrollTimeout = settings.getAsTime("watcher.execution.scroll.timeout", TimeValue.timeValueSeconds((long)30L));
        this.triggeredWatchParser = triggeredWatchParser;
    }

    public void start() {
        this.started.set(true);
    }

    public boolean validate(ClusterState state) {
        IndexMetaData indexMetaData = state.getMetaData().index(INDEX_NAME);
        if (indexMetaData != null) {
            if (!state.routingTable().index(INDEX_NAME).allPrimaryShardsActive()) {
                this.logger.debug("not all primary shards of the [{}] index are started, so we cannot load previous triggered watches", new Object[]{INDEX_NAME});
                return false;
            }
        } else {
            this.logger.debug("triggered watch index doesn't exist, so we can load", new Object[0]);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        this.stopLock.lock();
        try {
            this.started.set(false);
        }
        finally {
            this.stopLock.unlock();
        }
    }

    public void put(TriggeredWatch triggeredWatch) throws Exception {
        this.ensureStarted();
        this.accessLock.lock();
        try {
            IndexRequest request = new IndexRequest(INDEX_NAME, DOC_TYPE, triggeredWatch.id().value()).source(XContentFactory.jsonBuilder().value((Object)triggeredWatch)).opType(IndexRequest.OpType.CREATE);
            this.client.index(request, (TimeValue)null);
        }
        catch (IOException e) {
            throw Exceptions.ioException("failed to persist triggered watch [{}]", e, triggeredWatch);
        }
        finally {
            this.accessLock.unlock();
        }
    }

    public void put(TriggeredWatch triggeredWatch, final ActionListener<Boolean> listener) throws Exception {
        this.ensureStarted();
        try {
            IndexRequest request = new IndexRequest(INDEX_NAME, DOC_TYPE, triggeredWatch.id().value()).source(XContentFactory.jsonBuilder().value((Object)triggeredWatch)).opType(IndexRequest.OpType.CREATE);
            this.client.index(request, new ActionListener<IndexResponse>(){

                public void onResponse(IndexResponse response) {
                    listener.onResponse((Object)true);
                }

                public void onFailure(Throwable e) {
                    listener.onFailure(e);
                }
            });
        }
        catch (IOException e) {
            throw Exceptions.ioException("failed to persist triggered watch [{}]", e, triggeredWatch);
        }
    }

    public void putAll(List<TriggeredWatch> triggeredWatches, final ActionListener<List<Integer>> listener) throws Exception {
        if (triggeredWatches.isEmpty()) {
            listener.onResponse((Object)Collections.EMPTY_LIST);
            return;
        }
        if (triggeredWatches.size() == 1) {
            this.put(triggeredWatches.get(0), new ActionListener<Boolean>(){

                public void onResponse(Boolean success) {
                    listener.onResponse(Collections.singletonList(0));
                }

                public void onFailure(Throwable e) {
                    listener.onFailure(e);
                }
            });
            return;
        }
        this.ensureStarted();
        try {
            BulkRequest request = new BulkRequest();
            for (TriggeredWatch triggeredWatch : triggeredWatches) {
                IndexRequest indexRequest = new IndexRequest(INDEX_NAME, DOC_TYPE, triggeredWatch.id().value());
                indexRequest.source(XContentFactory.jsonBuilder().value((Object)triggeredWatch));
                indexRequest.opType(IndexRequest.OpType.CREATE);
                request.add(indexRequest);
            }
            this.client.bulk(request, new ActionListener<BulkResponse>(){

                public void onResponse(BulkResponse response) {
                    ArrayList<Integer> successFullSlots = new ArrayList<Integer>();
                    for (int i = 0; i < response.getItems().length; ++i) {
                        BulkItemResponse itemResponse = response.getItems()[i];
                        if (itemResponse.isFailed()) {
                            TriggeredWatchStore.this.logger.error("could store triggered watch with id [{}], because failed [{}]", new Object[]{itemResponse.getId(), itemResponse.getFailureMessage()});
                            continue;
                        }
                        IndexResponse indexResponse = (IndexResponse)itemResponse.getResponse();
                        successFullSlots.add(i);
                    }
                    listener.onResponse(successFullSlots);
                }

                public void onFailure(Throwable e) {
                    listener.onFailure(e);
                }
            });
        }
        catch (IOException e) {
            throw Exceptions.ioException("failed to persist triggered watches", e, new Object[0]);
        }
    }

    public List<Integer> putAll(List<TriggeredWatch> triggeredWatches) throws Exception {
        this.ensureStarted();
        try {
            BulkRequest request = new BulkRequest();
            for (TriggeredWatch triggeredWatch : triggeredWatches) {
                IndexRequest indexRequest = new IndexRequest(INDEX_NAME, DOC_TYPE, triggeredWatch.id().value());
                indexRequest.source(XContentFactory.jsonBuilder().value((Object)triggeredWatch));
                indexRequest.opType(IndexRequest.OpType.CREATE);
                request.add(indexRequest);
            }
            BulkResponse response = this.client.bulk(request, (TimeValue)null);
            ArrayList<Integer> successFullSlots = new ArrayList<Integer>();
            for (int i = 0; i < response.getItems().length; ++i) {
                BulkItemResponse itemResponse = response.getItems()[i];
                if (itemResponse.isFailed()) {
                    this.logger.error("could store triggered watch with id [{}], because failed [{}]", new Object[]{itemResponse.getId(), itemResponse.getFailureMessage()});
                    continue;
                }
                IndexResponse indexResponse = (IndexResponse)itemResponse.getResponse();
                successFullSlots.add(i);
            }
            return successFullSlots;
        }
        catch (IOException e) {
            throw Exceptions.ioException("failed to persist triggered watches", e, new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void delete(Wid wid) throws Exception {
        this.ensureStarted();
        this.accessLock.lock();
        try {
            DeleteRequest request = new DeleteRequest(INDEX_NAME, DOC_TYPE, wid.value());
            this.client.delete(request);
            this.logger.trace("successfully deleted triggered watch with id [{}]", new Object[]{wid});
        }
        finally {
            this.accessLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<TriggeredWatch> loadTriggeredWatches(ClusterState state) {
        IndexMetaData indexMetaData = state.getMetaData().index(INDEX_NAME);
        if (indexMetaData == null) {
            this.logger.debug("no .triggered_watches indices found. skipping loading awaiting triggered watches", new Object[0]);
            return Collections.emptySet();
        }
        if (!state.routingTable().index(INDEX_NAME).allPrimaryShardsActive()) {
            throw Exceptions.illegalState("not all primary shards of the [{}] index are started.", INDEX_NAME);
        }
        int numPrimaryShards = indexMetaData.getNumberOfShards();
        RefreshResponse refreshResponse = this.client.refresh(new RefreshRequest(new String[]{INDEX_NAME}));
        if (refreshResponse.getSuccessfulShards() < numPrimaryShards) {
            throw Exceptions.illegalState("refresh was supposed to run on [{}] shards, but ran on [{}] shards", numPrimaryShards, refreshResponse.getSuccessfulShards());
        }
        SearchRequest searchRequest = this.createScanSearchRequest();
        SearchResponse response = this.client.search(searchRequest, null);
        ArrayList<TriggeredWatch> triggeredWatches = new ArrayList<TriggeredWatch>();
        try {
            if (response.getTotalShards() != response.getSuccessfulShards()) {
                throw Exceptions.illegalState("scan search was supposed to run on [{}] shards, but ran on [{}] shards", numPrimaryShards, response.getSuccessfulShards());
            }
            if (response.getHits().getTotalHits() > 0L) {
                response = this.client.searchScroll(response.getScrollId(), this.scrollTimeout);
                while (response.getHits().hits().length != 0) {
                    for (SearchHit sh : response.getHits()) {
                        String id = sh.getId();
                        try {
                            TriggeredWatch triggeredWatch = this.triggeredWatchParser.parse(id, sh.version(), sh.getSourceRef());
                            this.logger.debug("loaded triggered watch [{}/{}/{}]", new Object[]{sh.index(), sh.type(), sh.id()});
                            triggeredWatches.add(triggeredWatch);
                        }
                        catch (Exception e) {
                            this.logger.error("couldn't load triggered watch [{}], ignoring it...", (Throwable)e, new Object[]{id});
                        }
                    }
                    response = this.client.searchScroll(response.getScrollId(), this.scrollTimeout);
                }
            }
        }
        finally {
            this.client.clearScroll(response.getScrollId());
        }
        return triggeredWatches;
    }

    private SearchRequest createScanSearchRequest() {
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder().size(this.scrollSize);
        SearchRequest searchRequest = new SearchRequest(new String[]{INDEX_NAME});
        searchRequest.source(sourceBuilder);
        searchRequest.searchType(SearchType.SCAN);
        searchRequest.types(new String[]{DOC_TYPE});
        searchRequest.scroll(this.scrollTimeout);
        searchRequest.preference("_primary");
        return searchRequest;
    }

    private void ensureStarted() {
        if (!this.started.get()) {
            throw Exceptions.illegalState("unable to persist triggered watches, the store is not ready", new Object[0]);
        }
    }
}

